/*
	file_type: Determine (input) file type.

	part of DerMixD
	(c)2004-14 Thomas Orgis, licensed under GPLv2
*/

#include "basics.hxx"

#ifdef HAVE_MAGIC
#include <magic.h>
#endif

#include <string>

#include "tstring.hxx"

#include "file_type.hxx"
#ifdef IN_SNDFILE
#	include "in/drv/in_sndfile.hxx"
#endif

using std::string;

#include "debug.hxx"
#define ME "file_type"

namespace input
{

struct file_type::my_private
{
#ifdef HAVE_MAGIC
	magic_t cookie;
	bool prepared;
#endif
};

file_type::file_type(): parts(new struct my_private)
{
	CONSTRUCT(ME);
#ifdef HAVE_MAGIC
	parts->prepared = false;
	// Old versions of libmagic do not know MAGIC_MIME_TYPE.
	// MAGIC_MIME works just as well, just need to strip encoding.
	parts->cookie = magic_open(MAGIC_MIME|MAGIC_ERROR|MAGIC_SYMLINK);
	if(parts->cookie == NULL)
	SERROR("Magic parts->cookie creation failed!");
#endif
}

file_type::~file_type()
{
	DESTBEGIN(ME);
#ifdef HAVE_MAGIC
	if(parts->cookie != NULL) magic_close(parts->cookie);
#endif

	delete parts;
	DESTEND(ME);
}

void file_type::prepare()
{
#ifdef HAVE_MAGIC
	if(!parts->prepared && parts->cookie != NULL)
	{
		if(magic_load(parts->cookie, NULL) != 0)
		MERROR("Failed to load magic database: %s", magic_error(parts->cookie));
		parts->prepared = true;
	}
#endif
}

const string file_type::prefix(const string filename)
{
	size_t preend = filename.find("://");
	if(preend == string::npos) return "";

	string pre = filename.substr(0,preend);
	// The prefix should not contain slashes or : itself... basic sanity.
	if(pre.find_first_of(":/") != string::npos) return "";

	return pre;
}

const string file_type::ending(const string filename)
{
	// Take the rightmost dot, no slashes after it -- that's the ending.
	size_t dot = filename.find_last_of('.');
	if(dot == string::npos) return "";

	if(filename.find_first_of('/', dot) != string::npos) return "";

	// Oh, dear... theoretically, I'd have to take care of size_t overflow.
	// I need at least one more character after the dot.
	if(filename.length() <= dot+1) return "";

	return filename.substr(dot+1);
}

const string file_type::mime_guess(const string filename)
{
#ifdef HAVE_MAGIC
	if(parts->cookie == NULL) return "";

	const char *tmp = magic_file(parts->cookie, filename.c_str());
	if(tmp)
	{
		// This feels stupid ... and also ... does this leak memory?
		string buf(tmp);
		buf.erase(buf.find(';'));
		return buf;
	}
	else
#endif
	return "";
}

struct idmap
{
	const char *name;
	type_id value;
};

static const struct idmap endmap[] =
{
	 { "mpeg", mpg123 }
	,{ "mpg",  mpg123 }
	,{ "mp1",  mpg123 }
	,{ "mp2",  mpg123 }
	,{ "mp3",  mpg123 }
	,{ "wav",  sndfile  }
	,{ "raw_s16", raw_s16 }
};

// We are assuming ASCII-only characters...
type_id file_type::ending_to_type(const string ending)
{
	std::string lowerend = ending;
	lowercase(lowerend);
	for(size_t i=0; i<sizeof(endmap)/sizeof(struct idmap); ++i)
	{
		if(lowerend == endmap[i].name)
		return endmap[i].value;
	}
	return invalid;
}

// Yeah, not very elaborate.
static const struct idmap mimemap[] =
{
	 { "audio/mpeg",  mpg123 }
	,{ "audio/wav",   sndfile }
	,{ "audio/wave",  sndfile }
	,{ "audio/x-wav", sndfile }
	,{ "application/ogg", vorbis }
	,{ "audio/ogg", vorbis }
};

type_id file_type::mime_to_type(const string mime)
{
	for(size_t i=0; i<sizeof(mimemap)/sizeof(struct idmap); ++i)
	{
		if(mime == mimemap[i].name)
		return mimemap[i].value;
	}
	return invalid;
}

type_id file_type::guess(const string filename)
{
	prepare();

	type_id type = invalid;

	// First check if we have (fakes) URL thingie that names the type.
	type = name_to_id(prefix(filename));
	if(type != invalid) return type;

	// Investigate file for possible types.
	type = mime_to_type(mime_guess(filename));
	if(type != invalid) return type;

	// If MIME check is not enough, try file name ending.
	type = ending_to_type(ending(filename));
	if(type != invalid) return type;

	// If it's still not clear, resort to brute force checking with libsndfile.
	// I use only libsndfile here because it supports only formats that identify itselves with headers (well, except raw files...).
#ifdef IN_SNDFILE
	if(type == invalid)
	{
		snd_file tester;
		if(tester.do_open(filename))
		type = sndfile;
	}
#endif

	return type;
}

} // namespace input
