#ifndef DMD_H_ERRORCHAIN
#define DMD_H_ERRORCHAIN

/*
	errorchain: a list of error objects, to explain some failure
	
	part of DerMixD
	(c)2009-2010 Thomas Orgis, licensed under GPLv2
*/

#include "errorcodes.hxx"

#include <string>
#include <vector>

// Well, this one is possibly similar to somthing that one could call exception class...
class error
{
	public:
	int code;        // An error code that is at least unique for the issuing entity.
	std::string who;      // The name of the class / entity that issues the error.
	std::string text;     // Textual description of the error.
	std::string sys_text; // Textual description of a system error that may lie below.

	// Two total constructors...
	// One that fetches a system error itself...
	error(const std::string you, int errcode, const std::string txt);
	// ... and one that takes the a system code you hand in (including 0 for explicitly nothing).
	error(const std::string you, int errcode, const std::string txt, int sys_errno);
	// And one constructor for use of persistent error objects belonging to some issuer.
	error(const std::string you);
	~error();
	// Set the error state, return false (for "not good, an error occured").
	// This one retrieves system error state.
	bool occur(int errcode = dmd::err::SOMETHING, const std::string txt="");
	// This one takes given system error code (including 0 for nothing).
	bool occur(int errcode, const std::string txt, int sys_errno);
	// Shortcut to check if an error occured.
	bool occured(){ return (code != dmd::err::NOTHING); };
	// Change back to neutral error state.
	void reset(){ code=dmd::err::NOTHING; text=""; sys_text=""; };
	// Take over error state, reset the former.
	// Also returns false if there is an error, true if there isn't.
	bool take_over(error &former);

	// Retrieve a textual representation, caller deletes.
	std::string* as_string();
};

class errorchain
{
	private:
	std::vector<error*> items;

	public:
	errorchain();
	~errorchain();
	// Add a new error to the end of the chain.
	// The error will be deleted by the chain destructor.
	void push(error *);
	// Turn the chain of errors into a list of strings (lines).
	// The first error is the first entry (line).
	// To clarify: The error strings are appended to the vector.
	void stringify(std::vector<std::string*> &);
	// Check if there are errors, by looking at the number of them.
	size_t count(){ return items.size(); };
	bool empty(){ return items.empty(); };
	void clear();
};

#endif
