/*
	errorchain: a list of error objects, to explain some failure
	
	part of DerMixD
	(c)2009-2013 Thomas Orgis, licensed under GPLv2
*/

#include "errorchain.hxx"
#include "syserror.hxx"
#include "tstring.hxx"

#include "shortcuts.hxx"
#include "debug.hxx"

using namespace std;

error::error(const string you, int errcode, const string txt):
	code(errcode), who(you), text(txt)
{
	CONSTRUCT("error");
	syserror serr;
	if(serr.problem()) sys_text = serr.str();
}

error::error(const string you, int errcode, const string txt, int sys_errno):
	code(errcode), who(you), text(txt)
{
	CONSTRUCT("error");
	if(sys_errno != 0)
	{
		syserror serr;
		sys_text = serr.str(sys_errno);
	}
}

error::error(const string you): code(dmd::err::NOTHING), who(you)
{
	CONSTRUCT("error");
}

error::~error()
{
	DESTRUCT("error");
}

bool error::occur(int errcode, const string txt)
{
	code = errcode;
	text = txt;
	sys_text = "";
	return false;
}

bool error::occur(int errcode, const std::string txt, int sys_errno)
{
	code = errcode;
	text = txt;
	if(sys_errno != 0)
	{
		syserror serr;
		sys_text = serr.str(sys_errno);
	}
	else sys_text = "";

	return false;
}

bool error::take_over(error &former)
{
	code = former.code;
	text = former.text;
	sys_text = former.sys_text;
	former.reset();
	return !occured();
}

// "text (code) (system: Bla bla)"
string* error::as_string()
{
	std::string *ts = new std::string;

	*ts += who;
	*ts += ": ";
	if(text != ""){ *ts += text; *ts += " "; }

	strprintf(*ts, "(%i)", code);

	if(sys_text != "")
	{
		*ts += "; (system: ";
		*ts += sys_text;
		*ts += ")";
	}

	return ts;
}

errorchain::errorchain(){ CONSTRUCT("errorchain"); }
errorchain::~errorchain()
{
	DESTBEGIN("errorchain");
	clear();
	DESTEND("errorchain");
}

void errorchain::push(error *err)
{
	MDEBUG("[errorchain %p] Pusing %p (%s/%s).", this, err, err->who.c_str(), err->text.c_str());
	if(err != NULL)
	items.push_back(err);
}

void errorchain::clear()
{
	FOR_VECTOR(error*, items, e)
	{
		MDEBUG("[errorchain %p] Cleanup, deleting %p.", this, *e);
		delete *e;
	}
	items.clear();
}

void errorchain::stringify(vector<string*> &lines)
{
	FOR_VECTOR(error*, items, e) lines.push_back((*e)->as_string());
}
