/*
	main: the outer shell
	
	part of DerMixD
	(c)2004-2012 Thomas Orgis, licensed under GPLv2
*/

#include "basics.hxx"
#include "config.hxx"
#include "version.hxx"
#include "mixer.hxx"
#include "coms/socket_watch.hxx"
#include "coms/socketeer.hxx"
#include "coms/client_action_handler.hxx"
#include "param_init.hxx"
#include "tstring.hxx"
#include "coms/client_api.hxx"
#include "daemon.hxx"
#include "single.hxx"
#include "threads/semaphore.hxx"
#include "sigfried.hxx"

#include <string>
using std::string;
#include <vector>
using std::vector;

#include "debug.hxx"
#include "shortcuts.hxx"


int run_mixer();
bool read_config();
void print_help();

int main(int argc, char** argv)
{
	int ret = 0;
	bool goodcommand = true;
	param.program = "dermixd";
	param.shortcut("-c", "daemon=n");
	param.shortcut("-r", "remote=y");
	param.shortcut("-n", "defsetup=n");
	param.shortcut("-V", "version=y");
	param.shortcut("--version", "version=y");

	// I might build in some tolerance for build-time optional parameters.
	// Migration of old configs is also always a funny issue.
	goodcommand = read_config();
	if(!param.parse(argc, argv)) goodcommand = false;

	if(argc > 0)
	{
		MERROR("There is command line left after parsing (starting at \"%s\"). I am confused.", argv[0]);
		goodcommand = false;
	}

	if(goodcommand && param.as_bool("help"))
	{
		print_help();
		return 0;
	}
	else if(!goodcommand)
	{
		SERROR("Some command line / config file parsing trouble... please read the help (switch -h or parameter help=yes).");
		return 1;
	}
	else if(param.as_bool("version"))
	{
		printf("%s v%s.%s\n", program.c_str(), version.c_str(), subversion.c_str());
		return 0;
	}
	else if(param.as_bool("api"))
	{
		printf("Full API listing:\n\n");
		vector<string*> ah;
		dmd::api::allhelp(dmd::api::client_world, ah);

		FOR_VECTOR(string*, ah, line)
		printf("%s\n", (*line)->c_str());

		return 0;
	}
	else if(param.as_bool("param", "print"))
	{
		param.write();
		return 0;
	}

	if(!param.as_bool("tcp") && !param.as_bool("unix"))
	{
		SERROR("Please be reasonable and let me communicate via TCP or UNIX domain socket!");
		return 3;
	}

	if(param.as_bool("daemon"))
	{
		printf("Becoming a daemon...\n");
		if(!daemonize("/dev/null", param("output")))
		{
			SERROR("Daemonization failed!");
			return 2;
		}
	}

	// Before going multithreaded, there is setup code to run.
	// I want that to run before going multithreaded, but after forking.
	if(single_init() != 0){	SERROR("Single init did not work!"); return 3; }

	ret = run_mixer();

	// Single-thread finalize.
	single_exit();

	// After all threads exited... should I still worry about outdated cache for this variable?
	// There is no concurrent access anymore...
	if(dmd::sigfried::fatal_signal)
	{
		fprintf(stderr, "I got some fatal signal (%i), killing myself after ordered cleanup.\n", dmd::sigfried::fatal_signal);
		// Default handler should be in place.
		kill(getpid(), dmd::sigfried::fatal_signal);
	}

	return (ret == 0 ? EXIT_SUCCESS : EXIT_FAILURE);
}

int run_mixer()
{
	semaphore clients_go;
	// Data setup is simple: One mixer instance ( that includes management of anything audio and processing of commands.
	dmd::mixer jam_master(clients_go);
	// ...and one instance of the communication infrastructure.
	dmd::socket_watch watson(jam_master.actions, clients_go);

	fprintf(stderr, "Going to start the mixer...\n");

	// The plan:
	// First prepare the mixer engine, then hook up the communication.
	// At last, start the main mixer processing routine.

	// That should take care of initial channel setup.
	if(!jam_master.init())
	{
		MERROR("%s mixer initialization failed.", program.c_str());
		return 10;
	}

	// That fires up communication.
	if(!watson.init())
	{
		MERROR("%s communicator initialization failed.", program.c_str());
		return 20;
	}

	// If not daemonized, open an interactive console (not fancy, but functional).
	if(!param.as_bool("daemon")) watson.add_socketeer(new dmd::socketeer(STDIN_FILENO, STDOUT_FILENO, watson));

	// Watch out for TCP connections.
	if(param.as_bool("tcp"))
	if(!watson.add_tcp((porttype)param.as_uint("port"), param.as_bool("remote")))
	{
		MERROR("Unable to listen on TCP socket %s!", param("port").c_str());
		return 21;
	}

	// Watch out for UNIX domain socket connections.
	if(param.as_bool("unix"))
	if(!watson.add_unix(param("socket")))
	{
		MERROR("Unable to listen on UNIX domain socket %s!", param("socket").c_str());
		return 22;
	}

	// Defer signal handling to the specialized thread.
	dmd::sigfried::start(jam_master.actions);
	// That's the actual wixer work. Run the main loop.
	if(!jam_master.run())
	{
		MERROR("%s main mixer action failed.", program.c_str());
		return 30;
	}
	// Get my signals back.
	dmd::sigfried::stop();

	// I am building on the assumption that destruction of stack objects works in the correct order:
	// First destruct the socket watch, then the mixer.

	// If nothing went wrong, we finish normally.
	fprintf(stderr, "%s mixer finished by itself.\n", program.c_str());
	return 0;
}

bool read_config()
{
	bool good = true;
	vector<std::string> names;
	names.push_back(GLOBALCONFIG);
	const char *home = getenv("HOME");
	if(home != NULL)
	{
		names.push_back("");
		strprintf(names.back(), "%s/%s",home, LOCALCONFIG); 
	}
	FOR_VECTOR(std::string, names, n)
	{
		FILE *cf;
		MDEBUG("Trying config file '%s'.", (*n).c_str());
		if((cf=fopen((*n).c_str(), "r")) != NULL)
		{
			MDEBUG("Reading config file %s.", (*n).c_str());
			good = param.read(cf);
			fclose(cf);
			if(!good)
			{
				MERROR("Trouble reading config file '%s'.", (*n).c_str());
				break;
			}
		}
	}
	return good;
}

void print_help()
{
	printf("%s -- %s\n\n", program.c_str(), "a tcp-controlled audio file mixing daemon in the tradition of mixplayd (well, on the outside)");
	printf("This is version %s.%s, written and thus copyrighted in %s by Thomas Orgis. Licensed under the GPL 2. No warranty of any kind. Have fun;-)\n\n", version.c_str(), subversion.c_str(), years.c_str());
	printf("%s\n\n", "Usage: dermixd [parameters]");
	printf("\nThe parameters can also set via configuration file ($HOME/%s or %s).\n", LOCALCONFIG, GLOBALCONFIG);
	param.help(stdout);
}
