#ifndef DMD_H_DEBUG
#define DMD_H_DEBUG

// This header defines some macros that help to produce helpful messages that relate to their position in the code. It's all macros that, depending on switches, can reduce to no-op.

#include <stdio.h>

// pointer printf helper, to get consistent addresses with polymorphism
#define P(ptr) dynamic_cast<void*>(ptr)

#if \
	   (defined MDEBUG)    || (defined SDEBUG)   \
	|| (defined MXDEBUG)   || (defined SXDEBUG)  \
	|| (defined MWARNING)  || (defined SWARNING) \
	|| (defined MERROR)    || (defined SERROR)   \
	|| (defined MNOTE)     || (defined SNOTE)    \
	|| (defined CONSTRUCT) || (defined DESTRUCT) \
	|| (defined DESTBEGIN) || (defined DESTEND)
#error "Debugging macros are predefined! Conflict?"
#endif

// extreme debugging
#ifdef XDEBUG
#ifndef DEBUG
#define DEBUG
#endif
#endif

// We have a system... first letter
// M = more arguments than just a format string.
// S = printout of a simple string, no extra variables
// If there's a X as second letter, it's an extra-verbose message only to be enabled at higher debugging level.

#ifdef DEBUG
#undef NDEBUG
#define SDEBUG(s) fprintf(stderr, "[" __FILE__ ":%i] debug: %s\n", __LINE__, s)
#define MDEBUG(s, ...) fprintf(stderr, "[" __FILE__ ":%i] debug: " s "\n", __LINE__, __VA_ARGS__)
#else
#define NDEBUG
#define MDEBUG(s, ...) do {} while(0)
#define SDEBUG(s) do {} while(0)
#endif

#ifdef XDEBUG
#define MXDEBUG(s, ...) MDEBUG(s, __VA_ARGS__)
#define SXDEBUG(s) SDEBUG(s)
#else
#define MXDEBUG(s, ...) do {} while(0)
#define SXDEBUG(s) do {} while(0)
#endif

// General messages, warnings and errors without debugging context but with the added value of having file:line info prepended.

#define MWARNING(s, ...) fprintf(stderr, "[" __FILE__ ":%i] warning: " s "\n", __LINE__, __VA_ARGS__)
#define SWARNING(s) fprintf(stderr, "[" __FILE__ ":%i] warning: %s\n", __LINE__, s)
#define MERROR(s, ...) fprintf(stderr, "[" __FILE__ ":%i] error: " s "\n", __LINE__, __VA_ARGS__)
#define SERROR(s) fprintf(stderr, "[" __FILE__ ":%i] error: %s\n", __LINE__, s)

#define MNOTE(s, ...) fprintf(stderr, "[" __FILE__ ":%i] note: " s "\n", __LINE__, __VA_ARGS__)
#define SNOTE(s) fprintf(stderr, "[" __FILE__ ":%i] note: %s\n", __LINE__, s)

// Constructor / destructor tracing, single argument is the class name.

#ifdef DEBUG_MEM
#define MEMDEBUG
#endif

#ifdef MEMDEBUG
#define CONSTRUCT(s) fprintf(stderr, "_C %p %s\n", this, s)
#define DESTBEGIN(s) fprintf(stderr, "_D %p %s\n", this, s)
#define DESTEND(s) fprintf(stderr, "_F %p %s\n", this, s)
#define DESTRUCT(s) DESTBEGIN(s); DESTEND(s)
#else
#define CONSTRUCT(s) do {} while(0)
#define DESTBEGIN(s) do {} while(0)
#define DESTEND(s) do {} while(0)
#define DESTRUCT(s) do {} while(0)
#endif


#endif
