/*
	socketeer: The thread communicating with clients (usually via a socket).
	
	part of DerMixD
	(c)2004-2011 Thomas Orgis, licensed under GPLv2
	
	Be very careful for cancelability of the thread. Generally, it is only cancelable for selected pieces of code (blocking for I/O or a semaphore).
	Generally, when allocating any memory (implicitly, via calls to external code...) it must be assured that the thread cannot be cancelled while doing so; or, if it's local memory that is not cleaned up by some destructor of a persistent object, the thread frees it itself. 
*/

#include "basics.hxx"
#include "version.hxx"

#include "coms/netcomm.hxx"

#include "coms/socketeer.hxx"
#include "coms/socket_watch.hxx"
#include "browse.hxx"

#include <string.h>

#include <string>
#include <vector>

#include "debug.hxx"

using std::vector;
using std::string;

namespace dmd
{

socketeer::socketeer(int sock, socket_watch &mummy):
	 thread("socket reader", nice::socketeer)
	,sb(sock), id(sock)
	,writer(sb, mummy.bnd)
	,mum(&mummy), cd(), actor(*this, writer, mummy)
{
	CONSTRUCT("socketeer");
}

socketeer::socketeer(int in_fd, int out_fd, socket_watch &mummy):
	 thread("socket reader", nice::socketeer)
	,sb(in_fd, out_fd, false), id(in_fd)
	,writer(sb, mummy.bnd)
	,mum(&mummy), cd(), actor(*this, writer, mummy)
{
	CONSTRUCT("socketeer");
}

socketeer::~socketeer()
{
	DESTBEGIN("socketeer");
	DESTEND("socketeer");
}

/*
	socket communication handler
	
	here we get commands, parse them, then pass to the mixer (possibly waiting for result from different thread) and normally respond to client
*/
void socketeer::thread_work()
{
	// No dynamic memory in this function ... we can get cancelled in some other method call!
	MDEBUG("[socketeer %p] startup", this);

	//hello
	writer.write_copy("[connect] " + program + " v" + version);
	MDEBUG("[socketeer %p] Waiting until I am allowed to work with the mixer...", this);
	// Block on the semaphore if it is zero, repost immediately to make the one signal heard by everyone.
	mum->clients_go->wait(this);
	mum->clients_go->post();

	while(1) //run endlessly until close/shutdown or read error
	{
		//get command string from socket
		ms = "";
		// The integer is safe on the stack, also with cancellation.
		int erg = sb.read_line(ms, this);

		if(erg < 0)
		{
			MERROR("[socketeer %p] Error on reading: %i, faking close command", this, erg);
			sb.read_line(ms, this);
			MERROR("[socketeer %p] still there after second read", this, erg);
			ms = "close";
		}

		if(ms == "")
		{
			MDEBUG("[socketeer %p] got empty string", this);
			continue;
		}

		MDEBUG("[socketeer %p] got msg: %s", this, ms.c_str());

		// Let the spies get their juice.
		if(mum->bnd.active)
		{
			mum->bnd.call_em();
			mum->bnd.hand_over_incoming(ms, writer.id);
			mum->bnd.hang_up();
		}

		// The meat. The business.
		// I am not interested in the return value here, any error handling is dealt with by giving the generated messages to the client.
		actor.handle(ms);
		// The actor should trigger as less writing as possible itself.
		writer.writes(actor.answers);
		actor.answers.clear();

		// The end, voluntarily.
		if(actor.got_final)
		{
			MDEBUG("[socketeer %p] This action was final... good-bye.", this);
			// It registers itself for removal... even on global QUIT, this shall work (or at least be ignored, because all socketeers are killed anyway.
			mum->socketeer_to_reaper(this);
			return;
		}
	} //end of endless loop
}

} // namespace
