/*
	socket_writer: The thread that is responsible for writing on a socket.
	
	part of DerMixD
	(c)2005-9 Thomas Orgis, licensed under GPLv2
	
	depends on netcomm.h to provide actual socket reading/writing
*/

#include "basics.hxx"
#include "coms/socket_writer.hxx"
#include "coms/spies.hxx"
#include "version.hxx"
#include <vector>
#include <string>
#include <sstream>
#include "shortcuts.hxx"
#include "debug.hxx"

using namespace std;

#define ME "socket_writer"

namespace dmd {

socket_writer::socket_writer(socket_buffer &mysink, spies &mybnd):
	 thread("socket writer", dmd::nice::socketeer), id(mysink.insock)
	,sink(&mysink), bnd(&mybnd), is_end(false)
{
	CONSTRUCT(ME);
}

socket_writer::~socket_writer()
{
	DESTBEGIN(ME);
	FOR_VECTOR(swm, messages, i)
	{
		MXDEBUG("destruct: deleting leftover string at %p", (void*) i->msg);
		delete i->msg;
		SXDEBUG("destruct: done");
	}
	DESTEND(ME);
}

void socket_writer::thread_work()
{
	while(true)
	{
		// I do not want to get interrupted during flushing messages.
		// So, you can only cancel the thread while waiting for the semaphore.
		semo.wait(this);
		// Now flush it, handing it over to spies while at that.
		flush();
	}
}

void socket_writer::end()
{
	thread_kill();
	flush();
	// A last uttering, especially for those clients that get involuntarily kicked out of communication...
	string goodbye = "[close] " + program + " shutting down communication. Enjoy the silence!";
	sink->writeln(goodbye);
}

// Add message to internal list and poke the writer thread.
void socket_writer::write(string *msg, bool spy)
{
	lock.lock();
	if(msg == NULL || msg->c_str() == NULL)
	{
		SERROR("Tried to push null pointer! Must be a programming error ... triggering a segfault (look at the coredump...");
		printf("%s\n", msg->c_str());
	}
	else messages.push_back(swm(msg, spy));

	// This poke could be better off outside the locked section...
	semo.post(); // Poke!
	lock.unlock();
}

void socket_writer::writes(vector<string*> &msgs)
{
	lock.lock();
	FOR_VECTOR(string*, msgs, i)
	{
		if(*i == NULL || (*i)->c_str() == NULL)
		{
			SERROR("Tried to push null pointer! Must be a programming error ... triggering a segfault (look at the coredump...");
			printf("%s\n", (*i)->c_str());
		}
		else
		{
			messages.push_back(swm(*i, false));
			// Hm, poking once for each message.
			semo.post();
		}
	}
	lock.unlock();
	// Strings are deleted in the writer...
}

void socket_writer::flush()
{
	// First, grab all current messages and move them into a private list.
	// That way, new messages can still arrive while flushing the old ones, minimizing blocking.
	lock.lock();
	// Construct a copy.
	std::vector<swm> old_messages = messages;
	messages.clear();
	semo.zero(); // No need to wake the writer thread for any of the old messages.
	lock.unlock();

	// Now we are busy writing here, the outside world can continue to happily post new messages.

	bool spying = bnd->active;
	if(spying) bnd->call_em();
	FOR_VECTOR(swm, old_messages, i)
	{
		sink->writeln(*i->msg);
		// Now give all spies their info... all but myself!
		if(spying && !i->spied) bnd->hand_over_outgoing(*i->msg, id);

		delete i->msg;
	}
	if(spying) bnd->hang_up();
}

void socket_writer::write_copy(const string &msg, bool spy)
{
	string* tmsg = new string(msg);
	write(tmsg, spy);
}

void socket_writer::write_copies(vector<string*> &msgs)
{
	lock.lock();
	FOR_VECTOR(string*, msgs, i)
	{
		if(*i == NULL || (*i)->c_str() == NULL)
		{
			SERROR("Tried to push null pointer! Must be a programming error ... triggering a segfault (look at the coredump...");
			printf("%s\n", (*i)->c_str());
		}
		else
		{
			// allocation failures???
			string *tmsg = new string;
			*tmsg = **i;
			messages.push_back(swm(tmsg, false));
			semo.post();
		}
	}
	lock.unlock();
}

}
