#ifndef DMD_H_SOCKET_WATCH
#define DMD_H_SOCKET_WATCH

/*
	socket watch: Watch out for new connections, spawn handlers for these connections.

	part of DerMixD
	(c)2004-9 Thomas Orgis, licensed under GPLv2

	This shall be just the management shell over real communication! Creating and destroying the actual communication threads.
*/

#include "common.hxx"
#include "action.hxx"
#include "actionlist.hxx"
#include "threads/threads.hxx"
#include "coms/socket_writer.hxx"
#include "peers.hxx"
#include "coms/spies.hxx"
#include "coms/netcomm.hxx"

#include <vector>
#include <string>

namespace dmd
{

// The socket watch is the organisation of multiple single socket watchers.
// We have several threads originating from here:
// - the actual watcher threads, waiting for new connections
// - the socketeer reaper thread
// - the spawned communication threads, socketeer with associated socket_writer

// The socket watch also includes the infrastructure for spying on socket communication.

// Needed forward declarations... the watch comes later here, the socketeer in a separate header.
class socketeer;
class socket_watch;

class socket_watcher: public thread
{
	private:
	socket_listener* sl;
	int sock;
	socket_watch *mum;

	public:
	// The constructor takes over a socket listener which will be deleted on destruction.
	 socket_watcher(socket_listener *sli, socket_watch* mummy);
	~socket_watcher();
	void thread_work();
};

class socket_reaper: public thread
{
	private:
	// List of candidates for deletion.
	// That includes the associated socket writers.
	std::vector<socketeer*> deadites;
	real_mutex lock;
	semaphore sem;
	// Action happens via the socket watch.
	socket_watch *mum;

	public:
	 socket_reaper(socket_watch *mummy);
	~socket_reaper();
	// Push a new socketeer to be deleted.
	void push_deadite(socketeer *deadite);
	// Empty deadite list.
	void forget();

	void thread_work();
};

// This class is a container for the socket watcher threads and the spawned communication threads.
// Also it contains the socket reaper thread that is responsible 

class socket_watch: public thread_container
{
	private:
	// Add a precreated watcher to the list, start the thread.
	// Deletes the watcher if thread not startable or socket not online.
	bool add_watcher(socket_listener *sl);
	// We need one socket reaper thread.
	socket_reaper reaper;
	// Multiple watchers (TCP, UNIX domain).
	std::vector<socket_watcher*> watchers;
	// Socket communication is divided for input and output.
	// Each socketeer data structure _is_ the reader thread and contains a writer thread.
	// But the socketeer is no thread container itself, so the socket watch needs to keep track of both threads!
	std::vector<socketeer*> socketeers;
	// A mutex for serializing access to the socketeer list.
	real_mutex socket_lock;
	// Stop socket threads and delete socketeer object.
	void finish_socketeer(socketeer *deadite);

	public:
	// Socketeers need to access the spying/communication infrastructure.
	spies bnd; // Tapping on other's communication.
	peers pee; // Communicate with others.
	// Pointer to mixer's action list, socketeers need to work on this.
	actionlist *actions;
	// Semaphore that we wait for before actually accepting connections from clients.
	semaphore *clients_go;

	// Constructor does only minimal work.
	// For starting any threads, use init().
	 socket_watch(actionlist &acts, semaphore &clients_go);
	// Likewise, you can call exit() to kill the threads... but he destructor tries to ensure that, too.
	~socket_watch();

	// Actual initialization, starting reaper thread.
	// If this returns false... well, then it failed and the state should be unchanged.
	bool init();
	// Kills/deletes all threads.
	// It tries to be nice and sends good-bye messages to sockets that are still there.
	// Note: This is called in the destructor automatically.
	void exit();

	// External hooks to actually add socket watcher threads.
	// This code is not designed to be called from multiple threads... we could add some mutexes though, if needed.
	// Also, one could add code to remove watchers at runtime... but that's not needed yet.
	// Add a watcher on a TCP port (IPv4 for now...)
	bool add_tcp(const porttype port, const bool remote);
	// Add a watcher on a local UNIX domain socket.
	bool add_unix(const std::string name);

	// Socket watchers use this to start socketeer+socket_writer threads.
	void add_socketeer(int sock);
	// Same with any freshly created socketeer.
	void add_socketeer(socketeer *);
	// This is called to mark a socketeer for deletion.
	// Actually a thin wrapper over the hidden interface of the socket reaper.
	void socketeer_to_reaper(socketeer *deadite);
	// The socket reaper uses this to actually end a socketeer's life.
	void reap_socketeer(socketeer *deadite);
};

} // namespace

#endif
