/*
	socket watch: Watch out for new connections, spawn handlers for these connections.
	
	part of DerMixD
	(c)2004-2009 Thomas Orgis, licensed under GPLv2
*/

#include "basics.hxx"
#include "coms/socket_watch.hxx"
#include "coms/socketeer.hxx"

#include "debug.hxx"
#include "shortcuts.hxx"

namespace dmd
{

// =====================================
// The Socket Watch
// =====================================


socket_watch::socket_watch(actionlist &acts, semaphore &go_clients):
	reaper(this), actions(&acts), clients_go(&go_clients)
{
	CONSTRUCT("socket_watch");
}

socket_watch::~socket_watch()
{
	DESTBEGIN("socket_watch");
	exit();
	DESTEND("socket_watch");
}

bool socket_watch::init()
{
	return create_thread(&reaper);
}

void socket_watch::exit(void)
{
	// Always take care for all watchers.
	MDEBUG("Reaping %zu watchers.", watchers.size());
	FOR_VECTOR(socket_watcher*, watchers, w)
	{
		kill_thread(*w);
		delete *w;
	}
	watchers.clear();

	MDEBUG("[socket_watch %p] killing reaper %p", this, &reaper);
	// Reaper must be killed before messing with socketeers here.
	kill_thread(&reaper);
	reaper.forget();

	MDEBUG("[socket_watch %p] reaping all socketeers", this);
	socket_lock.lock();

	bnd.eliminate_all();
	pee.eliminate_all();

	FOR_VECTOR(socketeer*, socketeers, s)
	{
		MDEBUG("[socket_watch %p] reaping socketeer %i", this, (*s)->id);
		finish_socketeer(*s);
	}
	socketeers.clear();
	socket_lock.unlock();
	MDEBUG("[socket_watch %p] exit done", this);
}

bool socket_watch::add_watcher(socket_listener *sl)
{
	if(!sl->online)
	{
		MERROR("[socket_watch %p] Listener not online!", this);
		delete sl;
		return false;
	}

	socket_watcher *wd = new socket_watcher(sl, this);

	if(create_thread(wd))
	{
		watchers.push_back(wd);
		return true;
	}
	else
	{
		MERROR("[socket_watch %p] Cannot start watcher!", this);
		delete wd;
		return false;
	}
}

bool socket_watch::add_tcp(const porttype port, const bool remote)
{
	socket_listener *sl = new port_listener(port,remote);
	fprintf(stderr, "[socket_watch %p] Wanting to listen on port %"PRIiMAX", remote mode: %s\n", this, (intmax_t)port, remote ? "on" : "off");
	return add_watcher(sl);
}

bool socket_watch::add_unix(const std::string name)
{
	socket_listener *sl = new unix_listener(name);
	fprintf(stderr, "[socket_watch %p] Wanting to listen on UNIX socket '%s'.\n", this, name.c_str());
	return add_watcher(sl);
}

void socket_watch::add_socketeer(int sock)
{
	add_socketeer(new socketeer(sock, *this));
}

void socket_watch::add_socketeer(socketeer *aramis)
{
	if(!aramis) return;
	// The socketeer constructor includes construction of the contained socket_writer.
	// Just no thread is started yet!
	// It's a bit tricky to decide in what order to add to list and create threads.
	// The socketeer thread might immediately loose its connection and request being reaped... in that case, it should be in the list already... or not?
	// At least the socket_writer is uncritical.
	if(!create_thread(&aramis->writer))
	{
		MERROR("Unable to create socket writer thread for %i.", aramis->id);
		delete aramis;
	}

	socket_lock.lock();
	socketeers.push_back(aramis);
	socket_lock.unlock();

	if(!create_thread(aramis))
	{
		MERROR("Unable to create socket reader thread for %i.", aramis->id);
		// Hand it over to normal reaping.
		socketeer_to_reaper(aramis);
	}
}

void socket_watch::socketeer_to_reaper(socketeer *deadite)
{
	reaper.push_deadite(deadite);
}

void socket_watch::reap_socketeer(socketeer *deadite)
{
	bool found = false;
	// Going to possibly reap a spy, so be safe here and eliminate it first.
	// I want to stop other from sending messages to a dead spy...
	bnd.eliminate(&deadite->writer);
	pee.eliminate(&deadite->writer);

	MDEBUG("[socket_watch %p] going to lock socketeers", this);
	// Remove it from my list.
	socket_lock.lock();
	FOR_VECTOR(socketeer*, socketeers, portos)
	if(*portos == deadite)
	{
		found = true;
		socketeers.erase(portos);
		break;
	}
	MDEBUG("[socket_watch %p] unlocking socketeers", this);
	socket_lock.unlock();

	// If the socketeer was not in the list, we assume it has been deleted already by another thread.
	if(found) finish_socketeer(deadite);
	else MERROR("Tried to delete socketeer at %p, which is not in my list (deleted already?)!", deadite);
}

void socket_watch::finish_socketeer(socketeer *deadite)
{
	MDEBUG("[socket_watch %p] finishing socketeer %p", this, deadite);
	// 1. First stop the reader.
	kill_thread(deadite);

	// 2. Tell the writer to quit.
	MDEBUG("[socket_watch %p] reader %p killed, ending writer %p", this, deadite, &deadite->writer);
	deadite->writer.end();
	// The thread should be killed already, but still calling this to remove it from list.
	kill_thread(&deadite->writer);

	MDEBUG("[socket_watch %p] reader %p killed, writer %p ended ... deleting", this, deadite, &deadite->writer);
	// Now no one should still need the socketeer's data... so, away with it!
	delete deadite;
	MDEBUG("[socket_watch %p] socketeer %p gone", this, deadite);
}

// =====================================
// A Single Socket Watcher
// =====================================

socket_watcher::socket_watcher(socket_listener *sli, socket_watch* mummy):
	thread("socket watcher", nice::sockets), sl(sli), sock(-1), mum(mummy)
{
	CONSTRUCT("socket_watcher");
}

socket_watcher::~socket_watcher()
{
	DESTBEGIN("socket_watcher");
	thread_kill(); // Playing safe.
	if(sl != NULL) delete sl;

	// The thread could have been cancelled just after accepting a connection.
	if(sock >= 0) close(sock);
	DESTEND("socket_watcher");
}

// Wait for connections on a socket and spawn socketeers to handle them.
void socket_watcher::thread_work()
{
	while(1)
	{
		// Wait for a new socket connection, while being cancelable.
		sock = -1;
		MDEBUG("[socket_watcher %p] Waiting for connection...", this);
		sock = sl->accept_connection(this);
		MDEBUG("[socket_watcher %p] Got one (perhaps)!", this);

		// Now handle it
		if(sock >= 0)
		{
			MDEBUG("[socket_watcher %p] Adding socketeer on socket %i", this, sock);
			mum->add_socketeer(sock);
		}
		else SERROR("[socket_watcher %p] Problem accepting a connection!");
	}
}


// =====================================
// The Socket Reaper
// =====================================


socket_reaper::socket_reaper(socket_watch *mummy): thread("socket reaper", nice::socket_reaper), mum(mummy)
{
	CONSTRUCT("socket_reaper");
}

// It should not be necessary to kill the thread here, but we play safe.
socket_reaper::~socket_reaper()
{
	DESTBEGIN("socket_reaper");
	thread_kill();
	DESTEND("socket_reaper");
}

void socket_reaper::thread_work()
{
	while(42)
	{
		sem.wait(this);

		lock.lock();
		MDEBUG("[reaper %p] reaping %zu socketeers", this, deadites.size());
		// There is a list of socketeers to be wiped out properly; without zombies!
		FOR_VECTOR(socketeer*, deadites, deadite)
		{
			mum->reap_socketeer(*deadite);
			MDEBUG("[reaper %p] reaped %p", this, *deadite);
		}

		deadites.clear();
		lock.unlock();

		MDEBUG("[reaper %p] done for now", this);
	}
}

void socket_reaper::push_deadite(socketeer *deadite)
{
	lock.lock();
	MDEBUG("pushing %p on deadites list", deadite);
	deadites.push_back(deadite);
	lock.unlock();
	sem.post();
}

void socket_reaper::forget()
{
	lock.lock();
	deadites.clear();
	lock.unlock();
}

} // namespace
