/*
	netcomm: tcp and unix port watching and basic socket communication

	part of DerMixD
	(c)2004-2012 Thomas Orgis, licensed under GPLv2
	
	initial tcp code derived from:
	
	netcomm.c for mixplayd
	
	provides functions for communiction on a tcp socket
	(c) Siegfried Wagner 01/06/27
	
	license: GNU General Public License (GPL)
	         For details see the file COPYING included with this archive
	last update: 01/08/28 by Siegfried Wagner
*/

// Hack for solaris headers that contain a struct mutex.
#define mutex solaris_mutex
#include "basics.hxx"
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <fcntl.h>
#include <sstream>
#include <sys/un.h>
#include <strings.h> // bzero
#undef mutex

#include "coms/netcomm.hxx"
#include "syserror.hxx"
#include "threads/threads.hxx"

#include "debug.hxx"

using namespace std;

//MSG_NOSIGNAL doesn't exist on Tru64...
//but it is _very_ handy when the client already closed the connection and I want to write, say, just \r\n
//without nosignal, dermixd dies...
//so, let's just ignore SIGPIPE in the main thread and additionally define it to zero here if MSG_NOSIGNAL is not there...
//I guess that even when ignoring the signal in won't hurt preventing it in recv/send.
//Beware: dermixd even dies on careless send(..., "\r\n", ) if the client is away regardless of MSG_NOSIGNAL or ignoring - but only on slow machines or otherwise slowed down (inside gdb)!
#ifndef MSG_NOSIGNAL
#define MSG_NOSIGNAL 0
#endif

// Create a TCP socket.
int bind_tcp(porttype port_number, bool remote);
// Create a UNIX domain socket in file system.
int bind_unix(const char *name);


socket_listener::socket_listener(void)
{
	CONSTRUCT("socket_listener");
	online = false;
}

socket_listener::~socket_listener()
{
	DESTRUCT("socket_listener");
}

port_listener::port_listener(porttype port_number, bool remote)
{
	CONSTRUCT("port_listener");
	sock = bind_tcp(port_number, remote);
	online = (sock < 0) ? false : true;
}

unix_listener::unix_listener(const string name)
{
	CONSTRUCT("unix_listener");
	sockname = name;
	sock = bind_unix(name.c_str());
	online = (sock < 0) ? false : true;
}

// Just close the TCP port.
port_listener::~port_listener()
{
	DESTBEGIN("port_listener");
	if(online){ close(sock); }
	DESTEND("port_listener");
}

// Close the socket and remove the stale file system entry.
unix_listener::~unix_listener()
{
	DESTBEGIN("unix_listener");
	if(online)
	{
		close(sock);
		unlink(sockname.c_str());
	}
	DESTEND("unix_listener");
}

int bind_tcp(porttype port_number, bool remote)
{
	int sock;
	struct sockaddr_in svr;
	int flag=1;
	syserror serr;
	
	bzero((char*) &svr, sizeof(svr)); //important on Tru64/Alpha!
	// TODO: Make this IPv6, too... or just add a selection.
	svr.sin_family = AF_INET;
	svr.sin_port = htons(port_number);
	if(!remote) svr.sin_addr.s_addr = inet_addr("127.0.0.1");
	else svr.sin_addr.s_addr = INADDR_ANY;
	
	if( (sock = socket(AF_INET, SOCK_STREAM, 0)) < 0)
	{
		perror ("error creating listening socket");
		return -1;
	}
	
	if(setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &flag, sizeof(flag)) < 0)
	{
		perror ("error setting SO_REUSEADDR on listening socket");
		return -1;
	}
	
	if(bind(sock, (struct sockaddr *) &svr, sizeof(svr)) < 0)
	{
		fprintf(stderr, "bind: failed to bind to port %d: %s.\n\n", port_number, serr.str());
		if(port_number < 1024 && getuid ())
			fprintf (stderr, "A likely reason is that you tried to bind low-numbered ports but you are not root!\n");
		close (sock);
		return -1;
	}
	
	if (listen (sock, SOMAXCONN) < 0)
	{
		fprintf(stderr, "error listening: %s\n", serr.str());
		close (sock);
		return -1;
	}
	
	//I want blocking
	//fcntl(sock, F_SETFL, O_NONBLOCK);
	fcntl(sock, F_SETFD, FD_CLOEXEC); //close on exec (after forking)
	
	return sock;
}

int bind_unix(const char *name)
{
	int sock;
	struct sockaddr_un svr;
	syserror serr;

	if(strlen(name) >= sizeof(svr.sun_path)/sizeof(char))
	{
		MERROR("Too long UNIX domain socket name: %s", name);
		return -1;
	}
	bzero((char*) &svr, sizeof(svr)); //important on Tru64/Alpha?

	if( (sock = socket(AF_UNIX, SOCK_STREAM, 0)) < 0)
	{
		MERROR("Unable to create listening UNIX domain socket (%s).", serr.str());
		return -1;
	}
	svr.sun_family = AF_UNIX;
	strcpy(svr.sun_path, name);
	unlink(name);
	if(bind(sock, (struct sockaddr *) &svr, sizeof(svr)) < 0)
	{
		MERROR("Bind for UNIX domain socket \"%s\" failed: %s.", name, serr.str());
		close (sock);
		return -1;
	}
	if(listen (sock, SOMAXCONN) < 0)
	{
		MERROR("No listening: %s", serr.str());
		close(sock);
		return -1;
	}
	fcntl(sock, F_SETFD, FD_CLOEXEC); //close on exec (after forking)

	return sock;
}

//wait for and accept connection requests (delivering new socket for communication)
int socket_listener::accept_connection(dmd::thread *ted)
{
	//struct sockaddr_in cli;
	//socklen_t clilen;
	int clisock;
	
	//clilen = (socklen_t) sizeof(cli);
	//clisock = accept(sock, (struct sockaddr *)&cli, &clilen);
	if(ted != NULL) ted->thread_cancelable(true);

	clisock = accept(sock, NULL, NULL);

	if(ted != NULL) ted->thread_cancelable(false);
	
	//I want blocking...
	//fcntl(clisock, F_SETFL, O_NONBLOCK);
	fcntl(clisock, F_SETFD, FD_CLOEXEC); //close on exec (after forking)
	
	return clisock;
}

//simple reader object

socket_buffer::socket_buffer(int in_fd, int out_fd, bool close_on_end_):
	pos(0), fill(0), close_on_end(close_on_end_), insock(in_fd), outsock(out_fd)
{
	CONSTRUCT("socket_buffer");
	// The default case: One file descriptor for input and output.
	if(outsock < 0) outsock = insock;

	buf[MSG_BUF_SIZE-1] = '\0';
}

socket_buffer::~socket_buffer()
{
	DESTBEGIN("socket_buffer");
	if(close_on_end)
	{
		close(insock);
		if(outsock != insock) close(outsock);
	}
	DESTEND("socket_buffer");
}

/*
algo:

when still sth. in buffer (indicated by buffed == true, beginning at pos)
begin search for next \n
if found before end: make \0, add to msg, update the pos, 

this may go on in a loop if the line to read is really long... 

*/
int socket_buffer::read_line(string& msg, dmd::thread *ted)
{
	int msglen = 0;
	int i;
	
	while(1)
	{
		//cout << "fill: " << fill << endl;
		if(fill)
		{
			//search for next line end
			for(i=pos;i<fill;++i)
			{
				// Use numeric values for LF and CR, cause we are not speaking native text here but network socket.
				if(buf[i] == 0x0a){ buf[i] = '\0'; break; }
				if(buf[i] == 0x0d){ buf[i] = '\0'; }
			}
			//cout << "i: " << i << endl;
			if( (msglen += i - pos) > MAX_MSG_LENGTH )
			{
				MERROR("msglen of %i is greater than %i", msglen, MAX_MSG_LENGTH);
				fill = 0;
				pos = 0;
				return -2; //interpet this!
			}
			//copy line starting at pos and ending at i
			msg += (buf + pos);
			if(i < fill) //really found a \n
			{
				//the line was the buffer: i is last character
				if(i == fill-1)
				{
					fill = 0;
					pos = 0;
				}
				//there is sth. left: i is smaller
				else
				{
					pos = i + 1;
				}
				return msglen;
			}
			//not at end: reading sth. next time...
			else{ fill = 0; pos = 0; } //pos=0 here or a few lines below?
		}
		else
		{
			pos = 0;
			MDEBUG("[socket_buffer %i|%i] refill", insock, outsock);
			if(ted != NULL) ted->thread_cancelable(true);

			fill = read(insock, buf, MSG_BUF_SIZE-1);

			if(ted != NULL) ted->thread_cancelable(false);

			MDEBUG("[socket_buffer %i|%i] got %zd bytes", insock, outsock, fill);
			if(fill < 1)
			{
				//we would have blocked if all was right, wouldn't we?
				//getting no data means that sth. is broken...
				//an error is an error anyway
				fill=0;
				msg = "";
				return -1;
			}
		}
	}
}

int socket_buffer::write_exactly(string& msg)
{
	if (outsock<0) return -1;

	MDEBUG("[socket_buffer %i|%i] going to write \"%s\"", insock, outsock, msg.c_str());
	return write(outsock, msg.c_str() , msg.length());
}

int socket_buffer::writeln(string &msg)
{
	if (outsock<0) return -1;

	MDEBUG("[socket %i|%i] going to write \"%s\"", insock, outsock, msg.c_str());
	if(write(outsock, msg.c_str() , msg.length()) != -1)
	return write(outsock, "\x0d\x0a", 2);
	else return -1;
}
