/*
	client_api: The action definitions that matter to DerMixD clients.
	
	part of DerMixD
	(c)2006-2012 Thomas Orgis, licensed under GPLv2
	
	This is the place where the main mixer actions - ro be received and prepared by socketeers - are defined.
	
	Have a look at the def struct in action.hxx to get an idea about how this works.
	Be aware of the ACT_MAX_ARG limit that currently says that an action definition can contain only 5 arguments!
	This limit can easily changed in action.hxx, though.
	
	Define an action as element of this list via
	
	{ <numerical id>, "command", "info text", <number of arguments>,
		{ <list of argument names> },
		{ <list of numerical argument types> },
		<flags>
	}
	
	The numerical id is used for easy identification of an action in both socketeers and mixer.
	Command and info text should be obvious...
	It may be beneficial to know the possible argument types: ACT_INT for integers, ACT_FLOAT for floats (doubles, actually), ACT_STRING for strings
	
	Also the last entry, flags, may need some explanation. As one may already have seen, you can put the OR of some flag constants in here to activate their flags.
	There are some flags for use inside the action's code:
	
	ACT_CONTAINER means that the action wraps over further action(s), the last argument must be a string that can be parsed into the subaction to be contained
	ACT_NOSUB prohibits placement of this action inside of a container
	ACT_OPTARG makes arguments optional; the action may be either parsed with setting all or none of the arguments.
	
	Then, there are some flags with meaning to socketeer/mixer:
	
	SOCK_LOCAL: handling of action inside socketeer without bothering the mixer
	SOCK_NOTIFY: triggered action reports back to socketeer, which has to notify the mixer afterwards
	SOCK_POST: this action needs postprocessing by socketeer before sending to mixer
	SOCK_IN: action on an inchannel (channel id as first integer argument required!)
	SOCK_OUT: action on an outchannel (channel id as first integer argument required!)
	SOCK_ANS: a specialized answer is to be given (involding a return value...)
	SOCK_COMPANY: The action is only valid in a full setup where there is a community of socketeers (a socket_watch).

	Be sure not to omit the last entry with id 0! That one is the end marker... without it you'll get prompt segfaulting!
*/

#include "coms/client_api.hxx"

#include "debug.hxx"

namespace dmd {
namespace api {

const def in_start =
{ IN_START, "start", "start playback on inchannel", 1,
	{ "channel" },
	{ ACT_SIZE },
	SOCK_IN
};
const def out_start =
{ OUT_START, "outstart", "start playback on outchannel", 1,
	{ "channel" },
	{ ACT_SIZE },
	SOCK_OUT
};
const def in_script =
{ IN_SCRIPT, "script", "script an action triggered by an inchannel passing certain time/position, command is just any valid (and allowed) command with arguments", 3,
	{ "channel", "time in seconds", "command" },
	{ ACT_SIZE, ACT_FLOAT, ACT_STRING },
	ACT_CONTAINER|ACT_NOSUB|SOCK_IN|SOCK_POST
};
const def in_nscript =
{ IN_SCRIPT, "nscript", "script an action triggered by an inchannel passing certain time/position, script will be executed n times (one for each triggering)", 4,
	{ "channel", "n", "time in seconds",  "command" },
	{ ACT_SIZE, ACT_INT, ACT_FLOAT, ACT_STRING },
	ACT_CONTAINER|ACT_NOSUB|SOCK_IN
};

const def client_world[] =
{
	{ IN_VOLUME, "volume", "set inchannel volume factor(s)", 3,
		{ "channel", "volume(left/both)", "volume(right)" },
		{ ACT_SIZE, ACT_FLOAT, ACT_FLOAT },
		SOCK_IN|SOCK_ANS|ACT_OPTONE|SOCK_POST
	},
	{ IN_PREAMP, "preamp", "set inchannel preamplification", 2,
		{ "channel", "dB" },
		{ ACT_SIZE, ACT_FLOAT },
		SOCK_IN|SOCK_ANS
	},
	{ IN_BASS, "bass", "set inchannel bass eq factor", 2 ,
		{ "channel", "value" },
		{ ACT_SIZE, ACT_FLOAT},
		SOCK_IN|SOCK_ANS
	},
	{ IN_MID, "mid", "set inchannel mid eq factor", 2 ,
		{ "channel", "value" },
		{ ACT_SIZE, ACT_FLOAT},
		SOCK_IN|SOCK_ANS
	},
	{ IN_TREBLE, "treble", "set inchannel treble eq factor", 2 ,
		{ "channel", "value" },
		{ ACT_SIZE, ACT_FLOAT},
		SOCK_IN|SOCK_ANS
	},
	{ IN_EQ, "eq", "set all three eq factors", 4 ,
		{ "channel", "bass", "mid", "treble" },
		{ ACT_SIZE, ACT_FLOAT, ACT_FLOAT, ACT_FLOAT},
		SOCK_IN
	},
	{ IN_PAUSE, "pause", "pause inchannel", 1 ,
		{ "channel" },
		{ ACT_SIZE },
		SOCK_IN
	},
	{ IN_SPEED, "speed", "set playback speed factor (1=normal)", 2 ,
		{ "channel", "value" },
		{ ACT_SIZE, ACT_FLOAT},
		SOCK_IN|SOCK_ANS 
	},
	{ IN_PITCH, "pitch", "increase/decrease playback speed factor", 2 ,
		{ "channel", "value" },
		{ ACT_SIZE, ACT_FLOAT},
		SOCK_IN|SOCK_ANS 
	},
	in_start,
	{ IN_SEEK, "seek", "absolute seek on inchannel", 2 ,
		{ "channel", "position in seconds" },
		{ ACT_SIZE, ACT_FLOAT },
		SOCK_NOTIFY|SOCK_IN|SOCK_ANS 
	},
	{ IN_RSEEK, "rseek", "relative (from current position) seek on inchannel", 2 ,
		{ "channel", "offset in seconds" },
		{ ACT_SIZE, ACT_FLOAT },
		SOCK_NOTIFY|SOCK_IN|SOCK_ANS 
	},
	{ BIND, "bind", "bind input channel to output channel", 2 ,
		{ "inchannel", "outchannel" },
		{ ACT_SIZE, ACT_SIZE },
		0
	},
	{ UNBIND, "unbind", "release bond between input channel and output channel", 2 ,
		{ "inchannel", "outchannel" },
		{ ACT_SIZE, ACT_SIZE },
		0
	},
	{ OUT_EJECT, "outeject", "Un-load resource on outchannel (eject the tape).", 1,
		{ "channel" },
		{ ACT_SIZE },
		SOCK_OUT 
	},
	out_start,
	in_script,
	in_nscript,
	{ IN_SHOWSCRIPT, "showscript", "show the script commands for programmed actions on an inchannel", 1,
		{ "channel" },
		{ ACT_SIZE },
		ACT_NOSUB|SOCK_IN 
	},
	{ IN_CLEARSCRIPT, "delscript", "remove scripting actions from inchannel", 1,
		{ "channel" },
		{ ACT_SIZE },
		SOCK_IN 
	},
	{ FOLLOW, "follow", "let one inchannel follow another (start folower gaplessy after leader stopped)", 2 ,
		{ "leader", "follower" },
		{ ACT_SIZE, ACT_SIZE },
		0
	},
	{ IN_NOFOLLOW, "nofollow", "release the followership on a leader", 1 ,
		{ "leader" },
		{ ACT_SIZE },
		SOCK_IN 
	},
	{ FULLSTAT, "fullstat", "give full status info on all in- and outchannels (several lines)", 0,
		{ },
		{ },
		ACT_NOSUB 
	},
	{ SAY, "say", "just say something (I put [saying] in front); useful when timed somehow", 1,
		{ "the message" },
		{ ACT_STRING },
		0 
	},
	{ IN_WATCH, "watch", "watch inchannel (get messages on events and ongoing playback)", 1,
		{ "channel" },
		{ ACT_SIZE },
		SOCK_IN 
	}, //better nosub?
	{ IN_UNWATCH, "unwatch", "stop watching inchannel (see watch)", 1,
		{ "channel" },
		{ ACT_SIZE },
		SOCK_IN 
	}, //nosub?
	{ IN_LOAD, "load", "load track on inchannel (be prepared for starting it)", 2,
		{ "channel", "track" },
		{ ACT_SIZE, ACT_STRING },
		ACT_NOSUB|SOCK_NOTIFY|SOCK_IN|SOCK_PATH 
	},
	{ IN_INLOAD, "inload", "load track on inchannel (be prepared for starting it)", 3,
		{ "channel", "driver", "track" },
		{ ACT_SIZE, ACT_STRING, ACT_STRING },
		ACT_NOSUB|SOCK_NOTIFY|SOCK_IN|SOCK_PATH 
	},
	// Think about making that one SOCK_NOTIFY...
	{ IN_EJECT, "ineject", "Eject track on input channel (stop playback, release resources).", 1,
		{ "channel" },
		{ ACT_SIZE },
		ACT_NOSUB|SOCK_IN
	},
	{ IN_EJECT, "eject", "Alias for ineject.", 1,
		{ "channel" },
		{ ACT_SIZE },
		ACT_NOSUB|SOCK_IN
	},
	{ IN_SCAN, "scan", "scan input properties", 2,
		{ "channel", "list of properties" },
		{ ACT_SIZE, ACT_STRING },
		ACT_NOSUB|SOCK_NOTIFY|SOCK_IN 
	},
	{ IN_EFFECT_ADD, "effect", "add an effect to an input channel's effect chain", 3,
		{ "channel", "effect name", "position (0 = end of chain)" },
		{ ACT_SIZE, ACT_STRING, ACT_SIZE },
		ACT_NOSUB|SOCK_IN
	},
	{ IN_EFFECT_REMOVE, "effect-remove", "remove a effect from an input channel's effect chain", 2,
		{ "channel", "position (0 = end of chain)" },
		{ ACT_SIZE, ACT_SIZE },
		ACT_NOSUB|SOCK_IN
	},
	{ IN_EFFECT_BYPASS, "effect-bypass", "enable/disable bypassing of a effet (so, disable/enable the effect;-)", 3,
		{ "channel", "position (0 = end of chain)", "bypass value (0/1 for false/true)" },
		{ ACT_SIZE, ACT_SIZE, ACT_INT },
		ACT_NOSUB|SOCK_IN
	},
	{ IN_EFFECT_PARAM, "effect-param", "change input effect parameters", 3,
		{ "channel", "position (0 = end of chain)", "parameter string (format to be decided, maybe depending on effect) TODO: make the parameter string optional to give current effect settings" },
		{ ACT_SIZE, ACT_SIZE, ACT_STRING },
		SOCK_IN
	},
	{ IN_EFFECT_LIST, "effect-list", "show the effect chain of given input channel", 1,
		{ "channel" },
		{ ACT_SIZE},
		SOCK_IN
	},
	{ EFFECT_QUERY, "effect-query", "query list of available audio effects, or information about a specific one", 1,
		{ "name" },
		{ ACT_STRING },
		ACT_NOSUB|SOCK_LOCAL|ACT_OPTARG
	},
	{ IN_EFFECT_HELP, "effect-help", "help info for certain input effect", 2,
		{ "channel", "position (0 = end of chain)" },
		{ ACT_SIZE, ACT_SIZE },
		SOCK_IN
	},
	{ OUT_LOAD, "outload", "load resource on outchannel with specified driver/device (active right away), if driver == default, the resource is ignored, just the internal defaults used", 3,
		{ "channel", "driver", "resource" },
		{ ACT_SIZE, ACT_STRING, ACT_STRING },
		ACT_NOSUB|SOCK_NOTIFY|SOCK_OUT|SOCK_PATH 
	},
	{ IN_LENGTH, "length", "determine exact length of track (stops channel, decodes track till end, seeks back to where it left off)", 1,
		{ "channel" },
		{ ACT_SIZE },
		SOCK_NOTIFY|SOCK_IN|SOCK_ANS 
	},
	{ IN_PREREAD, "preread", "read through a file once (to have it cached by file system/kernel)", 1,
		{ "file" },
		{ ACT_STRING },
		SOCK_LOCAL|SOCK_PATH|ACT_NOSUB 
	},
	//hm, optional argument... socketeer can handle with two action parse calls
	{ ADD_IN, "addin", "add input channel (with optional nick name)", 1,
		{ "nick name" },
		{ ACT_STRING },
		ACT_NOSUB|ACT_OPTARG|SOCK_POST 
	},
	{ REM_IN, "remin", "remove input channel", 1,
		{ "id" },
		{ ACT_SIZE },
		ACT_NOSUB 
	},
	{ ADD_OUT, "addout", "add output channel (with optional nick name)", 1,
		{ "nick name" },
		{ ACT_STRING },
		ACT_NOSUB|ACT_OPTARG|SOCK_POST 
	},
	{ REM_OUT, "remout", "remove output channel", 1,
		{ "id" },
		{ ACT_SIZE },
		ACT_NOSUB 
	},
	{ ID, "id", "print my full identification", 0,
		{ },
		{ },
		ACT_NOSUB|SOCK_LOCAL 
	},
	{ CLOSE, "close", "close current connection", 0,
		{ },
		{ },
		SOCK_POST|SOCK_FINAL
	},
	{ FADEOUT, "fadeout", "dummy to remind you to do custom fading via script actions", 0,
		{ },
		{ },
		ACT_NOSUB|SOCK_LOCAL|ACT_OPTARG 
	},
	{ SLEEP, "sleep", "let me sleep until any client wants something", 0,
		{ },
		{ },
		0 
	},
	{ QUIT, "shutdown", "lay down and die gracefully", 0,
		{ },
		{ },
		SOCK_POST|SOCK_FINAL
	},
	{ HELP, "help", "give some info / usage pattern on a command", 1,
		{ "the command" },
		{ ACT_STRING },
		ACT_OPTARG|SOCK_LOCAL 
	},
	{ SHOWAPI, "showapi", "show the whole API (list help for all commands)", 0,
		{ },
		{ },
		SOCK_LOCAL|ACT_NOSUB 
	},
	{ THREADSTAT, "threadstat", "list spawned threads", 0,
		{ },
		{ },
		ACT_NOSUB 
	},
	{ LS, "ls", "list file/directory", 1,
		{ "dir/file" },
		{ ACT_STRING },
		ACT_NOSUB|SOCK_LOCAL|ACT_OPTARG 
	},
	{ CD, "cd", "change directory (for client actions)", 1,
		{ "dir" },
		{ ACT_STRING },
		ACT_NOSUB|SOCK_LOCAL 
	},
	{ PWD, "pwd", "print current working directory", 0,
		{ },
		{ },
		ACT_NOSUB|SOCK_LOCAL 
	},
	{ SPY, "spy", "spy on client communication", 0,
		{ },
		{ },
		ACT_NOSUB|SOCK_LOCAL|SOCK_COMPANY
	},
	{ UNSPY, "unspy", "stop spying on client communication", 0,
		{ },
		{ },
		ACT_NOSUB|SOCK_LOCAL|SOCK_COMPANY
	},
	{ SHOWID, "showid", "toggle showing of channel id in responses to channel commands", 1,
		{ "1(on) or 0(off)" },
		{ ACT_INT },
		ACT_NOSUB|SOCK_LOCAL 
	},
	{ PEER, "peer", "send a message to another peer (client).", 3,
		{ "your peer name", "peer's peer name", "message" },
		{ ACT_STRING, ACT_STRING, ACT_STRING },
		ACT_NOSUB|SOCK_LOCAL|SOCK_COMPANY
	}, 
	{ ADDPEER, "addpeer", "add a peer entry, registering for messages", 2,
		{ "peer name", "description" },
		{ ACT_STRING, ACT_STRING },
		ACT_NOSUB|SOCK_LOCAL|SOCK_COMPANY
	},
	{ REMPEER, "rempeer", "remove a peer entry", 1,
		{ "peer name" },
		{ ACT_STRING },
		ACT_NOSUB|SOCK_LOCAL|SOCK_COMPANY
	},
	{ SHOWPEERS, "showpeers", "show a list of peers with optional description", 0,
		{ },
		{ },
		ACT_NOSUB|SOCK_LOCAL|SOCK_COMPANY
	},
	{ FEEDBACK, "feedback", "disable/enable waiting for mixer feedback before submitting the next action (applies only to actions that do fine without feedback)", 1,
		{ "value (0/1)" },
		{ ACT_INT },
		ACT_NOSUB|SOCK_LOCAL
	},
	{ NOTHING, "", "", 0, { }, { }, 0 }
};

action* one_time_act(const std::string clientstring, comm_data *cd)
{
	action *act = new action(client_world);
	if(act)
	{
		act->comdat = cd;
		act->disposable = true; // A channel action handle is supposed to delete it.
		if(!act->parse(clientstring))
		{
			if(cd != NULL)
			{
				std::string errmsg;
				act->success(errmsg);
				cd->errors.push(new error("client_api", dmd::err::IO, errmsg));
			}
			delete act;
			return NULL;
		}
	}
	return act;
}

// Some special actions that only the client action handler issues.
static const def in_notify =
{ NOTIFY, "<internal intput notify>", "notification", 0,
	{ },
	{ },
	ACT_CONTAINER|SOCK_IN
};
static const def out_notify =
{ NOTIFY, "<internal output notify>", "notification", 0,
	{ },
	{ },
	ACT_CONTAINER|SOCK_OUT
};
static const def notify = 
{ NOTIFY, "<internal notify>", "notification", 0,
	{ },
	{ },
	ACT_CONTAINER
};


action* make_notify_act(action *base_act)
{
	if(!(base_act->def->flags & api::SOCK_NOTIFY))
	{
		SERROR("[notify_act()] Was asked to create notification for action that doesn't need notification! Messup!");
		return NULL;
	}
	action *notify_act = new action;
		// Prepare notification
	notify_act->def = ((base_act->def->flags & api::SOCK_IN)
			? &api::in_notify
			: ((base_act->def->flags & api::SOCK_OUT)
				? &api::out_notify
				: &api::notify));
	notify_act->set_sub(new action);
	*(notify_act->subaction) = *base_act;
	notify_act->subaction->swr = NULL;
	notify_act->subaction->comdat = NULL;
	notify_act->disposable = true;
	return notify_act;
}


}} // namespace
