/*
	client_action_handler: The code to process client actions, from parsing to sending to mixer, waiting for response... storing messages to be sent back to clients.

	part of DerMixD
	(c)2004-2011 Thomas Orgis, licensed under GPLv2
*/

#include "basics.hxx"

#include "coms/client_action_handler.hxx"
#include "version.hxx"
#include "threads/threads.hxx"
#include "actionlist.hxx"
#include "coms/socket_watch.hxx"
#include "coms/client_api.hxx"
#include "browse.hxx"
#include "tstring.hxx"
#include "audio/effect.hxx"

#include <string.h>
#include <string>
#include <vector>

#include "debug.hxx"
#include "shortcuts.hxx"

using std::vector;
using std::string;

#define ME "client_action_handler"

namespace dmd
{

client_action_handler::client_action_handler(thread &ted, socket_writer &w, socket_watch &sowa):
	 got_final(false), got_error(false)
	,act(api::client_world), actions(sowa.actions),  cancelee(&ted), writer(&w), watch(&sowa)
	,feedback(true)
{
	CONSTRUCT(ME);
	init();
}

client_action_handler::client_action_handler(thread &ted, actionlist &actl):
	 got_final(false), got_error(false)
	,act(api::client_world), actions(&actl), cancelee(&ted), writer(NULL), watch(NULL)
	,feedback(true)
{
	CONSTRUCT(ME);
	init();
}

// Stuff all constructors run.
void client_action_handler::init()
{
	act.swr = writer;
	act.comdat = &cd;
}

client_action_handler::~client_action_handler()
{
	DESTBEGIN(ME);
	// If there are answers left. I assume they were not fetched (cancelled thread?) and thus need reaping to prevent a memory leak.
	FOR_VECTOR(std::string*, answers, s) delete *s;
	DESTEND(ME);
}

void client_action_handler::handle(const string &cmd)
{
	// Actually handle the command...
	handle_command(cmd);

	// ... then always clean up after it.
	ans = "";
	//clear the actions
	act.clear();
	cd.clear();
}

// Here comes internal code for structured action handing.

void client_action_handler::handle_command(const string &cmd)
{
	//parse command, including full expansion of script into subactions
	if(!act.parse(cmd))
	{
		MDEBUG("client_action_handler %p] Unable to parse: %s", this, cmd.c_str());
		answers.push_back(new string);
		act.success(*answers.back()); // Action failed to parse... success=outcome (ye olde style;-)
		return;
	}

	/*
		there are some possibilities...

		let's summarize:
			1. actions with SOCK_LOCAL require no communication with mixer and are handled here
			2. SOCK_POST indicates that there is some postprocessing to do (polish action parameters)
			3. Handled in channel threads now: SOCK_NOTIFY adds an appropriate notify action for mixer
			4. plain normal actions... just give the parsed action to mixer

			post and notify can be combined
	*/

	// First check if this is a valid action in the current setting.
	if(watch == NULL && act.def->flags & api::SOCK_COMPANY)
	{
		give_error("Cannot execute this action without socketeer company (no socket_watch there).");
		return;
	}

	// Path filtering just happens...
	if(act.def->flags & api::SOCK_PATH && !act.strings.empty())
	bro.adjust_path(act.strings.back());

	// Do postprocessing.
	if(act.def->flags & api::SOCK_POST && !postprocess_action())
	{
		give_error("Postprocessing failed (programming error on the server side).");
		return;
	}

	// Action! 
	if(act.def->flags & api::SOCK_LOCAL)
	handle_local_action();
	else
	handle_mixer_action();

	// Lazyness... if there is an answer string hanging around, it is pushed to the client.
	if(ans != "") answers.push_back(new string(ans));

	// The final action has been handled, but the calling socketeer (or whoever) still needs to care about it's own reaping.
	if(act.def->flags & api::SOCK_FINAL) got_final = true;
}

bool client_action_handler::postprocess_action()
{
	switch(act.def->id)
	{
		case api::IN_VOLUME:
			// If no value specified, assume neutral volume.
			if(act.floats.empty()) act.floats.push_back(1.);
			// Now fill up the missing channel.
			if(act.floats.size() < 2) act.floats.push_back(act.floats[0]);
		break;
		case api::IN_SCRIPT: //only the normal one-time script needs postprocessing (adding 1)
			act.longs.push_back(1);
		break;
		case api::ADD_IN:
		case api::ADD_OUT:
			//there is the nickname as optional argument, mixer needs something
			if(act.strings.size() == 0) act.strings.push_back("");
		break;
		// QUIT and CLOSE: socketeer thread will exit after successfuly handling the action (or be cancelled during semaphore waiting on global shutdown).
		case api::CLOSE:
			act.msg("Goodbye... hope you'll come around again for some funky mixing!", ans);
		break;
		case api::QUIT:
			act.msg("This is the end, my friend!", ans);
		break;
		default:
			MERROR("[client_action_handler %p] dunno what to postprocess on action %s (programming error)!", this, act.def->name);
			return false;
	}

	return true;
}

void client_action_handler::handle_mixer_action()
{
	/*
		Push the main action to mixer and wait.

		In case of QUIT and CLOSE commands, the socketeer will be cancelled when waiting.
		Otherwise, cd will contain an error code indicating the success of the action.
		In case of error, there is also (possibly) a textual description
	*/
	// lock, push, post, unlock, wait
	MDEBUG("[client_action_handler %p] pushing action", this);
	if(!feedback && !(act.def->flags & api::SOCK_FINAL) && !(act.def->flags & api::SOCK_NOTIFY))
	{
		// In feedback-less mode, only push a disposable copy of the action and be done with it.
		action* hero = new action(act);
		hero->disposable = true;
		hero->comdat = NULL;
		// The socket writer stays there. Doesn't block following actions.
		MDEBUG("[client_action_handler %p] actually pushing feedback-less copy %p", this, hero);
		actions->push(hero);
		return;
	}

	actions->push(&act);

	// The thread needs to be cancellable here, because we could push the action while the mixer is already shutting down and will not respond -- instead is trying to cancel all socketeers.
	cd.waitress->wait(cancelee);

	MDEBUG("[client_action_handler %p] waiting done, postprocessing...", this);

	// This might vanish
	if(act.def->flags & api::SOCK_FINAL)
	return;

	/*
		processing of return value either of:

		- single action
		(- notify)
	*/

	// That includes errors happening on notification, as that uses use the same comm_data.
	if(act.have_errors())
	{
		// Construct some hopefully meaningful error trace.
		answers.push_back(new std::string);
		act.error_begin(*answers.back());
		cd.errors.stringify(answers);
		answers.push_back(new std::string);
		act.end(*answers.back());
		got_error = true;
		return;
	}

	// Only if all the actions were successful we will proceed to give normal answers.

	if(!cd.retlines.empty())
	{ // Someone prepared answers to give! Pack +begin and -end around and out with it!
		answers.push_back(new string);
		act.begin(*answers.back());
		if(act.def->id == api::THREADSTAT)
		{ // Special: mixer started it... I finish with the list of socket/communication threads.
			cd.retval_ulong += watch->thread_info(cd.retlines);
			answers.push_back(new std::string("thread count: "));
			strprintf(*answers.back(), "%lu", cd.retval_ulong);
		}
		// Insert the verbatim lines ... we have the agreement that no line will start with [ to be mistaken as delimiting line.
		answers.insert(answers.end(), cd.retlines.begin(), cd.retlines.end());
		cd.retlines.clear();
		answers.push_back(new string);
		act.end(*answers.back());
		// SOCK_ANS shouldn't be set for this... else a bogus ans will be created.
		// Make sure we invalidate any answer that might have been prepared (think about CLOSE/QUIT... those might be better fixed, though).
		ans = "";
	}
	else if(ans == "") act.success(ans); // Create basic success msg, unless there is a message already.

	if(act.def->flags & api::SOCK_ANS) //append some specific info?
	{
		ans +=  ": ";
		switch(act.def->id)
		{
			case api::IN_VOLUME:
				strprintf(ans, "%g %g", act.floats[0], act.floats[1]);
			break;
			case api::IN_LENGTH:
				// I should insert spaces before "s" and "kHz" for correctness... but then, it doesn't really matter.
				strprintf(ans, "%0.3fs; %lu samples; %lukHz", cd.retval, cd.retval_ulong, cd.retval_ulong2);
			break;
			// Equalizer settings don't explicitly return anything...
			// they might even ge active only later, after loading a track.
			// We have to assume that the input values are accepted as is.
			case api::IN_EQ:
				strprintf(ans, "%0.3f %0.3f %0.3f", act.floats[0], act.floats[1], act.floats[2]);
			break;
			case api::IN_BASS:
			case api::IN_MID:
			case api::IN_TREBLE:
				strprintf(ans, "%0.3f", act.floats[0]);
			break;
			default:
				strprintf(ans, "%0.3f", cd.retval);
		}
	}
}

void client_action_handler::handle_local_action()
{
	switch(act.def->id)
	{
		case api::ID: act.msg("DerMixD v" + version + "." + subversion, ans); break;
		case api::FADEOUT: give_error("um, ah... I don't do this... sorry. Use scripting instead"); break;
		case api::HELP:
			if(! act.strings.empty())
			{
				act.msg("",ans);
				api::help(api::find_def(act.strings[0], api::client_world),ans);
			}
			else
			{
				act.msg("usage: help command; available commands: ",ans);
				api::list(api::client_world, ans);
			}
		break;
		case api::SHOWAPI:
		{
			answers.push_back(new string);
			act.begin(*answers.back());
			answers.push_back(new string("Full API listing:"));
			answers.push_back(new string(""));
			api::allhelp(api::client_world, answers);
			answers.push_back(new string);
			act.end(*answers.back());
		}
		break;
		case api::EFFECT_QUERY:
		{
			if(act.strings.size() >= 1)
			{
				answers.push_back(new string);
				act.begin(*answers.back());
				if(dmd::effect::query(act.strings[0], answers))
				{
					answers.push_back(new string);
					act.end(*answers.back());
				}
				else
				{
					delete answers.back();
					answers.pop_back();
					give_error("That filter does not seem to be available, or you made an error in query syntax.");
				}
			}
			else
			{
				act.msg("available effects:", ans);
				for(size_t i=1; i<dmd::effect::id::count; ++i)
				{
					ans += " ";
					ans += dmd::effect::id::names[i];
				}
			}
		}
		break;
		case api::LS:
			answers.push_back(new string);
			act.begin(*answers.back());
			bro.browse(act.strings, answers);
			answers.push_back(new string);
			act.end(*answers.back());
		break;
		case api::PWD:
		{
			string workdir;
			if(bro.pwd(workdir)) act.msg(string("dir: ") + workdir, ans);
			else give_error("unable to get directory");
		}
		break;
		case api::CD:
			if(act.strings.empty()) give_error("I need a directory!");
			else
			{
				if(!bro.change_dir(act.strings[0]))
				give_error("cannot change dir");
				else act.success(ans);
			}
		break;
		// Note about actions accessing the watch: We have a safety barrier for that earlier (SOCK_COMPANY flag).
		case api::SPY: watch->bnd.send(writer);       act.success(ans); break;
		case api::UNSPY: watch->bnd.eliminate(writer);  act.success(ans); break;
		case api::SHOWID: act.idmask = act.longs[0] == 0 ? 0 : api::SOCK_IN|api::SOCK_OUT; act.success(ans); break;
		case api::PEER:
			/* [peer from to] */
			act.fullname += " ";
			act.fullname += act.strings[0];
			act.fullname += " ";
			act.fullname += act.strings[1];
			if(watch->pee.message(act))
			act.success(ans);
			else give_error("No such peer registered.");
		break;
		case api::ADDPEER:
			if(watch->pee.add(writer, act.strings[0], act.strings[1]))
			act.success(ans);
			else give_error("Peer name already taken?");
		break;
		case api::REMPEER:
			if(watch->pee.remove(act.strings[0]))
			act.success(ans);
			else give_error("No such peer registered.");
		break;
		case api::SHOWPEERS:
			answers.push_back(new string);
			act.begin(*answers.back());
			watch->pee.list(answers);
			answers.push_back(new string);
			act.end(*answers.back());
		break;
		case api::IN_PREREAD:
		{
			off_t bytes = read_file(act.strings.back());
			if(bytes >= 0)
			{
				act.success(ans);
				strprintf(ans, ": "OFF_P" bytes read.", (off_p)bytes);
			}
			else give_error("Some problem...");
		}
		break;
		case api::FEEDBACK:
			feedback = (bool)act.longs[0];
			act.success(ans);
		break;
		default: give_error("Unknown local action.");
	}
}

void client_action_handler::give_error(const std::string msg)
{
	act.give_error(msg, ans);
	got_error = true;
}


} // namespace
