/*
	channel: generic code for in/output channels

	part of DerMixD
	(c)2004-2011 Thomas Orgis, licensed under GPLv2
	
*/

#include "basics.hxx"

#include "action.hxx"
#include "channel.hxx"

#include <string>
#include <vector>
using std::string;
using std::vector;

#include "debug.hxx"
#define CMDEBUG(s, ...) MDEBUG( "[ch %zu:%s] " s, id, name.c_str(),  __VA_ARGS__)
#define CSDEBUG(s) MDEBUG( "[ch %zu:%s] " s, id, name.c_str())

//#define DEBUG_SCRIPT
//#define DEBUG_MIXER2
//#define DEBUG_RESET

namespace dmd
{

channel::channel(const size_t idv, const std::string namev):
	 thread_container(namev)
	,working(false), offline(false)
	,status(dmd::channel::NODEV), name(namev)
{
	set_id(idv);
}

channel::~channel()
{
	delscript();
}

// There could be other stuff happening, too...
void channel::set_id(const size_t idv)
{
	id = idv;
}

//get all actions falling in upcoming chunk, sorted ascending in time 
bool channel::getactions(unsigned int chunksize, vector<action*> &nowscript)
{
	float now,then;
	now_and_then(chunksize,now,then);
	MXDEBUG("[ch %zu:%s] really looking for action between %g and %g", id, name.c_str(), now, then);
	if(script.empty()) MXDEBUG("[ch %zu:%s] script is empty", id, name.c_str());
	else MXDEBUG("[ch %zu:%s] window: %g to %g", id, name.c_str(), script.front().time, script.back().time);
	//only search when the window is open
	if
	(
		(!script.empty())
		&& (script.front().time < then)
		&& (script.back().time >= now)
	)
	{
		MXDEBUG("[ch %zu:%s] really looking for action", id, name.c_str());
		for(vector<timeaction>::iterator t = script.begin(); t != script.end(); ++t)
		{
			if((*t).time >= now)
			{
				if((*t).time < then)
				{
					MDEBUG("[ch %zu] found action type %u at %g (%u), count %li", id, (*t).act->def->id, (*t).time, (*t).count);
					if((*t).count > 0 && --(*t).count == 0)
					{
						nowscript.push_back((*t).act);
						MDEBUG("[ch %zu] erasing action %p", id, (*t).act);
						script.erase(t);
						--t; //after that ++t should point to the next item... hopefully
					}
					else //make a copy so that action can be executed again
					{
						action* copyact = new action;
						*copyact = *(*t).act;
						nowscript.push_back(copyact);
					}
				}
				else break; //since script is sorted ascending, there should be nothing more to be found since time >= then
			}
		}
	}
	return (!script.empty());
}

bool channel::getpostactions(vector<action*> &nowscript)
{
	MXDEBUG("[ch %s] really looking for postaction", name.c_str());
	for(vector<timeaction>::iterator t = postscript.begin(); t != postscript.end(); ++t)
	{
		if((*t).count > 0 && --(*t).count == 0)
		{
			nowscript.push_back((*t).act);
			postscript.erase(t);
			--t; //after that ++t should point to the next item... hopefully
		}
		else //make a copy so that action can be executed again
		{
			action* copyact = new action;
			*copyact = *(*t).act;
			nowscript.push_back(copyact);
		}
	}
	return (!postscript.empty());
}

//add action to internal script in an ascending sorted way
vector<timeaction>::size_type channel::addaction(const double time, const long count,  action* act)
{
	timeaction ta;
	ta.time = time;
	ta.count = count;
	ta.act = act;
	ta.act->disposable = true; // After final processing, it shall be deleted by the action handler.
	if(ta.count == 0) ta.count = 1;
	CMDEBUG("adding action %p of type %i, timed at %gs, count %li", ta.act, ta.act->def->id, ta.time, ta.count);
	//think about reverse iterator!
	vector<timeaction>::iterator t;
	for(t = script.begin(); t != script.end(); ++t)
	{
		if( (*t).time > time ) break;
	}
	script.insert(t, ta);
	return script.size();
}

vector<timeaction>::size_type channel::addpostaction(const double time, const long count,  action* act)
{
	timeaction ta;
	ta.time = time;
	ta.count = count;
	ta.act = act;
	if(ta.count == 0) ta.count = 1;
	CMDEBUG("adding postaction %p of type %i, timed at %gs, count %li", ta.act, ta.act->def->id, ta.time, ta.count);
	//think about reverse iterator!
	vector<timeaction>::iterator t;
	for(t = postscript.begin(); t != postscript.end(); ++t)
	{
		// smaller times (larger absolute value) are closer to te end
		// time -2s is scheduled before -1s
		if( (*t).time > time ) break;
	}
	postscript.insert(t, ta);
	return postscript.size();
}

void channel::delscript()
{
	for(vector<timeaction>::iterator t = script.begin(); t != script.end(); ++t)
	{
		delete (*t).act;
	}
	script.clear();
	for(vector<timeaction>::iterator t = postscript.begin(); t != postscript.end(); ++t)
	{
		delete (*t).act;
	}
	postscript.clear();
}

/*
	The socket of dead_writer is being closed:
	Set the writer in all script actions that may want to produce output to NULL so that there is no attempt to use the long gone dead_writer in the future.
*/
void channel::script_nirvana(socket_writer* dead_writer)
{
	CMDEBUG("writer %p going to nirvana, sending messages to the same place", dead_writer);

	for(vector<timeaction>::iterator t = script.begin(); t != script.end(); ++t)
	if((*t).act->swr == dead_writer) (*t).act->swr = NULL;

	for(vector<timeaction>::iterator t = postscript.begin(); t != postscript.end(); ++t)
	if((*t).act->swr == dead_writer) (*t).act->swr = NULL;
}


// Channel state names...
// Placed here because channel state is defined together with channels.
// This stuff might become merged into the channel class, though... 

static const char* state_texts[] =
{
	 "dead", "playing", "paused", "nodev", "idle", "stopped"
	,"seek-pause", "seek-play", "loading"
};

const char* channel::state_text()
{
	if(status < 0) return "???";
	else if(status >= sizeof(state_texts)/sizeof(char*))
	{
		MERROR("Some new state code encountered? (%i)", (int)status);
		return "!!!";
	}
	else return state_texts[status];
}

} // namespace
