/*
	browse: support some cd, ls, pwd

	part of DerMixD
	(c)2007-2010 Thomas Orgis, licensed under GPLv2
*/

#include "basics.hxx"

// For browsing.
#include <dirent.h>
#include <sys/stat.h>
// For getcwd and chdir.
#include <fcntl.h>
#include <errno.h>
#include "browse.hxx"
#include "tstring.hxx"

#include <vector>
using std::string;
using std::vector;

#include "debug.hxx"
#include "shortcuts.hxx"

using std::malloc;
using std::realloc;
using std::free;

// Reasonable value may be 100 or 1000
#define PATH_STEP 100

// That sucks, attempt at portability with readdir_r().
// Do I want to think about integer overflow?
// Argh, reading the manpage of pathconf sends a shiver down my spine! How nuts are these people? There's no way to make sure that the programs won't just randomly crash because of this specification madness!
void* dirent_scratch(const char *path)
{
	size_t s = offsetof(struct dirent, d_name);
	// This may totally not work, according to specification.
	long pl = pathconf(path, _PC_NAME_MAX);
	if(pl < 1) return NULL;

	return malloc(s+pl+1);
}

// Internal utilities.

// Could update that to re-use existing storage.
static char *get_the_cwd()
{
	size_t bs = PATH_STEP;
	char *buf = (char*) malloc(bs);
	while((buf != NULL) && getcwd(buf, bs) == NULL)
	{
		char *buf2 = (char*) realloc(buf, bs+=PATH_STEP);
		if(buf2 == NULL){ free(buf); buf = NULL; }
		else MDEBUG("cwd: increased buffer to %zu", bs);

		buf=buf2;
	}
	return buf;
}

static void list_item(const char *i, vector<string*> &ah)
{
	struct stat fst;
	if(!stat(i, &fst))
	{
		ah.push_back(new string);
		if(S_ISDIR(fst.st_mode)) *ah.back() = "D";
		else *ah.back() = "F";

		*ah.back() += " ";
		*ah.back() += i;
	}
	else ah.push_back(new string(string("! Cannot stat ") + i));
}

// Oh, I am so non-portable;-)
static bool is_absolute(string &path)
{
	if(!path.empty())
	{
		return (path[0] == '/');
	}
	return false;
}

namespace dmd
{

// The browser class.

browser::browser(): cwd(NULL)
{
	update_cwd();
}

browser::~browser()
{
	if(cwd != NULL) free(cwd);
}

// Check current working directory and store it.
void browser::update_cwd()
{
	if(cwd != NULL) free(cwd);

	cwd = get_the_cwd();
}

void browser::browse(std::vector<std::string> &what, std::vector<std::string*> &listing)
{
	update_cwd();
	if(what.empty()) what.push_back(".");

	FOR_VECTOR(string, what, i)
	{
		struct stat fst;
		if(!stat(i->c_str(), &fst))
		{
			if(S_ISDIR(fst.st_mode))
			{
				DIR* the_dir = opendir(i->c_str());
				if(the_dir != NULL)
				{
					void *dirscratch = dirent_scratch(i->c_str());
					if(cwd == NULL || dirscratch == NULL){ listing.push_back(new string("! Error: No current working directory / no dirent scratch space.")); continue; }
					if(chdir(i->c_str())!=0){ listing.push_back(new string("! Error: chdir")); continue; }

					struct dirent *entry;
					while((readdir_r(the_dir, (struct dirent*)dirscratch, &entry) == 0) and entry != NULL)
					{
						MDEBUG("dirent %s", entry->d_name);
						list_item(entry->d_name, listing);
					}
					if(chdir(cwd)!=0) listing.push_back(new string("! Error: chdir(cwd)"));

					free(dirscratch);
					closedir(the_dir);
				}
			}
			else list_item(i->c_str(), listing);
		}
		else listing.push_back(new string("cannot stat " + *i));

		if(i+1 != what.end())
		{
			SDEBUG("empty line");
			// an empty line ... eh, what's the idea again? separating entries...
			listing.push_back(new string);
		}
	}
	listing.push_back(new string);
}

// Play safe here: We cannot be totally sure that the current working directory still matches the cached one.
// It's a process/thread wide thing...
bool browser::pwd(string &workdir)
{
	update_cwd();

	if(cwd == NULL) return false;
	else
	{
		workdir = cwd;
		return true;
	}
}

// Just change directory... no need to update cwd as all routines relying on it update it anyway.
bool browser::change_dir(const string location)
{
	return (chdir(location.c_str()) == 0);
}

void browser::adjust_path(string &path)
{
	// First, if it is an ordinary file, it should exist.
	// If it does not exist, I assume it's some URL; something I cannot path-adjust anyway.
	struct stat fst;
	if(!stat(path.c_str(), &fst)) // == exists
	if(!is_absolute(path)) // == relative path
	{	
		// I can read it here, but need to make the path absolute so that mixer process can load the file.
		update_cwd();
		if(cwd != NULL)
		{
			string npath;
			strprintf(npath, "%s/%s", cwd, path.c_str());
			path = npath;
			MDEBUG("adjusted path: %s", path.c_str());
		}
	}
}

bool existing_file(const std::string name, const std::string dir)
{
	std::string path = dir + "/" + name;
	struct stat fst;
	return (!stat(path.c_str(), &fst) && !S_ISDIR(fst.st_mode));
}

void suffix_files(const string dir, std::vector<std::string*> &files, const std::string suffix)
{
	DIR* the_dir = opendir(dir.c_str());
	if(the_dir != NULL)
	{
		void *dirscratch = dirent_scratch(dir.c_str());
		if(dirscratch == NULL) return;

		struct dirent *entry;
		while((readdir_r(the_dir, (struct dirent*)dirscratch, &entry) == 0) and entry != NULL)
		{
			MDEBUG("dirent %s", entry->d_name);
			size_t dlen = strlen(entry->d_name);
			if(existing_file(entry->d_name, dir) && (suffix.length() == 0 || (dlen>suffix.length() && suffix == entry->d_name+dlen-suffix.length())))
			{
				SDEBUG("Looks good, pushing.");
				dlen -= suffix.length();
				entry->d_name[dlen] = 0;
				files.push_back(new std::string(entry->d_name));
			}
		}

		free(dirscratch);
		closedir(the_dir);
	}
}

}

// Just read... no games with seeking and hoping that that caches.
off_t read_file(const string &name)
{
	off_t length = 0;
	char buffer[4096];
	int fd = open(name.c_str(), O_RDONLY);
	if(fd < 0) return -1;

	do
	{
		errno = 0;
		ssize_t got = read(fd, buffer, 4096);
		if(got < 0)
		{
			if(errno != EINTR)
			{
				close(fd);
				return -1; // else: just continue reading
			}
		}
		else
		{
			length += got;
			if(got < 4096) break;
		}

	} while(1);

	close(fd);
	return length;
}
