/*
	scan_power: Scanner that computes the loudness via RMS amplitude, like the normalize tool.

	part of DerMixD
	(c)2008 Thomas Orgis, licensed under GPLv2
*/

#define DMD_NEED_AUDIO_AMP

#include "audio/scan_power.hxx"
#include "tstring.hxx"

#include <cmath>

#include "debug.hxx"

using std::sqrt;
using std::log10;

// Would make sense to change that to something else than linear mean...?
// Perhaps something considering the frequency (variation) of chunks?
// Anyhow, I'm compatible with normalize for now.
double power_window::smooth()
{
	double val = 0;
	for(unsigned int i=0; i<fill; ++i)
	val += power[i];

	val /= fill; // fill > 0!
	fill = 0;
	return val;
}

bool power_window::add()
{
	power[fill] = sum/summed;
	summed = 0;
	sum = 0;
	return (++fill == size);
}

power_scanner::power_scanner(audio_format *af)
{
	CONSTRUCT("power_scanner");
	fmt = af;
	maxpow = 0;
	channels  = fmt->channels;
	chunksize = fmt->rate / CHUNKS_PER_SEC; // Integer division, so may not exactly equal a second.
	win = new power_window[channels];
	if(win == NULL) good = false;
	else good = true;
}

power_scanner::~power_scanner()
{
	DESTRUCT("power_scanner");
}

void power_scanner::scan(mixer_buffer &sb)
{
	if(sb.channels != channels) good=false;

	if(!good) return;

	for(unsigned int c=0; c<channels; ++c)
	for(unsigned int i=c; i<sb.fill*channels; i+=c+1)
	{
		// Use the plain amplitude from 16bit or whatever format... scaling to 1.0 at the end, saving us many divisions.
		double amplitude = (double)audio_amp(sb.ptr[i]);
		// Square of the amplitude is local power.
		// Ideally, I would take the difference between max/min value in window as double amplitude...
		// ...but that may be another power scanner...
		win[c].sum += amplitude*amplitude;
		// Now we could have completed a chunk, add it's mean to the window.
		if(++win[c].summed == chunksize)
		{
			if(win[c].add()) // If one window is full.
			{
				double wpower = win[c].smooth();
				if(wpower>maxpow) maxpow = wpower;
			}
		}
	}
}

void power_scanner::report(std::string &rs)
{
	rs = "power: ";
	if(!good)
	{
		rs = "error: power scan failed";
		return;
	}
	// Finish off unfinished windows.
	for(unsigned int c=0; c<channels; ++c)
	{
		// The last (incomplete) chunk is ignored, as it doesn't have the same weight.
		// So just caring about the completed chunks.
		if(win[c].fill)
		{
			double wpower = win[c].smooth();
			if(wpower>maxpow) maxpow = wpower;
		}
	}
	// Now get the root (volume) and the scale right.
	maxpow = audio_scale_vol(sqrt(maxpow));
	strprintf(rs, "%g level", maxpow);
	if(maxpow>1e-11) strprintf(rs, ", %g dBFS", audio_vol2dbfs(maxpow));
}
