/*
	fifo_buffer: audio data FIFO buffer, as such

	part of DerMixD
	(c)2004-9 Thomas Orgis, licensed under GPLv2
*/

#include "basics.hxx"
#include "audio/fifo_buffer.hxx"
#include "mathhelp.hxx"
#include <cstring>
// Both define abs(), stdlib for integers, math for floats...
// Should that move into mathhelp?
#include <cstdlib>
#include <cmath>

#include "debug.hxx"

using std::memcpy;
using std::abs;

fifo_buffer::fifo_buffer():
zeroscan(false), zero_range(0), zero_level(0)
{
	CONSTRUCT("fifo_buffer");
	reset();
}

fifo_buffer::fifo_buffer(int c, size_t s):
audio_buffer(c, s), zeroscan(false), zero_range(0), zero_level(0), readpos(0), frontskip(0), backskip(0)
{
	CONSTRUCT("fifo_buffer");
	reset();
}

fifo_buffer::~fifo_buffer(){ DESTRUCT("fifo_buffer"); } // Normal buffer destrucion is inherited, everything else is trivial.

void fifo_buffer::reset(void)
{
	got_end = false;
	front_scanned = false;
	back_scanned = false;
	readpos = 0;
	frontskip = 0;
	backskip = 0;
}

void fifo_buffer::advance(size_t &index, const size_t offset)
{
	index = (index + offset) % size;
}

size_t fifo_buffer::advanced(const size_t index, const size_t offset)
{
	return (index + offset) % size;
}

bool fifo_buffer::get_writeblock(audio_io_type* &point, size_t &free_samples)
{
	if(fill >= size) return false; // The > is paranoia.

	// Readpos + fill is the protected area. We write after that.
	// Wrapping limits the continuous block.
	size_t wpos = readpos;
	advance(wpos, fill);
	free_samples = size-fill;
	if(wpos+free_samples > size) free_samples = size-wpos;

	point = ptr + wpos*channels;

	MXDEBUG("[%p] next write to %zu (%zu samples free)", (void*)this, wpos, free_samples);

	return true;
}

void fifo_buffer::written(const size_t samples)
{
	fill += samples;
	MDEBUG("[fifo_buffer %p] written: %zu (%zu/%zu)", this, samples, fill, size);
	if(zeroscan)
	{
		if(!front_scanned && (fill >= zero_range || got_end))
		front_zeroscan(zero_level, zero_range);

		if(got_end && !back_scanned)
		back_zeroscan(zero_level, zero_range);
	}
}

// After each write, written() has to be called -- so all possible zero scanning is done for now when calling available(). 
size_t fifo_buffer::available(void)
{
	if(zeroscan)
	{
		if(!front_scanned) return 0;

		if(back_scanned) return fill;

		if(fill > zero_range) return fill - zero_range;

		return 0; // Sorry, we need everything to ensure backzero scan.
	}
	else return fill;
}

// I assume that target has the correct channel count!
size_t fifo_buffer::extract(audio_io_type *buf, size_t wanted_samples)
{
	size_t gonna_get = min(wanted_samples, available());
	MDEBUG("[fifo_buffer %p] extracting %zu samples of fill %zu", this, wanted_samples, fill);
	size_t ret = gonna_get;

	if(readpos+gonna_get >= size) // if exactly up to the edge or over it
	{
		// first read up to the edge
		size_t gonna_get_now = size-readpos;
		memcpy( (void*) buf,
		        (void*) (const_cast<audio_io_type*> (ptr + channels * readpos)),
		        gonna_get_now * blocksize() );
		buf  += gonna_get_now*channels;
		fill -= gonna_get_now;
		gonna_get      -= gonna_get_now;
		readpos = 0; // We have read up to the end -- wrap!
	}
	// Now simply read from readpos on... there is no wrapping anymore.
	if(gonna_get)
	{
		memcpy( (void*) buf,
		        (void*) (const_cast<audio_io_type*> (ptr + channels * readpos)),
		        gonna_get * blocksize() );
		fill        -= gonna_get;
		readpos += gonna_get;
	}

	return ret;
}

// This is a lot more simple when not needing to copy the data...
size_t fifo_buffer::skip(size_t &wanted_samples)
{
	size_t to_skip = min(wanted_samples, available());
	advance(readpos, to_skip);
	fill -= to_skip;
	return to_skip;
}

void fifo_buffer::discard(void)
{
	fill = 0;
	readpos = 0; // Not really necessary, just tidy.
	got_end = false;
	back_scanned = false;
	backskip = 0;
}



// frontzeroscan() and backzeroscan()
//	Trying to dismiss some data that we consider to be leading/trailing silence by advancing the reader pointer and/or decreasing the fill.

void fifo_buffer::set_zero(bool enable, audio_io_type level, size_t range)
{
	zeroscan = enable;
	zero_level = level;
	// Make sure we have at least one sample more in the buffer than needed for scanning.
	// Second thought: Limit to _half_ of the size, so that we still have sensible performance in reading.
	// Filling up the buffer in single-sample steps because the rest is hold for zero scanning is not very helpful.
	size_t zlimit = size/2;
	if(zlimit == 0)
	{
		zero_range = 0;
		MERROR("[fifo_buffer %p] zeroscan setup with tiny buffer of %zu???", this, size);
	}
	else zero_range = range < zlimit ? range : zlimit;
}

#define sample(pos, ch) (ptr[(pos)*channels+(ch)])

// I simplified the writing... performance not main botherage.
void fifo_buffer::front_zeroscan(const audio_io_type level, size_t range)
{
	MDEBUG("[%p] front_zeroscan with level %"AUDIO_IO_P, (void*)this, level);
	front_scanned = true; // Whatever happens now, we did scan the front.
	frontskip = 0;

	// if we got less, we have less.
	if(fill < range) range = fill;

	for(size_t my_pos = 0; my_pos < range; ++my_pos)
	{
		for(unsigned int ch=0; ch<channels; ++ch)
		{
			audio_io_type val = abs(sample(advanced(readpos, my_pos), ch));
			MXDEBUG("[%p] front: abs(%zu,%u) = %"AUDIO_IO_P" > %"AUDIO_IO_P"?", (void*)this, my_pos, ch, val, level);
			if(val > level)
			{
				// Found non-zero!
				frontskip = my_pos;
				fill -= frontskip;
				advance(readpos, my_pos);
				MDEBUG("[%p] front: skipped %zu samples", (void*)this, frontskip);
				return;
			}
		}
	}

	MDEBUG("[%p] front: no silence skipped", (void*)this);
}


// Think about re-using backskip to avoid multiple scanning of the same data!
void fifo_buffer::back_zeroscan(const audio_io_type level, size_t range)
{
	MDEBUG("[%p] back_zeroscan with level %"AUDIO_IO_P"", this, level);
	back_scanned = true; // Whatever happens now, we did scan the front.
	backskip = 0;

	// if we got less, we have less.
	if(fill < range) range = fill;

	for(size_t backpos = 0; backpos < range; ++backpos)
	{
		for(unsigned int ch=0; ch<channels; ++ch)
		{
			audio_io_type val = abs(sample(advanced(readpos, fill-1-backpos), ch));
			MXDEBUG("[%p] back: abs(%zu,%u) = %"AUDIO_IO_P" > %"AUDIO_IO_P"?", (void*)this, backpos, ch, val, level);
			if(val > level)
			{
				// Found non-zero!
				backskip = backpos;
				fill -= backskip;
				MDEBUG("[%p] back: skipped %zu samples", (void*)this, backskip);
				return;
			}
		}
	}

	MDEBUG("[%p] back: no silence skipped", (void*)this);
}
