#ifndef DMD_H_EFFECTCHAIN
#define DMD_H_EFFECTCHAIN
/*
	effectchain: A chain of audio effects.

	part of DerMixD
	(c)2010 Thomas Orgis, licensed under GPLv2

	This shall host (somewhat) arbitrary effecting/transformation of (incoming) audio data. The sample format is fixed; not sure about channel count. One might add up/downmix effects ... as well as resampling. That means that the input channel audio format depends on the effects.
	Yes, that makes sense exactly because I want to give the option of better resampling algorithms to users. The DerMixD built-in is a least-effort-that-works method.
	Another complication is sample count... effects can generally need some arbitrary number of input samples to produce a certain amount of output samples. The worst case could even mean that the correct amount of samples is not known in advance.
	In general, I need to feed input samples to a effect until it's done, possibly leaving some part of a buffer unconsumed, or hidden in the effect's guts.

	Now, how to design that, properly? The clean and pretty way would be via recursion, the last effect querying output from the previous effect until it produced the desired amount of output samples, this scheme cascading down to the beginning (reading from the input file).
	Is this the best way? What kind of buffering do I need in the worst case?

	Think...

	Also: Inserting some effect in between might influence everything after the new effect... perhaps I should support only adding/removing at the end of the chain for now.
	Otherwise, the chain needs to be reconfigured, and that might be impossible when some downstream effects don't support the changed situation. In that case, the new effect needs to get popped out again, the old situation restored... let's keep it simple for now. Only add/remove at the end.

	Idea for the chain is that the first element is the input file itself, all the rest is optional. The effectchain class wraps all over this.
	Apart from reading, any manipulation of the input file is done elsewhere.

	Reading up on LADSPA ... I get the impression that those plugins are supposed to process predetermined blocks of audio in-line, even truly using the same memory for input and output... one might optimize for that. Though: Will those plugins really work efficiently in the latter mode? Won't they do a copy themselves?
*/


#include "errorchain.hxx"
#include "audio/audio_io.hxx"
#include "audio/effect.hxx"

namespace dmd
{
namespace effect
{

// The chain of effects, at last.
// A thin wrap over a vector of effects, with some management added.
class chain
{
	private:
	origin beginning; // Also part of the vector below.
	std::vector<effect*> effects;

	public:
	error err;

	 chain();
	~chain();
	// Sets a new source, resets effects.
	bool set_source(audio::source *the_source);
	bool receive(mixer_buffer &buf, size_t wanted_samples)
	{
		// There's always at least one effect (the source)!
		return effects.back()->receive(buf, wanted_samples);
	};
	// Reset the whole chain, from the beginning.
	// If one effect fails, the whole operation failed.
	bool reset();
	audio_format *format(){ return &effects.back()->format; }
	// Add a new effect into the chain, at optional position.
	// pos=0 means adding at the end (the origin effect is a fixed given).
	bool insert(const std::string type, size_t pos=0);
	// Remove effect from the chain, end or given position.
	// Heck, this could fail if downstream effects fail to reset!
	bool remove(size_t pos=0);
	void clear(); // Remove all effects but the origin.
	// Further modification of the list.
	// Move a effect from certain position to new position.
	bool move(size_t from, size_t to);
	// Switch bypass state for a effect, explicitly.
	// So, either try to activate or deactivate...
	bool set_bypass(bool val, size_t pos=0);
	// Collect errors from effects, append to chain.
	void collect_errors(errorchain &errors);
	// Change parameters of certain effect.
	// Wondering about handling of interference from other clients.
	// Effect positions in the chain can change; they can get removed.
	// But then, it's not worth bothering about that as long as the same problem exists for mixer channels.
	// This could return bool and reset the effect... check validity of parameters...
	bool set_pars(const std::string pars, size_t pos=0);
	// Give a terse listing of the chain, a line for each effect (except the origin).
	void list(std::vector<std::string*> &lines);
	// Pass the verbose help lines for a specific effect.
	// Again, pos=0 means the last one... you cannot do anything with the origin.
	bool help(std::vector<std::string*> &lines, size_t pos=0);
};

}
}
#endif
