#include "basics.hxx"
#include "audio/effectchain.hxx"
#include "shortcuts.hxx"
#include "mathhelp.hxx"
#include "tstring.hxx"

#include "debug.hxx"

namespace dmd
{
namespace effect
{

chain::chain(): err("effectchain")
{
	CONSTRUCT("effect::chain");
	effects.push_back(&beginning);
}

chain::~chain()
{
	DESTBEGIN("effect::chain");
	clear();
	DESTEND("effect::chain");
}

bool chain::set_source(audio::source *the_source)
{
	beginning.realsource = the_source;
	// If there is no source, reset of effects makes no sense. They can just fail.
	if(the_source != NULL) reset();

	return true; // Never fail... failed reset does not render the chain invalid.
}

bool chain::reset()
{
	err.reset();
	FOR_VECTOR(effect*, effects, fi)
	{
		// We don't fail anymore for effect reset. But still, a warning should be emitted.
		if(!(*fi)->reset())
		err.occur(dmd::err::BAD_STATE, "Cannot reset an effect.");
	}
	return true;
}

void chain::clear()
{
	while(effects.size() > 1)
	{
		remove();
	}
}

bool chain::remove(size_t pos)
{
	if(effects.size() < 2) return true;

	if(pos != 0)
	{
		err.occur(dmd::err::TODO, "Chain removal code not yet working for non-end positions.");
		return false;
	}

	// 0 is for the last entry.
	if(pos == 0) pos = effects.size()-1;

	// A bad given position... not that many effects around.
	// Deletion succeeds nevertheless.
	if(pos > effects.size()) return true;

	// Delete the effect instance itself.
	delete effects[pos];
	// Remove it from the list. Man, this syntax is ugly.
	effects.erase(effects.begin()+pos);

	if(beginning.realsource != NULL) reset();

	return true;
}

bool chain::insert(const std::string type, size_t pos)
{
	if(pos != 0)
	{
		err.occur(err::TODO, "Insertion only at end for now.");
		return false;
	}

	effect* insertee = new_effect(type);
	if(insertee == NULL)
	{
		err.occur(err::NOT_SUPPORTED, std::string("Filter type '") + type + "' does not seem to be available.");
		return false;
	}
	// Only try to reset the effect if there really is an active source.
	// Otherwise, we will have to error out later.
	if(insertee->set_source(effects.back()))
	{
		// Don't care for errors here. Filter will be just bypassed.
		if(beginning.realsource != NULL) insertee->reset();

		effects.push_back(insertee);

		return true;
	}
	else
	{
		delete insertee;
		return err.occur(err::SETUP, "Filter cannot use its source.");
	}
}

bool chain::move(size_t from, size_t to)
{
	err.occur(err::TODO, "filerchain move not implemented");
	return false;
}

bool chain::set_bypass(bool val, size_t pos)
{
	if(pos == 0) pos = effects.size()-1;

	if(pos > effects.size())
	return err.occur(err::INVALID, "Bad effect position.");

	return effects[pos]->set_bypass(val);
}

void chain::collect_errors(errorchain &errors)
{
	FOR_VECTOR(effect*, effects, fi)
	{
		if((*fi)->err.occured()) errors.push(new error((*fi)->err));
	}
	if(err.occured()) errors.push(new error(err));
}

bool chain::set_pars(const std::string pars, size_t pos)
{
	if(pos == 0) pos = effects.size()-1;

	// A bad given position... not that many effects around.
	if(pos > effects.size())
	return err.occur(err::INVALID, "Bad effect position.");

	return effects[pos]->pars.parse(pars);
}

void chain::list(std::vector<std::string*> &lines)
{
	for(size_t pos = 1; pos < effects.size(); ++pos)
	{
		lines.push_back(new std::string);
		strprintf(*(lines.back()), "%s %zu: %s; ", effects[pos]->bypass ? "| " : "|-",  pos, effects[pos]->pars.name.c_str());
		effects[pos]->pars.namestring(*(lines.back()));
	}
}

bool chain::help(std::vector<std::string*> &lines, size_t pos)
{
	if(effects.size() < 2) return true;

	if(pos == 0) pos = effects.size()-1;

	// A bad given position... not that many effects around.
	if(pos > effects.size())
	return err.occur(err::INVALID, "Bad effect position.");

	effects[pos]->pars.helptext(lines);
	return true;
}


}
}
