/*
	timepitch: time stretching effect

	part of DerMixD
	(c)2010 Thomas Orgis, licensed under GPLv2

*/

#include "basics.hxx"

// Remember to clean up the SoundTouch header mess.
#include <soundtouch/SoundTouch.h>
#ifndef FLOAT_SAMPLES
	#error "I need SoundTouch with 32 bit floating point numbers. (This class also needs adaption once the mixer format changes, again.)"
#endif

#include "audio/effect/timepitch.hxx"
#include "mathhelp.hxx"

#include "debug.hxx"

soundtouch::SoundTouch testding;

namespace dmd
{
namespace effect
{

class timepitch::my_private
{
	public:
	soundtouch::SoundTouch handle;
};

// Mapping between effect parameters and SoundTouch configuration.
static const int parmap[] =
{
	 -1 // speed
	,-1 // pitch
	,SETTING_USE_AA_FILTER
	,SETTING_AA_FILTER_LENGTH
	,SETTING_USE_QUICKSEEK
	,SETTING_SEQUENCE_MS
	,SETTING_SEEKWINDOW_MS
	,SETTING_OVERLAP_MS
};

timepitch::timepitch(): effect(id::timepitch), flushed(false)
{
	CONSTRUCT("timepitch");
	parts = new my_private();

	pars.define("tempo", 1., "speed factor (>0, <1 stretches time, >1 accelerates)");
	pars.define("pitch", 1., "pitch factor");
	pars.define("use_aa_filter", 0., "Enable/disable anti-alias filter in pitch transposer (0 = disable).");
	pars.define("aa_filter_length", 0., "Pitch transposer anti-alias filter length (8 .. 128 taps, default = 32)");
	pars.define("use_quickseek", 0., "Enable/disable quick seeking algorithm in tempo changer routine (less CPU load, less quality).");
	pars.define("sequence_ms", 0., "Single processing sequence length in ms.");
	pars.define("seekwindow_ms", 0., "Time-stretch algorithm seeking window length in milliseconds for algorithm that finds the best possible overlapping location.");
	pars.define("overlap_ms", 0., "Overlap of sequences during reconstruction in ms.");

	read_settings();
}

timepitch::~timepitch()
{
	DESTBEGIN("timepitch");
	delete parts;
	DESTEND("timepitch");
}

void timepitch::read_settings()
{
	// Skip tempo and pitch.
	for(size_t i=2; i<sizeof(parmap)/sizeof(int); ++i)
	pars[i].value = (float) parts->handle.getSetting(parmap[i]);
}

bool timepitch::write_settings()
{
	for(size_t i=2; i<sizeof(parmap)/sizeof(int); ++i)
	{
		if(!pars[i].changed) continue;

		pars[i].changed = false;
		if(!parts->handle.setSetting(parmap[i], (int) pars[i].as_longint()))
		{
			pars[i].value = (float) parts->handle.getSetting(parmap[i]);
			return err.occur(dmd::err::BAD_PAR, std::string("Cannot use value for ") + pars[i].name);
		}
	}

	// Ignore tempo and pitch settings that are just too low.
	if(pars[0].changed)
	{
		pars[0].value = max(pars[0].value, (float)0.01);
		parts->handle.setTempo(pars[0].value);
		pars[0].changed = false;
	}
	if(pars[1].changed)
	{
		pars[1].value = max(pars[1].value, (float)0.1);
		parts->handle.setPitch(pars[1].value);
		pars[1].changed = false;
	}

	return true;
}

bool timepitch::do_receive(mixer_buffer &buf, size_t wanted_samples)
{
	// Apply changed settings, if any.
	if(pars.changed)
	{
		if(!write_settings()) return false;

		pars.changed = false;
	}

	// Be careful with the flush: SoundTouch introduces new silence on each flushing... can get endless loop that way. So need to keep track of the flush and avoid a second one.
	while(!flushed && parts->handle.numSamples() < wanted_samples)
	{
		SDEBUG("feeding more");
		feed.fill = 0;
		if(!source->receive(feed, feed.size))
		return err.occur(dmd::err::IO, "Cannot read from source.");

		MDEBUG("got %zu samples to feed", feed.fill);
		parts->handle.putSamples(feed.ptr, feed.fill);
		SDEBUG("fed up");
		if(feed.fill < feed.size)
		{
			SDEBUG("flushing");
			parts->handle.flush(); // Extract the end.
			flushed = true;
		}
	}
	size_t got = (size_t) parts->handle.receiveSamples(buf.free_ptr(), wanted_samples);
	MDEBUG("got %zu samples", got);
	wanted_samples -= got;
	buf.fill += got;

	return true;
}

bool timepitch::do_reset()
{
	flushed = false;
	parts->handle.clear();
	if(format.rate < 1 || format.channels < 1 || format.channels > 2)
	return err.occur(dmd::err::BAD_STATE, "Bad format for this filter; erroring out to avoid the exception thrown at me.");

	parts->handle.setSampleRate(format.rate);
	parts->handle.setChannels(format.channels);
	// Fixing the feeder buffer for now.
	return (feed.reallocate(format.channels, 512) == 512);
}


}
}
