/*
	ladspa: audio effect using LADSPA modules

	part of DerMixD
	(c)2011 Thomas Orgis, licensed under GPLv2

	Look for receive() at the very bottom...
*/

#include "basics.hxx"
#include "param_init.hxx"
#include "audio/effect/ladspa.hxx"
#include "module.hxx"
#include "tstring.hxx"
#include "browse.hxx"

#include <ladspa.h>
#include <dirent.h>
#include <sys/stat.h>

#include "shortcuts.hxx"
#include "mathhelp.hxx"
#include "debug.hxx"

#ifdef DMD_INTEGER
#error "LADSPA plugins only work with floating point samples."
#endif

static const char* default_ladspa_path = "/usr/local/lib/ladspa:/usr/lib/ladspa";

// Not as static variable as GCC would complain if unused.
// There's a place for preprocessor constants after all...
#define me "ladspa"
// suffix of library files
static const char* suffix = ".so";

// A little hackery to have safe storage for the LADSPA library path array.
// Damn, that's a little too much of hackery!
class ladspa_paths
{
	private:
	char *dir_buffer;

	public:
	std::vector<const char*> dirs;

	ladspa_paths(): dir_buffer(NULL){ CONSTRUCT("ladspa_paths"); }
	~ladspa_paths()
	{
		DESTBEGIN("ladspa_paths");
		if(dir_buffer != NULL) free(dir_buffer);

		DESTEND("ladspa_paths");
	}
	// Set path list from parameter or environment variable.
	void set()
	{
		fprintf(stderr, "Setting LADSPA paths...\n");

		const char *src;
		if(param("effect", "ladspa_path") != "") src = param("effect", "ladspa_path").c_str();
		else src = getenv("LADSPA_PATH");

		dirs.clear();
		if(dir_buffer != NULL) free(dir_buffer);

		dir_buffer = NULL;
		if(src == NULL)
		{
			src = default_ladspa_path;
			fprintf(stderr, "LADSPA_PATH not set, using internal default: %s\n", src);
		}

		size_t srclen = strlen(src);
		if(srclen == 0) return; // No directory there...

		dir_buffer = (char*)malloc(srclen+1);
		if(dir_buffer == NULL)
		{
			SERROR("Unable to get memory for dir buffer. LADSPA plugins won't work.");
			return;
		}
		memcpy(dir_buffer, src, srclen);
		dir_buffer[srclen] = 0;
		size_t lastpos = 0;
		for(size_t i=0; i<srclen; ++i)
		{
			if(dir_buffer[i] == ':')
			{
				dir_buffer[i] = 0; // Terminate.
				// Push non-empty directory names.
				if(lastpos < i) dirs.push_back(dir_buffer+lastpos);

				lastpos = i+1; // Next directory starts after the ":".
			}
		}
		// The last piece may be left.
		if(lastpos < srclen) dirs.push_back(dir_buffer+lastpos);

		fprintf(stderr, "LADSPA directories:\n");
		FOR_VECTOR(const char*, dirs, dn)
		fprintf(stderr, "%s\n", *dn);

		fprintf(stderr, "\n");
	}

	// Produce list of library names, without a check for duplicates.
	// The individual libraries are also not checked for contents, that's for further inquiry.
	void list_libs(std::vector<std::string*> &libnames)
	{
		FOR_VECTOR(const char*, dirs, dn)
		dmd::suffix_files(*dn, libnames, suffix);
	}

	// Find a named library, producing a full path.
	// Returns false if no such library is found.
	// Only the first occurence of the library counts!
	bool find_lib(const std::string name, std::string &fullpath)
	{
		// I forbid slashes in library names. No path escapes!
		if(name.find_first_of("/") != std::string::npos) return false;

		FOR_VECTOR(const char*, dirs, dn)
		{
			fullpath = *dn;
			fullpath += "/";
			fullpath += name;
			fullpath += suffix;
			if(dmd::existing_file(fullpath, "")) return true;
		}
		fullpath = "";
		return false;
	}
};

static ladspa_paths paths;

// Compute linear or logarithmic interpolations between two points (lower and upper boundary).
static float intervalue(LADSPA_PortRangeHintDescriptor rhi, float lowpart, float lowval, float highpart, float highval)
{
	if(rhi & LADSPA_HINT_LOGARITHMIC)
	return std::exp(lowpart*std::log(lowval) + highpart*std::log(highval));
	else
	return lowpart*lowval + highpart*highval;
}

namespace dmd
{
namespace effect
{

void ladspa_init()
{
	paths.set();
}

// Separation of functionality between this helper class and the outer ladspa class is not clear.
// Either this one should be just variable storage or all direct operations on LADSPA handles should happen here.
class ladspa::my_private
{
	public:
	LADSPA_Descriptor_Function descr;
	const LADSPA_Descriptor* plug; // The actual plugin.
	// TODO: Multiple instances to work with mono plugins on stereo data.
	LADSPA_Handle handle; // ... and the current instance.
	bool active;
	module dmod;
	// Mapping to port IDs. 
	// First/second channels ...
	std::vector<int> inports;
	std::vector<int> outports;
	// Mapping from parameter list index to port IDs
	std::vector<int> controls; // That's control input ports.
	std::vector<int> outcontrols; // That's control output ports.
	std::vector<float> diag_values; // Just some storage for output control values, might be queried in future.
	// A separate buffer for de-interleaving ... perhaps that could be spared in future.
	// Also, for now using separate input/output buffers ... most likely moving to in-place transformation and eliminating any buffer unless deinterleaving is needed.
	mixer_buffer inbuf, outbuf;
	// And another buffer for reading from the source before interleaving.
	// This HAS TO GO!
	mixer_buffer tmpin;

	my_private(): descr(NULL), plug(NULL), handle(NULL), active(false) { CONSTRUCT("ladspa_private"); };
	~my_private()
	{
		DESTBEGIN("ladspa_private");
		if(handle)
		{
			deactivate();
			plug->cleanup(handle);
		}
		DESTEND("ladspa_private");
	}
	bool instantiate(unsigned long rate)
	{
		cleanup();
		if(!plug) return false;
		handle = plug->instantiate(plug, rate);
		return (handle != NULL);
	}
	void cleanup()
	{
		deactivate();
		if(plug && handle)
		{
			plug->cleanup(handle);
			handle = NULL;
		}
	}
	void activate()
	{
		if(!plug || !handle || active) return;
		// The activate function is allowed to be NULL
		if(plug->activate) plug->activate(handle);

		active = true;
	}
	void deactivate()
	{
		// The deactivate function is allowed to be NULL
		if(plug && active && plug->deactivate) plug->deactivate(handle);

		active=false;
	}
	// Assume that the port counts match. Lazy. Better next time.
	void connect_ports(numparm_group &par)
	{
		if(!plug || !handle) return;

		for(size_t c = 0; c < min(inports.size(), inbuf.channels); ++c)
		plug->connect_port(handle, inports[c], inbuf.chan_ptr(c));

		for(size_t c = 0; c < min(outports.size(), outbuf.channels); ++c)
		plug->connect_port(handle, outports[c], outbuf.chan_ptr(c));

		for(size_t p = 0; p < min(controls.size(), par.parms.size()); ++p)
		plug->connect_port(handle, controls[p], &par.parms[p].value);

		// Also hook up the output control ports ... some storage needs to be provided!
		for(size_t p = 0; p < outcontrols.size(); ++p)
		plug->connect_port(handle, outcontrols[p], &(diag_values[p]));
	}
	// If sampling rate changes, some control values might need rescaling.
	void control_rate_scale(unsigned long rate, numparm_group &pars)
	{
		if(!plug || !handle) return;

		for(size_t i=0; i<controls.size(); ++i)
		{
			const LADSPA_PortRangeHint rh = plug->PortRangeHints[controls[i]];
			if(rh.HintDescriptor & LADSPA_HINT_SAMPLE_RATE)
			pars[i].change_scale(rate);	
		}
	}
	void run(mixer_buffer &buf)
	{
		inbuf.fill = 0;
		outbuf.fill = 0;
		inbuf.deinterleave(tmpin);

		plug->run(handle, inbuf.fill);

		outbuf.fill = inbuf.fill;
		outbuf.interleave(buf);
	}
};

ladspa::ladspa(): effect(id::ladspa), bufsize(512)
{
	CONSTRUCT(me);
	bufsize = param.as_size("buffer");
	parts = new my_private();
}

ladspa::~ladspa()
{
	DESTBEGIN(me);

	delete parts;

	DESTEND(me);
}

void ladspa::list_libs(std::vector<std::string*> &libs)
{
	paths.list_libs(libs);
}

bool ladspa::load_lib(const std::string lib)
{
	std::string libpath;
	if(!paths.find_lib(lib, libpath)) return err.occur(dmd::err::IO, "Cannot find library file.");

	if(!parts->dmod.open(libpath)) return err.take_over(parts->dmod.err);

	parts->descr = (LADSPA_Descriptor_Function) parts->dmod.get_symbol("ladspa_descriptor");
	if(parts->descr == NULL) return err.occur(dmd::err::IO, "Cannot find ladspa descriptor in library.");

	return true;
}

bool ladspa::list_plugins(const std::string lib, std::vector<std::string*> &plugins)
{
	if(!load_lib(lib)) return false;

	const LADSPA_Descriptor* mod;
	unsigned long idx = 0;
	while((mod = parts->descr(idx++)) != NULL)
	{
		// I don't like it that there is no safety for the portdescripor array.
		// But then, I rely on a binary blob anyway.
		int aiports = 0;
		int aoports = 0;
		int ciports = 0;
		int coports = 0;
		for(unsigned long p=0; p<mod->PortCount; ++p)
		{
			LADSPA_PortDescriptor pd = mod->PortDescriptors[p];
			if     (pd & LADSPA_PORT_CONTROL && pd & LADSPA_PORT_INPUT)  ++ciports;
			else if(pd & LADSPA_PORT_CONTROL && pd & LADSPA_PORT_OUTPUT) ++coports;
			else if(pd & LADSPA_PORT_AUDIO   && pd & LADSPA_PORT_INPUT)  ++aiports;
			else if(pd & LADSPA_PORT_AUDIO   && pd & LADSPA_PORT_OUTPUT) ++aoports;
		}
		plugins.push_back(new std::string);
		strprintf(*(plugins.back()), "%s (ain=%i,aout=%i,cin=%i,cout=%i) %s", mod->Label, aiports, aoports, ciports, coports, mod->Name);
	}
	return true;
}

bool ladspa::load(const std::string lib, const std::string plugin, const std::string fullname)
{
	parts->plug = NULL;
	pars.reset(id::id_to_name(type), id::info[type]);
	if(!load_lib(lib)) return false;

	const LADSPA_Descriptor* mod;
	unsigned long idx = 0;
	while((mod = parts->descr(idx++)) != NULL)
	{
		if(plugin == mod->Label)
		{
			parts->plug = mod;
			pars.reset(fullname != "" ? fullname : id::id_to_name(type), std::string(parts->plug->Name));
			get_ports();
			return true;
		}
	}
	return false;
}

void ladspa::get_ports()
{
	// pars is already cleared!
	parts->inports.clear();
	parts->outports.clear();
	parts->controls.clear();
	parts->diag_values.clear();
	if(!parts->plug) return;

	for(unsigned long p=0; p<parts->plug->PortCount; ++p)
	{
		LADSPA_PortDescriptor pd = parts->plug->PortDescriptors[p];
		if(pd & LADSPA_PORT_CONTROL && pd & LADSPA_PORT_INPUT)
		{
			const LADSPA_PortRangeHint rh = parts->plug->PortRangeHints[p];
			LADSPA_PortRangeHintDescriptor rhi = rh.HintDescriptor;
			std::string info = parts->plug->PortNames[p];
			float scale = 1.;

			if(rhi & LADSPA_HINT_SAMPLE_RATE)
			{
				scale = param.as_float("audio_rate"); // Hack for now ... there is no real sample rate yet!
				info += "; sample_rate";
			}
			if(rhi & LADSPA_HINT_TOGGLED) info += "; bool:true>0";
			if(rhi & LADSPA_HINT_INTEGER) info += "; int";
			if(rhi & LADSPA_HINT_LOGARITHMIC) info += "; log";
			// rh->HintDescriptor & LADSPA_ ... settle default value and and range
			// Gah! PortNames are free-form ... can only use them in help output.
			// Let's make up some name...
			std::string parname;
			strprintf(parname, "p%i", p);
			// For each control, one parameter is added ... so the indices match.
			size_t pi = pars.define(parname, 0., info);
			numparm  *par = &pars[pi];
			par->change_scale(scale);

			if(rhi & LADSPA_HINT_BOUNDED_BELOW)
			{
				par->bounded[0] = true;
				par->bounds[0]  = rh.LowerBound;
			}
			if(rhi & LADSPA_HINT_BOUNDED_ABOVE)
			{
				par->bounded[1] = true;
				par->bounds[1]  = rh.UpperBound;
			}
			
			switch(rhi & LADSPA_HINT_DEFAULT_MASK)
			{
				case LADSPA_HINT_DEFAULT_MINIMUM:
					if(par->bounded[0]) par->defval = par->bounds[0];
				break;
				case LADSPA_HINT_DEFAULT_LOW:
					if(par->bounded[0] && par->bounded[1])
					par->defval = intervalue(rhi, 0.75, par->bounds[0], 0.25, par->bounds[1]);
				break;
				case LADSPA_HINT_DEFAULT_MIDDLE:
					if(par->bounded[0] && par->bounded[1])
					par->defval = intervalue(rhi, 0.5, par->bounds[0], 0.5, par->bounds[1]);
				break;
				case LADSPA_HINT_DEFAULT_HIGH:
					if(par->bounded[0] && par->bounded[1])
					par->defval = intervalue(rhi, 0.25, par->bounds[0], 0.75, par->bounds[1]);
				break;
				case LADSPA_HINT_DEFAULT_MAXIMUM:
					if(par->bounded[1]) par->defval = par->bounds[1];
				break;
				case LADSPA_HINT_DEFAULT_0:
					par->defval = 0;
				break;
				case LADSPA_HINT_DEFAULT_1:
					par->defval = 1;
				break;
				case LADSPA_HINT_DEFAULT_100:
					par->defval = 100;
				break;
				case LADSPA_HINT_DEFAULT_440:
					par->defval = 440; // Could make that depend on a parameter for different tuning...
				break;
			}
			par->set(par->defval);

			parts->controls.push_back(p);
		}
		else if(pd & LADSPA_PORT_CONTROL && pd & LADSPA_PORT_OUTPUT)
		{
			parts->outcontrols.push_back(p);
			parts->diag_values.push_back(0);
		}
		else if(pd & LADSPA_PORT_AUDIO && pd & LADSPA_PORT_INPUT)
		{
			parts->inports.push_back(p);
		}
		else if(pd & LADSPA_PORT_AUDIO && pd & LADSPA_PORT_OUTPUT)
		{
			parts->outports.push_back(p);
		}
	}
}

bool ladspa::do_reset()
{
	MDEBUG("[effect::ladspa %p] do_reset", this);
	// If there is no plugin loaded, 
	if(!parts->plug)
	return err.occur(err::BAD_STATE, "No LADSPA plugin loaded.");

	if(!source) return true;

	format.channels = parts->outports.size();
	if(source->format.channels != parts->inports.size())
	return err.occur(err::BAD_STATE, "Input channel cound does not match LADSPA plugin.");

	parts->control_rate_scale(format.rate, pars);

	if(!parts->instantiate(format.rate))
	return err.occur(err::BAD_STATE, "Cannot instantiate plugin.");

	// Fix up the various buffers... better in parts...
	parts->tmpin.reallocate(source->format.channels, bufsize);
	parts->inbuf.reallocate(source->format.channels, bufsize);
	parts->outbuf.reallocate(format.channels, bufsize);
	parts->inbuf.interleaved = false;
	parts->outbuf.interleaved = false;
	if(parts->tmpin.size < bufsize || parts->inbuf.size < bufsize || parts->outbuf.size < bufsize)
	return err.occur(err::NOMEM, "Hah, we really detected OOM?");

	parts->connect_ports(pars);
	// We don't support realtime plugins that are bitch about being activated too early ... or do we?
	parts->activate();
	return true;
}


bool ladspa::do_receive(mixer_buffer &buf, size_t wanted_samples)
{
	parts->tmpin.fill = 0;
	if(source->receive(parts->tmpin, min(bufsize, min(buf.free_space(), wanted_samples))))
	{
		parts->run(buf);
		return true;
	}
	else return false;
}

} // namespace effect
} // namespace dmd
