#ifndef DMD_H_EFFECT
#define DMD_H_EFFECT
/*
	effect: single audio effects
*/

#include "errorchain.hxx"
#include "audio/audio_format.hxx"
#include "audio/audio_buffer.hxx"
#include "audio/audio_io.hxx"
#include "numparam.hxx"

namespace dmd
{
namespace effect
{

namespace id
{
	enum type_id
	{
		invalid = -1,
		#define X(a,b,c) a
		#include "blocks/effect_types.hxx"
		#undef X
		,count
	};
	#include "blocks/enummap_header.hxx"
	extern const char* info[count];
}

// One effect in the chain.
// It pulls data from any preceding effect, as needed.
class effect
{
	protected:
	// Hm, when effects are stored in some STL version of a linked list, 
	// this extra pointer here would be very redundant.
	// But how to avoid it?
	effect *source; // The previous effect in the chain.

	public:
	enum id::type_id type;
	bool bypass; // Switch to copy mode.
	bool user_bypass; // If the user explicitly wants to bypass, even if effect _could_ run.
	// Values and descriptions for extenal retrieval.
	// Values might be used live from here, too.
	numparm_group pars;
	// Effects can transform formats (channels, rate).
	// This is the output format of this effect.
	audio_format format;
	// When something fails, one might check this error structure.
	// Derived classes should reset the "who" part of the error.
	error err;

	effect(id::type_id id_ = id::copy);
	virtual ~effect();

	// Set/change the source. You need to call reset() later if the source really becomes active.
	// Sure that I want to support source == NULL here?
	bool set_source(effect *source_ = NULL);
	// Trigger the effect bypass (copy) state.
	// Switchin bypass on always suceeds (effect::reset() always works), switching off might fail and keep the bypass mode active.
	bool set_bypass(bool val)
	{
		// On each reset, the user bypass mode is tried to achieve,
		// resulting in the actual bypass mode depending on success.
		user_bypass = val;
		if(val){ bypass = true; effect::reset(); }
		else
		{
			if(reset()) bypass = false;
			else effect::reset();
		}
		return (bypass == val);
	}
	// The receive wrapper for bypassing. Only the source overrides that.
	virtual bool receive(mixer_buffer &buf, size_t wanted_samples);
	// Forget any state, re-initialize with possibly changed source format.
	// The default reset() will bypass the effect if the actual do_reset() fails.
	virtual bool reset();
	// Specific effects override that, default is copy.
	// Receive some data from the effect's output, adding to the given buffer.
	// The default implementation just copies data.
	virtual bool do_receive(mixer_buffer &buf, size_t wanted_samples);
	// Effects implement that, please.
	// This default method does just reset the format to that of the source, if there is any.
	// It is also always called by reset() before the overloaded do_reset() is called. So the overloaded method describes what happens after effect::do_reset().
	virtual bool do_reset();
	// Access to parameter values wrapped for possibly needed retrieval / setting operations (library calls).
	// bool setpar(size_t parid, float val);
	// bool getpar(size_t parid, float &val);
	// bool getpars(); // Needed to synchronize parameter values? Dunno.
};

// Factory for effects.
effect* new_effect(const std::string name);

// Report textual info about effects, or one effect (for clients).
// Assumed not to touch answers if it fails, there might be graceful failures within that, resulting in short output but still successful return.
bool query(const std::string name, std::vector<std::string*> &answers);

// The initial piece of every chain.
// It reads from an input file.
class origin: public effect
{
	public:
	// You set that to non-null.
	// ... then call reset().
	// There is no wrapper around this because the effectchain calls the reset in a big loop anyway.
	audio::source *realsource;

	 origin();
	~origin();

	// This just reads from the input file, directly.
	bool receive(mixer_buffer &buf, size_t wanted_samples);

	// If a new file is opened, call reset to get the new format.
	bool reset();
};

}
}
#endif
