/*
	effect: single audio effects
*/

#include "basics.hxx"
#include "audio/effect.hxx"
#include "mathhelp.hxx"
#include <cmath>

#include "comparse.hxx"

#ifdef EFFECT_TIMEPITCH
#include "audio/effect/timepitch.hxx"
#endif
#ifdef EFFECT_LADSPA
#include "audio/effect/ladspa.hxx"
#endif

#include "shortcuts.hxx"
#include "debug.hxx"

namespace dmd
{
namespace effect
{

namespace id
{
	const char* names[count] =
	{
		#define X(a,b,c) b
		#include "blocks/effect_types.hxx"
		#undef X
	};
	const char* invalid_name = "invalid";

	const char* info[count] =
	{
		#define X(a,b,c) c
		#include "blocks/effect_types.hxx"
		#undef X
	};
	#include "blocks/enummap.hxx"
}

// Generic effect framework.

effect::effect(id::type_id id_):
	 source(NULL)
	,type(id_), bypass(false), user_bypass(false)
	,pars(id::id_to_name(id_), id::info[id_])
	,err(std::string("effect ") + id::id_to_name(id_))
{
	CONSTRUCT("effect");
	reset();
}

effect::~effect()
{
	DESTRUCT("effect");
}

bool effect::set_source(effect* source_)
{
	source = source_;
	return true;
}

bool effect::receive(mixer_buffer &buf, size_t wanted_samples)
{
	MXDEBUG("[effect %p] receive, bypass=%i", this, bypass);
	if(bypass) return effect::do_receive(buf, wanted_samples);
	else return do_receive(buf, wanted_samples);
}

bool effect::do_receive(mixer_buffer &buf, size_t wanted_samples)
{
	MXDEBUG("[effect %p] do_receive for %zu samples", this, wanted_samples);
	if(source != NULL)
	return source->receive(buf, wanted_samples);
	else
	{
		err.occur(dmd::err::BAD_STATE, "No source to read from.");
		return false;
	}
}

bool effect::do_reset()
{
	MDEBUG("[effect %p] do_reset", this);
	err.reset();
	if(source != NULL)
	{
		format = source->format;
	}
	return true;
}

bool effect::reset()
{
	bypass = false;
	effect::do_reset(); // Always do this here so that overloaded do_reset() don't have to call this.
	if(user_bypass)
	{
		MDEBUG("[effect %p] reset with user bypass", this);
		bypass=true;
		return true;
	}
	else if(do_reset())
	{
		MDEBUG("[effect %p] reset successful", this);
		bypass = false;
		return true;
	}
	else
	{
		MDEBUG("[effect %p] reset failed, bypassing", this);
		bypass = true;
		// Repeat that here, to make sure.
		effect::do_reset();
		return false;
	}
}

// The origin special effect.

origin::origin(): effect(id::origin), realsource(NULL)
{
	CONSTRUCT("origin");
}

origin::~origin()
{
	DESTRUCT("origin");
}

bool origin::receive(mixer_buffer &buf, size_t wanted_samples)
{
	if(realsource != NULL)
	{
		if(realsource->mixbuf_read(buf, wanted_samples)) return true;
		else return err.take_over(realsource->err);
	}
	else
	{
		err.occur(dmd::err::BAD_STATE, "No real source to read from.");
		return false;
	}
}

bool origin::reset()
{
	err.reset();
	if(realsource == NULL) return false;

	format = realsource->format;
	return true;
}

// Some home-made small effects.
class amplify: public effect
{
	public:
	amplify(): effect(id::amplify){ pars.define("volume", 1., "linear volume factor"); };
	~amplify(){};
	bool do_receive(mixer_buffer &buf, size_t wanted_samples)
	{
		if(effect::do_receive(buf, wanted_samples))
		{
			float amp = pars[0].value;
			for(size_t i=0; i<buf.fill*buf.channels; ++i)
			buf.ptr[i] *= amp;

			return true;
		}
		else return false;
	};
};

class panamp: public effect
{
	public:
	panamp(): effect(id::panamp), upmix(false), upmixbuf(1,0)
	{
		pars.define("position", 0., "left/right position, -1 ... 0 ... +1");
		pars.define("volume", 1., "overall volume factor");
	}
	~panamp(){}
	bool do_receive(mixer_buffer &buf, size_t wanted_samples)
	{
		if(upmix)
		{
			if(upmixbuf.size < wanted_samples) upmixbuf.reallocate(wanted_samples);

			upmixbuf.fill = 0;
			if(effect::do_receive(upmixbuf, wanted_samples))
			{
				float amp[2];
				panvol(amp);
				for(size_t i=0; i<upmixbuf.fill; ++i)
				{
					buf(buf.fill+i, 0) = upmixbuf[i]*amp[0];
					buf(buf.fill+i, 1) = upmixbuf[i]*amp[1];
				}
				buf.fill += upmixbuf.fill;
				return true;
			}
			else return false;
		}
		else
		{
			if(effect::do_receive(buf, wanted_samples))
			{
				float amp[2];
				panvol(amp);
				for(size_t i=0; i<buf.fill; ++i)
				{
					buf(buf.fill+i, 0) *= amp[0];
					buf(buf.fill+i, 1) *= amp[1];
				}
				return true;
			}
			else return false;
		}
	}
	bool do_reset()
	{
		upmix = (format.channels == 1);
		if(!upmix) upmixbuf.reallocate(0);

		// Panning always creates stereo format.
		format.channels = 2;
		return true;
	}
	private:
	bool upmix;
	mixer_buffer upmixbuf;
	void panvol(float amp[2])
	{
		if(pars[0].value >= 0.)
		{
			amp[0] = 1 - pars[0].value;
			amp[1] = 1.;
		}
		else
		{
			amp[0] = 1.;
			amp[1] = 1 + pars[0].value;
		}
		amp[0] *= pars[1].value;
		amp[1] *= pars[1].value;
fprintf(stderr, "amp=%g %g\n", amp[0], amp[1]);
	}
};

// A fun effect, just to show some effect.
// This should be a prototype that convertes mono signal to stereo, too.
class autopan: public effect
{
	private:
	double alpha;

	public:
	autopan(): effect(id::autopan) { pars.define("frequency", 0.5, "pan sweeps per second"); }
	~autopan(){}
	bool do_receive(mixer_buffer &buf, size_t wanted_samples)
	{
		if(effect::do_receive(buf, wanted_samples))
		{
			if(buf.fill > 0) // assumed to be < format.rate ...
			{
				float amp[2];
				float ampstep[2];

				float timescale = pars[0].value/format.rate;

				float phase = alpha - floor(alpha);
				float ampcurve[2];
				ampcurve[0] = std::sin(alpha*2*pi);
				ampcurve[0] *= ampcurve[0];
				alpha += (float)buf.fill*timescale;
				if(alpha > 1.) alpha -= 1.;
				ampcurve[1] = std::sin(alpha*2*pi);
				ampcurve[1] *= ampcurve[1];

				// What kind of jumps do I have left?
				if(phase < 0.25)
				{
					// first channel volume fades in, second channel is at max.
					amp[0] = ampcurve[0];
					ampstep[0] = (ampcurve[1]-ampcurve[0])/buf.fill;
					amp[1] = 1.;
					ampstep[1] = 0.;
				}
				else if(phase < 0.75)
				{
					// second channel fades out and in
					amp[0] = 1.;
					ampstep[0] = 0.;
					amp[1] = ampcurve[0];
					ampstep[1] = (ampcurve[1]-ampcurve[0])/buf.fill;
				}
				else
				{
					// first channel fades out.
					amp[0] = ampcurve[0];
					ampstep[0] = (ampcurve[1]-ampcurve[0])/buf.fill;
					amp[1] = 1.;
					ampstep[1] = 0.;
				}

				for(size_t i=0; i<buf.fill; ++i)
				{
					buf[format.channels*i]  *= amp[0]+i*ampstep[0];
					if(format.channels ==2) buf[format.channels*i+1] *= amp[1]+i*ampstep[1];
				}
			}
			return true;
		}
		else return false;
	}
	bool do_reset()
	{
		alpha = 0.;
		return true;
	}
};

// At last, the factory (for any effect that is not the origin).

effect* new_effect(const std::string name)
{
	std::vector<std::string> tok;
	split(name, tok, ":");
	enum id::type_id nid = id::name_to_id(tok[0]);

	if(nid == id::invalid) return NULL;

	switch(nid)
	{
		case id::copy:
			return new effect();
		case id::amplify:
			return new amplify();
		case id::panamp:
			return new panamp();
		case id::autopan:
			return new autopan();
#ifdef EFFECT_LADSPA
		case id::ladspa:
		{
			// Full name is ladspa:lib:plugin
			if(tok.size() != 3)
			{
				MERROR("Badly formatted LADSPA effect name: %s", name.c_str());
				return NULL;
			}

			ladspa *lad = new ladspa();

			if(lad->load(tok[1], tok[2], name)) return lad;
			else
			{
				std::string *msg = lad->err.as_string();
				MERROR("Cannot load %s (%s):", name.c_str(), msg->c_str());
				delete msg;
				delete lad;
				return NULL;
			}
		}
#endif
#ifdef EFFECT_TIMEPITCH
		case id::timepitch:
			return new timepitch();
#endif
		default:
			return NULL;
	}
}

bool query(const std::string name, std::vector<std::string*> &answers)
{
	std::vector<std::string> tok;
	split(name, tok, ":");
	enum id::type_id nid = id::name_to_id(tok[0]);

	if(nid == id::invalid) return false;

#ifdef EFFECT_LADSPA
	// Special treatment for LADSPA if not fully specified.
	// Everything that does not specify a library and plugin name is a query for a list of libraries / plugins.
	if(nid == id::ladspa && tok.size() < 3)
	{
		ladspa *lad = new ladspa();
		if(name == "ladspa::")
		{
			// List all individual plugins in all libraries.
			std::vector<std::string*> libs;
			lad->list_libs(libs);

			FOR_VECTOR(std::string*, libs, lib)
			{
				std::vector<std::string*> plugins;
				if(lad->list_plugins(**lib, plugins))
				FOR_VECTOR(std::string*, plugins, plug)
				{
					answers.push_back(new std::string(**lib + ":" + **plug));
					delete *plug;
				}
				delete *lib;
			}
		}
		else if(name == "ladspa" || name == "ladspa:")
		{
			// Just show the libraries.
			lad->list_libs(answers);
		}
		else if(tok.size() == 2) // "ladspa:lib" ... the only possiblity left.
		{
			std::vector<std::string*> plugins;
			if(lad->list_plugins(tok[1], plugins))
			FOR_VECTOR(std::string*, plugins, plug)
			{
				answers.push_back(new std::string(tok[1] + ":" + **plug));
				delete *plug;
			}
		}
		delete lad;
		return true; // You'll see the results any way.
	}
	else
#endif
	{
		effect *phil = new_effect(name);
		if(phil == NULL) return false;
		else
		{
			phil->pars.helptext(answers);
			delete phil;
			return true;
		}
	}
}

}
}
