#ifndef DMD_H_AUDIO_IO
#define DMD_H_AUDIO_IO
/*
	audio_io: most generic properties if input/output audio thingies

	part of DerMixD
	(c)2010-2013 Thomas Orgis, licensed under GPLv2

	Those abstractions shall help divide and test code. But they also help to make me wonder how freaked-out a class hierarchy has to be.

*/

#include "audio/audio_format.hxx"
#include "audio/audio_buffer.hxx"
#include "errorchain.hxx"

#include <string>

// Putting this in the "audio" namespace with the other audio format stuff.
// Consider adding "dmd" namespace on top.
namespace audio
{

// A class to avoid duplicating a single member?
// Is this worth the possibility to, at some point, treat sources and sinks in the same context?
class iothing
{
	public:
	// Every audio input/output thing has a format (be it its input or output format).
	// The sample encoding is part of that, and will actually be used in future.
	audio_format format;
	off_t position; // The sample offset the IO thing is at (buf_read / buf_write is supposed to increment that).
	// Errors can occur.
	error err;
	iothing(const std::string name="iothing"): format(0,0,audio::audio_io_code), position(0), err(name) {};
	virtual ~iothing(){};
	// Sample encoding conversion. Possibly destructive on the source buffer.
	// This appends to the output buffer; not having enough space in there is an error.
	// Convert from I/O format to mixer.
	void append_to_mixer(audio_buffer &in, mixer_buffer &out);
	// From mixer to I/O.
	void convert_to_io(mixer_buffer &in, audio_buffer &out);
};

// A source adds a method for reading data into a mixer buffer.
class source: public iothing
{
	private:
	audio_buffer iobuffer; // Buffer with raw input samples for mixbuf_read.

	public:
	source(const std::string name="input"): iothing(name) {};
	virtual ~source(){};
	// Read I/O data into plain storage.
	bool plain_read(audio_io_type* point, size_t wanted_samples, size_t &got_samples);
	// Read I/O data into buffer.
	// Returns true unless an error occured. Just getting less samples is not an error, but indicates end of file.
	bool buf_read(audio_buffer &abuf, size_t wanted_samples);
	bool buf_read(audio_buffer &abuf){ return buf_read(abuf, abuf.size); };
	// Wrappers to ease reading to mixer buffers.
	bool mixbuf_read(mixer_buffer &abuf){ return mixbuf_read(abuf, abuf.size); };
	bool mixbuf_read(mixer_buffer &abuf, size_t wanted_samples);
	// The method the derived class shall implement:
	// Reading of raw data. We assume that we always get full audio frames.
	// Partial frames are ignored.
	// This can return false to indicate failure, but is expected to have set
	// got_bytes to match the data decoded up to reaching the error condition
	// (this data is still used). The source should be prepared for repeated
	// attempts to read regardless of return value.
	// Management stuff like incrementing position is done in the public read
	// methods!
	private:
	virtual bool do_read(void* buf, size_t wanted_bytes, size_t &got_bytes) = 0;
};

// A sink adds a method to write out data.
// 
class sink: public iothing
{
	public:
	sink(const std::string name="output"): iothing(name) {};
	virtual ~sink(){};
	// Write I/O data from buffer (amount indicated by buffer fill).
	bool buf_write(audio_buffer &abuf);
	// Write that amount of raw audio data.
	virtual bool do_write(const void* buf, size_t bytes) = 0;
};

}

#endif
