#ifndef DMD_H_AUDIO_FORMAT
#define DMD_H_AUDIO_FORMAT

/*
	audio_format: audio data format
	
	part of DerMixD
	(c)2004-2010 Thomas Orgis, licensed under GPLv2
*/

//#include <math.h>
#include <limits.h>
#include <iostream>
#include <sys/types.h>

/*
	Idea about audio formats: Internally, it's float or 32 bit integer, externally, it might be different.
	I am going to convert to float right in the input files, convert to the correct output format in the outputs.
	What I'm wondering about is the case of 16 bit in and out... preserve the scaling to 32767 to avoid a division on dermixd entry?
	But this is rather moot, as I will get floating point values from libmpg123 anyway, and integer mixing will be different. So, let's use float all around, and handle other input/output at the ends where it occurs.
*/
#ifdef DMD_INTEGER
#error "I/O of integer data disabled for now. Needs some kind of fixed-point math."
// Idea: Simple math that preserves 16 bit input/output values without transformation, just headroom for mixing...
typedef int32_t audio_type;
#define audio_code signed_32
#define AUDIO_SCALE 32767
#define AUDIO_P "i" // printf format
#else
typedef float audio_type;
#define audio_code float_32
#define AUDIO_SCALE 1 // The amplitude that corresponds to "full scale", might be something like 32767.
#define AUDIO_P "f" // printf format
#endif

// Hack for now, to get the framework up and to the flexible sample format handling later.
typedef int16_t audio_io_type;
#define audio_io_code signed_16
#define AUDIO_IO_P "i"

namespace audio
{

// The _i is for interleaved storage.
// Default for DerMixD's internals is serial channel storage.
enum code
{
	 signed_16=0
	,signed_16_i
	,signed_32
	,signed_32_i
	,float_32
	,float_32_i
	,codecount // Insert new encodings above that.
};

// Initialize soft clipping with that width.
// Without that, hard clipping occurs.
void clip_init(float width);

// Determine how many bytes a given number of mono samples needs in the given encoding.
size_t samples_to_bytes(enum code enc, size_t samples);
size_t bytes_to_samples(enum code enc, size_t bytes);

// Convert a floating point number representing a fraction of full scale to a corresponding sample value.
// So, [-1:1] -> [sample_min:sample_max], with hard clipping.
void scale_to_sample(float scale, enum code enc, void *sample);

// Encode from given encoding to the internal sample representation.
// You have preallocated the memory... samples gets set to the absolute sample count we got (using the given channel count, chopping off any leftover single sample).
void encode_from(enum code enc, const void* in, size_t bytes, audio_type *out, unsigned int channels, size_t &samples);
// Encode from the mixer format to an external encoding.
// Again: You allocated the memory for the maximum possible size, bytes contains the actual number of bytes written.
// Note: This modifies both buffers, consider the input data rubbish after the call.
// I need to investigate if this does any good... but in my application I don't need the input buffer values afterwards, so it doesn't obviously do harm, neither.
void encode_to(enum code enc, audio_type *in, unsigned int channels, size_t samples, void *out, size_t &bytes);

}

/*
	common struct for definition of audio formats	(sampling rate and number of channels)
	In theory one could support other channel counts than 2	but existing code relies on channels being 2 at some places! (Really?, still?)
*/
class audio_format
{
	public:
	enum audio::code encoding;
	unsigned long int rate;
	unsigned int channels;
	audio_format(unsigned long r=0, unsigned int ch=0, enum audio::code enc=audio::audio_code): encoding(enc), rate(r), channels(ch) { };
	~audio_format(){};
	size_t framesize(){ return audio::samples_to_bytes(encoding, channels); };
	// calculate seconds from samples
	double seconds(off_t pos){ return (double)pos/(double)rate; };
	// calculate samples from seconds
	off_t samples(const double sec){ return (off_t) (sec * rate);};
	// calculate bytes from samples
	off_t samples2bytes(off_t samples){ return samples*framesize(); };
	off_t bytes2samples(off_t bytes){ return bytes/framesize(); };
};

#define audio_vol(sample)    ((double)(sample)/AUDIO_SCALE)
#define audio_scale_vol(dsample)    ((dsample)/AUDIO_SCALE)
// Stupid guard to avoid gcc warning of non-used function.
// Well, if there was proper inlining across object files...
#ifdef DMD_NEED_AUDIO_AMP
// Again: I am interested in the concept of amplitude, which _can_ be the same as abs().
// The plain amplitude is un-normalized volume.
static audio_type audio_amp(audio_type sample)
{
#ifdef DMD_INTEGER
#error "Amplitude is limited for signed integers ... extra care for negative values... check that!"
	if(sample < -AUDIO_SCALE) return AUDIO_SCALE;
	else return sample < 0 ? -sample : sample;
#else
	return sample < 0 ? -sample : sample;
#endif
}
#endif
#define audio_vol2dbfs(vol)  (log10(vol)*20)
#define audio_dbfs2vol(dbfs) pow(10, dbfs/20)

#endif
