#ifndef DMD_H_AUDIO_FIFO
#define DMD_H_AUDIO_FIFO

/*
	audio_fifo: audio data FIFO buffer on a file

	part of DerMixD
	(c)2004-9 Thomas Orgis, licensed under GPLv2

	This one handles in a not very appealingly nicely coded but working way the (re)filling of an audio buffer from a file and copying from this buffer including wrapping to qualify as a FIFO buffer.

	This used to be code that handles multiple tracks coming through one pipe,
	now it is simplified to read from an input_file, which _may_ wrap around a pipe in future... but it is not on the plan now.

	Actually the idea is now not to let this one look as an audio buffer but as an input_file instead.
	That way we can nicely switch between buffered and unbuffered I/O.
	Open is the question on how to decide which input_file is going to be used. I need the switcher somewhere...
	I removed audio_buffer as parent class because we do not actually want to use this as a random-access buffer. It is about the interface, and that's a different one.

	TODO: sanitize handling of the wrapped input file. We need to wrap construction and destruction.
	Also error messages should be combined.
	Seek need to include zeroscan... with less work than brute force full scanning.
*/

#include <string>
#include <vector>
#include "audio/audio_buffer.hxx"
#include "threads/threads.hxx"
#include "mutex.hxx"
#include "threads/semaphore.hxx"
#include "audio/fifo_buffer.hxx"
#include "in/input_file.hxx"

namespace dmd {

// Yeah, some multiple inheritance fun!
// I know that this might not be the best way, but the thread stuff is defined defensively enough that it doesn't conflict and it is convenient.
class audio_fifo: public thread_container, public thread, public input_file
{
private:
	// Floating point values that are applied with the format (sample rate) of the actually open file.
	float zero_level;
	double zero_range, size_seconds;
	bool sem_there, reader_there;
	bool locked;
	off_t source_length;
	// Wait until want_samples are available.
	bool wait_for_it(size_t want_samples);
	bool wait_for_end(void);

	// Everything needs to be public so that fifo thread can access it.
	// Also keep in mind that any access to members is mutex-protected.
public:
	// We use the smart buffer (it can do zeroscan and such).
	fifo_buffer buffer;
	// There is a real input file below.
	input_file * source;

	semaphore rsem; // for activating reader thread
	semaphore dsem; // for indicating that there has addition of blocks of data
	real_mutex lock; // for ensuring some atomicity, apart from communication
	bool readstate; // false if some reading error occured
	bool got_end; //true when we reached a track end in the file
	bool goodbye; // For gracefully ending reader thread.

	// You need to provide a source input file with set_source() before opening / reading.
	// Also, the channel count is not given as parameter anymore, the internal default is corrected on loading a file anyway.
	// When re-using this buffer for general preprocessing, we might want to fix it to the mixer format and filter input stuff on the fly, while reading into the buffer.
	audio_fifo(int priority = 0);
	~audio_fifo();

	// audio_file API
	bool do_read(void* buf, size_t wanted_bytes, size_t &got_bytes);
	bool do_open(const std::string location);
	void do_close(void);
	bool do_seek(off_t pos);
	off_t do_length(void);
	bool do_eq(const std::vector<float> &val);

	// skip a number of samples, return difference of actually skipped to requested
	size_t skip(const size_t skip_samples);
	//omit zeroes at beginning (via advancement of position)
	void fresh(); // Empty buffer andreinit zeroscan stuff (new track coming).
	size_t get_zeroskip(); // get the skipped samples at front
	//resize internal buffer, with and without changing channel count
	size_t resize(size_t s);
	size_t resize(unsigned int c, size_t s);
	//give status on cerr... may vanish in future when I really believe in the code
	void status();
	// Set the actual souce file... you need to do that before opening any resource.
	// Actually, when deleting the source, call first set_source(NULL) to ensure that the FIFO thread is not acting any more.
	void set_source(input_file *the_source);

	// The actual thread work... like the name says.
	void thread_work();

	// Start and stop of reading thread.
	// Might be useful on the outside...
	void start(void);
	void stop(void);
};

}
#endif
