/*
	audio_fifo: audio data FIFO buffer, on a simple input_file (no pipe anymore)

	part of DerMixD
	(c)2004-2013 Thomas Orgis, licensed under GPLv2

	This code is only tested with two channels... but it should work with any channel count.
	Only basic restriction is that input channels == output channels; No fancy operations here!

	This class now features the background reading thread that makes such a FIFO really work as a buffer.
	Initially designed for having some space to do the zeroscan in, this thing now actually makes sense in the parallel world of DerMixD and hopefully helps preventing any unnecessary skip even durnig heavy system/disk load as long as the basic priorities (nice levels) are set accordingly.

	Currently, this is a locking FIFO, which is not really necessary. I could remove the lock... or not, because of the zeroscan business?

	A twist about zero scanning at the end:
	At every moment of handing data to the outside, we need to ensure either one of these states:
	1. We have the end in the buffer already.
	2. We have at least zero_range samples left in the buffer.
	This is important to ensure that the actual zeroscanned end does not slip through.

	Oh, and... do not forget seeking. Seeks near the end should take the end zero_range into account.

	I have to nail one thought here: This file is UGLY. Sorry, Thomas, it is. Fix that on some cosy winter evening...

	On error handling:
	The underlying input_file can return fals on do_read(). Eventually, this
	error has to be passed on, but as long as there is data in the buffer from
	earlier read attempts, the error should be postponed. One way to achieve
	that is ignoring read errors until there really is demaned more data than
	already decoded.
*/

#include "basics.hxx"

#include <sched.h>
#include <string>
#include <vector>

#include "audio/audio_fifo.hxx"
#include "action.hxx"
#include "audio/fifo_buffer.hxx"
#include "mathhelp.hxx"
#include "param_init.hxx"
#include "errorcodes.hxx"
#include "errorchain.hxx"

#include "debug.hxx"

using namespace std;

// Hack: separate fifo errors from source errors.
const unsigned char erroffset = 100;

// Read chunks of that maximum size in bytes.
// This limit is there to help other threads to get some scheduling.
// Experimental...?
static const size_t readblock = 16384;

namespace dmd {

// The fifo thread is never cancelable... it only leaves on request.
void audio_fifo::thread_work()
{
	while(1)
	{
		// New scheme: Always be ready for operation. Only stopping when buffer full.
		// Consumer pokes rsem when reading data from buffer. The fifo_reader will then chip in to refill.
		// Every refill operation that yields some data or EOF pokes dsem to inform the consumer.
		// ...also on every failure.
		// You can change the input_file in between, even.
		// Every operation on fifo data is protected by the big lock.
		// Also: No state is assumed between lock sessions, so you can switch tracks in between or even change the audio file.

		// When there is no work to do, the reader waits on rsem. This is needed to prevent busy waiting, which would be nasty.
		// It is important that the consumer pokes rsem on each operation that may change the situation of the buffer (like, opening a file, or, more usually, just extracting some bits from the buffer).
		MXDEBUG("[%p] fifo_reader waiting", (void*)this);
		rsem.wait();
		MXDEBUG("[%p] fifo_reader summoned", (void*)this);

		// How valid is this, still?
		// trying to do it in smaller chunks so that we have the possibility to switch threads in between
		// seems to work for extremely big buffers on my laptop, but not when doing the big fill while crossfading (2 channels playing)
		// this event seems able to trigger a situation only resolved by pausing and letting the buffer fill before restarting playback
		// how is this on other boxen (with NPTL?)

		lock.lock();
		while(buffer.expect_more() && readstate && !goodbye)
		{
			// We have a fill, we want to read up to one block at a time.
			// After being done with one block, the consumer thread has time to chip in and get some.
			size_t samplecount;
			audio_io_type * samplepoint;
			// Get current write position and free space, work on it if there is some free space.
			// Note: One could increase concurrency by reading into a local buffer, later copying to the real one.
			//       This would let the consumer extract data while the reader is busy decoding.
			if(buffer.get_writeblock(samplepoint, samplecount))
			{
				if(format.samples2bytes(samplecount) > readblock)
				samplecount = format.bytes2samples(readblock);

				MXDEBUG("[%p] reading %zu (%zu/%zu) samples", this, samplecount, buffer.fill, buffer.size);

				// Read desired amount of samples. We reached the end if we get 0 bytes.
				size_t read_samples;
				MXDEBUG("[audio_fifo %p] read from %p into %p till %p", this, source, samplepoint, samplepoint+samplecount);
				if(source != NULL)
				{
					bool good = source->plain_read(samplepoint, samplecount, read_samples);
					// Only calling end when really nothing came or an error occured.
					// For ordinary blocking I/O, a simple shortage of data would
					// suffice, but not for buffering from a pipe. One could consider
					// dropping pipe support since separate decoder processes are not
					// in use anymore. But perhaps they come into fashion again?
					if(!good)
					{
						MDEBUG("[audio_fifo %p] read failed :-(", this);
						err.take_over(source->err);
						readstate = false;
					}
					// Is end because of error still a proper end?
					// I say yes!
					if(!good || read_samples == 0) 
					{
						MDEBUG("[%p] ******* got end! *******", this);
						buffer.end_reached();
					}

					if(read_samples > samplecount)
					{
						MDEBUG("[%p] Too much data?! Bogus!", this);
						readstate = false;
						err.occur(dmd::err::IO, "Source gave more than wanted! This is a bug.");
					}
					else buffer.written(read_samples); // That also executes zeroscans.

					MXDEBUG("[%p] read state %i, fill: %zu / %zu", this, (int)readstate, buffer.fill, buffer.size);
				}
				else
				{
					err.occur(dmd::err::IO, "There is no source to read from?!");
					readstate = false;
				}

				MXDEBUG("[audio_fifo %p] read from %p", this, source);
			}
			else{ MXDEBUG("[%p] nothing to read here", this); break; }

			// We have read something, or at least tried. So notify the world.
			// The location of this semaphore may be sub-optimal...
			dsem.post();

			// On errors, just leave. Note that the lock is released outside of the loop.
			if(!readstate) break;

			// Unlock / lock to give another thread a chance to do something on the fifo.
			lock.unlock();
			sched_yield();
			lock.lock();
		}

		// Out of the filling loop: note that we still keep a hold on that lock.
		// Release it, but not before checking for the last goodbye.
		MXDEBUG("[%p] fifo_reader done for now, readstate=%i, got_end=%i", this, (int)readstate, (int)!buffer.expect_more());
		if(goodbye)
		{
			lock.unlock();
			break;
		}
		else lock.unlock();
	} // The big endless working loop.

	MDEBUG("[%p] fifo_reader finished", (void*)this);
}


// The audio_fifo is an inpute file wrapped around a fifo_buffer (that is filled from another input_file).
audio_fifo::audio_fifo(int priority):
	 thread_container("audio_fifo") // Nothing special.
	,thread("FIFO reader", priority)
	,input_file(input::invalid) // The FIFO is some special beast.
	,reader_there(false), locked(false)
	,source(NULL)
	,readstate(true), goodbye(false)
{
	CONSTRUCT("audio_fifo");
	size_seconds = param.as_double("input", "prebuffer");
	if(!(size_seconds > 0))
	{
		err.occur(dmd::err::BAD_PAR, "Won't create a zero-sized fifo buffer");
		return;
	}

	zero_level = param.as_float("input", "zerolevel");
	zero_range = param.as_double("input", "zerorange");
	if(zero_level < 0) zero_level = -zero_level;
	if(zero_range < 0) zero_range = -zero_range;
	if(zero_range > size_seconds) zero_range = size_seconds;

	stop(); // The reader should not try to do anything before we loaded a file!
	SDEBUG("Trying to create FIFO reader...");

	// Hehe, funky... I create the thread that is myself.
	if(create_thread(this))
	reader_there = true;
	else
	err.occur(dmd::err::SETUP, "Cannot start fifo_reader.");

	if(!reader_there)	MERROR("MAJOR BORKAGE in %p!!! No reader thread!", (void*)this);
}

audio_fifo::~audio_fifo()
{
	DESTBEGIN("audio_fifo");
	if(reader_there)
	{
		MDEBUG("[%p] going to stop fifo reader", (void*)this);
		stop();
		goodbye = true;
		rsem.post(); // Ensure that the fifo_reader is not waiting for a reader.
		start();
		kill_thread(this);
	}
	else start(); // Just unlocking the lock, to be tidy.

	DESTEND("audio_fifo");
}

// The thread barriers.
// We assume there is one thread working with the audio_fifo interface.
// We prevent multiple locking (deadlocking itself? hm I should read up on that, is this possible?) by a flag.
// Anyhow, it works like this.
// The fifo_reader takes the lock before doing any work, even before checking if any refill is needed.
// So when we have the lock here, the fifo_reader is stopped. Unlocking lets the fifo_reader loose immediately, unless the goodbye flag is set.

void audio_fifo::stop()
{
	if(locked) return;

	lock.lock();
	locked = true;
}

void audio_fifo::start()
{
	if(locked) lock.unlock();

	locked = false;
}

// Now comes the functionality of the outside world in form of the audio_file API.

// Read a certain amount of samples.
// The FIFO and zeroscan magic is hidden in the buffer.
bool audio_fifo::do_read(void* abuf, size_t wanted_bytes, size_t &got_bytes)
{
	if(!reader_there) return false;

	size_t wanted_samples = format.bytes2samples(wanted_bytes);
	audio_io_type *buf = (audio_io_type*)abuf;
	got_bytes = 0;
	while(wanted_samples)
	{
		stop(); // I work with the buffer!
		// Waiting for the buffer to be filled.
		// On read error, do not immediately bail out, only if all data is consumed.
		bool good_wait = wait_for_it(wanted_samples);

		// Extract what we're gonna get.
		size_t got_samples_now = buffer.extract(buf, wanted_samples);
		buf += got_samples_now*buffer.channels;
		wanted_samples -= got_samples_now;
		got_bytes += format.samples2bytes(got_samples_now);
		MXDEBUG("[%p] extracted %zu samples (total bytes: %zu)", this, got_samples_now, got_bytes);
		rsem.post();

		// If we did not get all and waiting failed (reader error), signal a problem.
		if(!good_wait && wanted_samples) return false;

		// If we got the end, it doesn't matter where the fifo_reader is stuck, so start() not needed.
		if(!buffer.expect_more()) break;

		start(); // The fifo_reader has got it again.
	}

	return true;
}

// Where should the logic for determining the correct type of input file go?
// What would work now is to close() and switch the source, then open a new file...
// Should the audio_fifo decide itself what input_file to use?

bool audio_fifo::do_open(const string location)
{
	stop();
	if(source == NULL)
	return err.occur(dmd::err::NO_DEVICE, "No source.");

	buffer.reset();
	readstate = true;
	if(source->do_open_safe(location))
	{
		format = source->format;

		// Apply the buffer size and zero range stuff with current sample rate.
		size_t bufsize = (size_t) format.samples(size_seconds);
		size_t zrange  = (size_t) format.samples(zero_range);
		audio_io_type zlevel;
		audio::scale_to_sample(zero_level, audio::audio_io_code, &zlevel);

		buffer.reallocate(format.channels, bufsize);
		position = 0;
		source_length = source->do_length();

		buffer.set_zero(param.as_bool("input", "zeroscan"), zlevel, zrange);

		// Be careful... buffer allocation could have failed.
		if(buffer.size < bufsize)
		{
			source->do_close();
			return err.occur(dmd::err::NOMEM, "Failed to resize FIFO buffer.");
		}

		rsem.post();
		// Not starting here mainly because we want to be able to apply stuff like EQ settings before the reader thread gets into gear.
		return true;
	}
	else
	{
		err = source->err;
		return false;
	}
}

void audio_fifo::do_close(void)
{
	stop();
	if(source != NULL) source->do_close();
}

// Wait until we can read a certain amount of samples (or reached the end).
// Call this while holding the lock!
//  stop()
//  wait_for_it()
//  start()
bool audio_fifo::wait_for_it(size_t want_samples)
{
	MDEBUG("[audio_fifo %p] waiting for %zu", this, want_samples);
	bool start_here = !locked;
	stop();
	while(readstate && buffer.available() < want_samples && buffer.expect_more())
	{
		MDEBUG("[audio_fifo %p] %zu<%zu (%zu/%zu) is not enough, waiting", this, buffer.available(), want_samples, buffer.fill, buffer.size);
		start();
		dsem.wait();
		stop();
	}
	MDEBUG("[audio_fifo %p] waiting done", this);
	if(start_here) start();

	return readstate;
}

bool audio_fifo::wait_for_end(void)
{
	MDEBUG("[audio_fifo %p] wait for end", this);
	size_t avail = buffer.available();
	while(avail > 0 || buffer.expect_more())
	{
		position += buffer.skip(avail);
		rsem.post();
		if(!wait_for_it(buffer.size)) return false;

		avail = buffer.available();
	}
	return readstate;
}

// Seeking is rather elaborate... we need to take care of microseeking in the buffer and source files with known / unknown lengths, in any case zeroscan has to be taken into account.
bool audio_fifo::do_seek(off_t pos)
{
	// Only start the reader here if it is stopped here.
	bool start_here = !locked;

	if(source == NULL)
	return err.occur(dmd::err::NO_DEVICE, "No source.");

	stop();

	MDEBUG("[audio_fifo %p] seeking, wait for front", this);

	// We need to be sure that the front is scanned, so at least one sample is available.
	if(!wait_for_it(1)) return false;

	// We got the offset now between the source and the outside world.
	// So we can compute the position to seek to... only problem:
	// We could seek too near to the end, inside the zero range. To prevent that, we need the exact length of the source file stored. It is not allowed to change...
	// Anyhow, let's first look if we need to seek at all...

	MDEBUG("[audio_fifo %p] seeking to %"OFF_P" from %"OFF_P", check if buffer (%zu)", this, pos, position, buffer.fill);
	// Check if the position is in buffer range.
	// We can only go forward here, going backwards needs more housekeeping and also luck in the reader thread not working fast.
	// Try this for two buffer lengths, to cover the safety for back zeroscan.
	if(pos > position)
	for(int loop=0; loop<2; ++loop)
	{
		size_t diff = (size_t)(pos-position);
		if(diff <= buffer.size)
		{
			if(!wait_for_it(diff)) return false;

			// For differing reasons, the skipped amount may be less than diff.
			position += buffer.skip(diff);
			rsem.post(); // Do not forget to signal that we pulled out data!
		}
		else break;
	}
	// If we reached the spot already or there is nothing more to come, we're at the end.
	if(pos == position || (pos > position && buffer.got_end))
	{
		MDEBUG("[audio_fifo %p] Reached position %"OFF_P" in buffer, at end: %i.", this, position, (int)buffer.got_end);
		return true;
	}

	MDEBUG("[audio_fifo %p] seeking, really need to", this);
	// OK, if we are still here, it is about serious seeking.
	size_t source_pos = pos+buffer.frontskip;
	if(source_length >= 0) // If there is a valid length, we can be accurate.
	{
		if(source_pos > source_length-buffer.zero_range)
		source_pos = source_length-buffer.zero_range;
	}
	// We need to live with the danger of hitting the end zero range.
	// Now seek to the computed location and skip up to the point (if necessary).

	// Discard everything in the buffer, also discard the end zeroscan result and end of file state.
	buffer.discard();
	MDEBUG("[audio_fifo %p] seeking, really seek", this);
	if(!source->do_seek(source_pos))
	{
		MDEBUG("[audio_fifo %p] seek failed", this);
		err.occur(dmd::err::IO, "Source cannot seek.");
	}
	position = source->position - buffer.frontskip;
	// We did serious action on the consumer side, make sure to signal to the reader to take action even when it found the end of the track before.
	rsem.post();
	// But the reader is still stopped!
	while(pos > position)
	{
		size_t diff = (size_t)(pos - position);
		if(!wait_for_it(diff)) return false;

		position += buffer.skip(diff);
		rsem.post();
		if(!buffer.expect_more()) break;
	}

	if(start_here) start();

	MDEBUG("[audio_fifo %p] seek successful", this);
	return true;
}

// This could invalidate current buffer contents and seek back.
// But do I want this? Fast-paced eq fading fun somehow precludes use of the big prebuffer (or rather uses a postprocessing eq filter at the mixing stage).
bool audio_fifo::do_eq(const vector<float> &val)
{
	bool start_here = !locked;
	if(source == NULL)
	return err.occur(dmd::err::NO_DEVICE, "No source.");

	stop();
	bool ret = source->do_eq(val);
	if(ret && buffer.fill > 0)
	{
		off_t step_back = buffer.fill;
		buffer.discard();
		MDEBUG("[audio_fifo %p] discarded buffer after EQ change, seeking back to %"OFF_P" from %"OFF_P".", this, (off_p)source->position-step_back, (off_p)source->position);
		ret = source->do_seek(source->position-step_back);
		if(!ret)
		{
			MERROR("[audio_fifo %p] EQ backseek failed still trying to continue...", this);
			err.occur(dmd::err::IO, "Source cannot seek.");
		}
		rsem.post();
	}
	if(start_here) start();

	return ret;
}

off_t audio_fifo::do_length(void)
{
	bool start_here = !locked;
	if(source == NULL)
	{
		err.occur(dmd::err::NO_DEVICE, "No source.");
		return -1;
	}
	// Now we need to ensure that we give the correct lenght.
	// With zeroscan, that means we need to peek near the end to see how many zeroes will be filtered.
	if(buffer.zeroscan)
	{
		// Wait until front is scanned, then seek near end and wait for it.
		// Then seek back... 
		// Question: Should we clear the zero status or use it for later?
		if(source_length < 0) return source_length;

		// Access to position doesn't need to be synchronized with stop().
		off_t save_pos = position;
		// That gets us to the end for sure (if source_length is correct).
		if(!do_seek(source_length)) return -1;

		// Ensure that there is nothing more, really.
		stop();
		if(!wait_for_end()) return -1;

		// Now we have to be at the end.
		off_t len = position;
		// Get back!
		if(!do_seek(save_pos)) return -1;

		if(start_here) start();

		return len;
	}
	else return source_length;
}

// This stops the fifo_reader and sets the source file.
// You need to open() a file afterwards!
void audio_fifo::set_source(input_file *the_source)
{
	stop();
	MDEBUG("[audio_fifo %p] set_source %p", this, the_source);
	source = the_source;
}

}
