#ifndef DMD_H_AUDIO_BUFFERPOOL
#define DMD_H_AUDIO_BUFFERPOOL

/*
	audio_bufferpool: A pool of buffers, with two camps.

	part of DerMixD
	(c)2010 Thomas Orgis, licensed under GPLv2

	This is a data structure for multibuffering intended for audio outputs.
	You have a set of buffers in the pool, which can be in the "available" or "used" camp.
	The mixer is supposed to wait until a buffer is available, then use that for mixing and finally send it over to the "used" camp, also issuing a message to the writer thread.
	The writer thread waits until it gets the message to pull a buffer from the "used" camp, gets the next one, writes to the output and pushes the buffer back into the "available" camp.

	Does that make sense? Let's see.

	The buffer pool takes the single buffer size and channel count from global parameters.
*/

#include "audio/audio_buffer.hxx"
#include "threads/semaphore.hxx"
#include "threads/threads.hxx"
#include <vector>
#include <queue>

namespace dmd
{

class audio_bufferpool: public real_mutex
{
	public:
	 audio_bufferpool();
	~audio_bufferpool();

	// Add a certain count of buffers to the pool.
	// Returns the count of successfully added buffers.
	size_t add(size_t c);
	// Not adding the respective method to remove buffers... might be complicated for buffers being in use

	// Check how many buffers are actually there.
	// Wait for free buffer being available / used buffer queued and extract it.
	// The given thread is made cancellable during wait.
	audio_buffer* fetch_avail(thread * = NULL);
	audio_buffer* fetch_used(thread * = NULL);
	// Add a buffer to the used list, or the list of available buffers again.
	void store_used(audio_buffer *);
	void store_avail(audio_buffer *);

	private:
	size_t bufsize;
	unsigned int channels;
	// The actual buffer pool, fixed size.
	std::vector<audio_buffer*> pool;
	// Two views on the pool, a queue of available and one of filled buffers.
	// One might opmimize the queue for fixed maximum size?
	std::queue<audio_buffer*> avail, used;
	// Semaphores for avail and used queues, respectively.
	semaphore asem, usem;
};

}
#endif
