/*
	audio_bufferpool: A pool of buffers, with two camps.

	part of DerMixD
	(c)2010 Thomas Orgis, licensed under GPLv2
*/

#include "basics.hxx"
#include "audio/audio_bufferpool.hxx"
#include "param_init.hxx"
#include "shortcuts.hxx"

#include "debug.hxx"

namespace dmd
{

#define CMDEBUG(s, ...)  MDEBUG( "[audio_bufferpool %p] " s, this,  __VA_ARGS__)
#define CSDEBUG(s)       MDEBUG( "[audio_bufferpool %p] " s, this)
#define CMXDEBUG(s, ...) MXDEBUG( "[audio_bufferpool %p] " s, this,  __VA_ARGS__)
#define CSXDEBUG(s)      MXDEBUG( "[audio_bufferpool %p] " s, this)

audio_bufferpool::audio_bufferpool()
{
	CONSTRUCT("audio_bufferpool");
	bufsize = param.as_size("buffer");
	channels = param.as_uint("channels");
	CMDEBUG("%i channels, size %zu", channels, bufsize);
}

audio_bufferpool::~audio_bufferpool()
{
	DESTBEGIN("audio_bufferpool");
	// Well... one can debate over the lock bringing real value here.
	// Usually, nobody should use the pool anymore when destructing...
	lock();
	// That one needs to delete the actual buffer objects.
	FOR_VECTOR(audio_buffer*, pool, buf) delete *buf;

	unlock();
	DESTEND("audio_bufferpool");
}

size_t audio_bufferpool::add(size_t count)
{
	size_t c = 0;
	lock();
	for(; c<count; ++c)
	{
		CMDEBUG("adding buffer %zu ", pool.size());
		audio_buffer* nb = new audio_buffer(channels, bufsize);
		if(nb == NULL)
		{
			SERROR("Dammit, cannot allocate buffers?");
			break;
		}
		if(nb->size != bufsize)
		{
			SERROR("That buffer in the pool wasn't able to resize.");
			delete nb;
			break;
		}
		CMDEBUG("pushing the buffer %p (%zu)", nb, nb->size);
		pool.push_back(nb);
		avail.push(nb);
		asem.post();
	}
	unlock();
	return c;
}

// If you try to fetch available buffers with no buffers there at all, you are to blame yourself.
audio_buffer* audio_bufferpool::fetch_avail(thread *ted)
{
	if(pool.size() == 0)
	{
		MERROR("[%p] Attempt to fetch_avail from an empty pool!", this);
		return NULL;
	}
	audio_buffer *buf = NULL;
	CSXDEBUG("waiting for available buffer");
	asem.wait(ted);
	lock();
	buf = avail.front();
	avail.pop();
	CMXDEBUG("got available %p (%zu/%zu), %zu buffers left", buf, buf->fill, buf->size, avail.size());
	unlock();
	return buf;
}

audio_buffer* audio_bufferpool::fetch_used(thread *ted)
{
	if(pool.size() == 0)
	{
		MERROR("[%p] Attempt to fetch_used from an empty pool!", this);
		return NULL;
	}
	audio_buffer *buf = NULL;
	CSXDEBUG("waiting for used buffer");
	usem.wait(ted);
	lock();
	buf = used.front();
	used.pop();
	CMXDEBUG("got used (%p, %zu/%zu), %zu left", buf, buf->fill, buf->size, used.size());
	unlock();
	return buf;
}

void audio_bufferpool::store_used(audio_buffer *buf)
{
	lock();
	used.push(buf);
	CMXDEBUG("pushed used buffer (%p, %zu/%zu), now got %zu", buf, buf->fill, buf->size, used.size());
	usem.post();
	unlock();
}

void audio_bufferpool::store_avail(audio_buffer *buf)
{
	lock();
	avail.push(buf);
	CMXDEBUG("pushed available buffer (%p, %zu), now got %zu", buf, buf->size, used.size());
	asem.post();
	unlock();
}


}
