#ifndef DMD_H_AUDIO_BUFFER
#define DMD_H_AUDIO_BUFFER

/*
	audio_buffer: audio data storage
	
	part of DerMixD
	(c)2004-2010 Thomas Orgis, licensed under GPLv2
*/

#include "audio/audio_format.hxx"
#include <cstring>

/*
	the audio buffer used throughout this project
	flexible in respect to different audio formats (including bits / audio_type)
	
	It provides mainly transparent allocation and reallocation of memory for the wanted number of samples
	together with administrative info as fill or virginity.
	
	...my first template class...
*/

//definition
// type, pointer type, size type, channel count type
template <class T, class PT>
class generic_audio_buffer
{
	public:
		unsigned int channels;
		size_t size;
		size_t fill;
		PT ptr;
		bool virgin;
		generic_audio_buffer();
		generic_audio_buffer(size_t s);
		generic_audio_buffer(unsigned int c, size_t s);
		~generic_audio_buffer();
		size_t reallocate(size_t s);
		size_t reallocate(unsigned int c, size_t s);
		void zero();
		// One block of samples for all channels.
		size_t blocksize(){ return sizeof(T)*channels; };
		// Whole size.
		size_t mysize(){ return blocksize()*size; };
		size_t my_current_size(){ return blocksize()*fill; };
		void printout(std::ostream &wayout = std::cout);
		void print_text(std::ostream &wayout);
		void print_text_header(std::ostream &wayout);
		void swap(generic_audio_buffer<T,PT> &willie);
		size_t free_space(){ return size > fill ? size-fill : 0; };
		PT free_ptr(){ return ptr+channels*fill; };
		// Direct access to storage. First raw index (disregarding channel interleaving).
		T& operator[](size_t i){ return ptr[i]; };
		// Second, addressing via offset and channel index (0-based, of course).
		T& operator()(size_t o, unsigned int c){ return ptr[o*channels+c]; };
};

//implementation

template <class T, class PT>
generic_audio_buffer<T,PT>::generic_audio_buffer():
	channels(2), size(0), fill(0), ptr(NULL), virgin(true)
{
	//construct("audio_buffer");
}

template <class T, class PT>
generic_audio_buffer<T,PT>::generic_audio_buffer(size_t s):
	channels(2), size(0), fill(0), ptr(NULL), virgin(true)
{
	//construct("audio_buffer");
	reallocate(s);
}

template <class T, class PT>
generic_audio_buffer<T,PT>::generic_audio_buffer(unsigned int c, size_t s):
	channels(c), size(0), fill(0), ptr(NULL), virgin(true)
{
	//construct("audio_buffer");
	reallocate(s);
}

template <class T, class PT>
generic_audio_buffer<T,PT>::~generic_audio_buffer()
{
	//destbegin("audio_buffer");
	if(ptr != NULL) delete[] ptr;
	//destend("audio_buffer");
}

// Change buffer size with memory reallocation.
// Keep current fill intact, if possible.
template <class T, class PT>
size_t generic_audio_buffer<T,PT>::reallocate(size_t s)
{
	if(s != size)
	{
		// Gnn... should use plain malloc/realloc.
		PT oldptr = ptr;
		if( (ptr = new T [s*channels]) != NULL ) size = s;
		else size = 0;
		virgin = false; // memory not cleared in any case
		if(fill && oldptr != NULL)
		{
			if(fill > size) fill = size;

			std::memcpy(ptr, oldptr, fill*blocksize());
		}
		if(oldptr != NULL) delete[] ptr;
	}
	return size;
}

// Change buffer size and channels with memory reallocation
// This destroys the old contents.
template <class T, class PT>
size_t generic_audio_buffer<T,PT>::reallocate(unsigned int c, size_t s)
{
	fill = 0;
	if(s*c != size*channels) //if the needed memory size is different
	{
		if(ptr != NULL) delete[] ptr;
		if( (ptr = new T [s*c]) == NULL ) s = 0;
		virgin = false; //memory not initialized
	}
	channels = c;
	size = s;
	return size;
}

//sweep it clean
template <class T, class PT>
void generic_audio_buffer<T,PT>::zero()
{
	//any benefit? seems not to be the case...
	//memset(const_cast<T*> (ptr), 0, mysize());
	for(size_t i = 0; i < size*channels; ++i) ptr[i] = 0;
	fill = 0;
	virgin = true;
}

//complete ascii printout
template <class T, class PT>
void generic_audio_buffer<T,PT>::printout(std::ostream &wayout)
{
	print_text_header(wayout);
	print_text(wayout);
}

//print header for ascii text output
template <class T, class PT>
void generic_audio_buffer<T,PT>::print_text_header(std::ostream &wayout)
{
	wayout << "#buffer printout" << "\n" << "#";
	for(unsigned int ch = 0; ch < channels; ++ch)
	{
		if(ch) wayout << "\t";
		wayout << "channel" << ch;
	}
	wayout << "\n";
}

//print buffer contents as ascii text
template <class T, class PT>
void generic_audio_buffer<T,PT>::print_text(std::ostream &wayout)
{
	for(size_t i = 0; i < fill*channels; i+=channels)
	{
		for(unsigned int ch = 0; ch < channels; ++ch)
		{
			if(ch) wayout << "\t";
			wayout << ptr[i+ch];
		}
		wayout << "\n";
	}
}

//swapping of the memory of buffers with same properties
template <class T, class PT>
void generic_audio_buffer<T,PT>::swap(generic_audio_buffer<T,PT> &willie)
{
	PT tmp = ptr;
	ptr = willie.ptr;
	willie.ptr = tmp;
	
	size_t tf = fill;
	fill = willie.fill;
	willie.fill = tf;
}

// The main buffer type for the mixer is defined as subclass of that.
typedef generic_audio_buffer<audio_type, audio_type *> mixer_buffer_base;
// Possibly the only incarnatio fo the simple buffer for the future: The buffers the mixer works on.
// Simple blocks of numbers in internal sample format.
// Plus: Some utility functions needed in the mixer.
// It's actually silly to create a new derived class just to add a simple method... wouldn't it make sense to just define a plain old function to work on a buffer object?
// Additional twist with this buffer: I am adding separate access routines for non-interleaved storage.
// There is not much safety against mixing interleaved and non-interleaved access... so, be careful, until I figured out how to do this properly.
class mixer_buffer: public generic_audio_buffer<audio_type, audio_type *>
{
	public:
	bool interleaved; // true if storage format is the default interleaved form.
	// Repeating the constructores sucks.
	 mixer_buffer(): mixer_buffer_base() {};
	 mixer_buffer(size_t s): mixer_buffer_base(s) {};
	 mixer_buffer(unsigned int c, size_t s): mixer_buffer_base(c,s) {};
	~mixer_buffer();
	// Fill with contents of named buffer, return number of consumed samples.
	size_t fill_in(mixer_buffer &source);
	// Adjust beginning of buffer towards the provided levels.
	// This helps perceived smoothness of gapless following.
	void smooth_beginning(audio_type *startval);
	// Drop a certain amount of samples from the beginning.
	// Shift the rest, reduce fill.
	void drop(size_t count);
	// chesamplify = convert CHannels, rESAMPLe, AMPLIFY.
	// This has been fullchansample() in audio_functions.
	void chesamplify(mixer_buffer& source, const size_t b, const float vol[2], audio_type startval[2]);
	// Non-interleaved access. Only valid when interleaved == false.
	audio_type* chan_ptr(unsigned int c);
	// (De)Interleave from/to another audio buffer, that is supposed to be interleaved.
	// Deinterleaving appends to the target buffer -- it needs to have enough free space!
	// The interleave method sets the interleaved flag; nothing else sets it back.
	// Oh, and you have to match buffer sizes before calling these methods. I don't like random malloc()s during time-critical operation.
	bool deinterleave(mixer_buffer &from);
	bool interleave(mixer_buffer &to);
};

// This one is temporary... fixing I/O buffer format to signed 16 bit for now.
// I'll do something flexible in future, but first get the whole thing going again.
// I imagine some generic audio buffer class that has the actual sample format stored at runtime.
typedef generic_audio_buffer<int16_t, int16_t *> audio_buffer;


#endif
