#ifndef DMD_H_ACTION
#define DMD_H_ACTION

/*
	action: data structure for communicating orders and their execution
	
	part of DerMixD
	(c)2004-2011 Thomas Orgis, licensed under GPLv2
	
	actions do the parsing and de-parsing (creation of command string) on their own, provided they have a matching set of definitions
	actions transport arguments
	actions transport communication links
	actions can contain actions...
	actions are important
*/

#include <sys/types.h>
#include <string>
#include <vector>

#include "coms/comm_data.hxx"
#include "threads/semaphore.hxx"
class error;

namespace dmd {
class socket_writer;

namespace api {

// IDs of all possible actions.
// Actual action definitions can be elsewhere, but we fix one common space for action IDs to prevent any accidential clashes.
enum aid
{
	 NOTHING = 0
	,NOTIFY
	,IN_PAUSE
	,IN_LOAD 
	,IN_EJECT 
	,IN_START
	,IN_SEEK
	,IN_RSEEK
	,BIND
	,UNBIND
	,IN_VOLUME
	,IN_PREAMP
	,IN_BASS
	,IN_MID
	,IN_TREBLE
	,IN_SPEED
	,IN_PITCH
	,IN_EQ
	,FOLLOW
	,IN_NOFOLLOW
	,NOTIFY_IN_LOAD
	,NOTIFY_IN_SEEK
	,IN_WATCH
	,IN_UNWATCH
	,IN_SCRIPT
	,IN_CLEARSCRIPT
	,IN_SHOWSCRIPT
	,IN_VOL
	,IN_LENGTH
	,IN_INLOAD
	,IN_SCAN
	,IN_PREREAD
	,IN_EFFECT_ADD
	,IN_EFFECT_REMOVE
	,IN_EFFECT_PARAM
	,IN_EFFECT_LIST
	,IN_EFFECT_HELP
	,IN_EFFECT_BYPASS

	,INWORK_PLAY
	,INWORK_NODEC

	,OUT_PAUSE
	,OUT_START
	,OUT_LOAD
	,OUT_EJECT
	,NOTIFY_OUT_LOAD
	,OUT_SCRIPT
	,OUT_CLEARSCRIPT

	// To be merged with inwork action IDs.
	,OUTWORK_PLAY
	,OUTWORK_OPEN
	,OUTWORK_CLOSE   // Close the currently loaded file/resource, if any.
	,OUTWORK_NODEC   // Actually kill the decoder (output_file) instance.
	,OUTWORK_QUIT

	,SLEEP
	,QUIT
	,ADD_IN
	,ADD_OUT
	,REM_IN
	,REM_OUT
	,FULLSTAT
	,CLOSE
	,TIME_SCRIPT
	,TIME_CLEARSCRIPT
	,SAY
	,THREADSTAT
	,LS
	,PWD
	,CD

	,ID
	,FADEOUT
	,HELP
	,SHOWAPI
	,SPY
	,UNSPY
	,SHOWID
	,SHOWPEERS
	,ADDPEER
	,REMPEER
	,PEER
	,EFFECT_QUERY

	,FEEDBACK
};

// Action argument type codes.
enum argt
{
	 ACT_INT = 1 // signed long integer
	,ACT_STRING  // text
	,ACT_FLOAT   // floating point value
	,ACT_OFFSET  // offset value (integer)
	,ACT_SIZE    // size-like integer (unsigned)
};

// Some flags detailing action properties.
enum flags
{
	// Flags for he action parsing itself.
	 ACT_CONTAINER = 0x1  // Is a container.
	,ACT_NOSUB     = 0x2  // Is not allowed inside a container.
	,ACT_OPTARG    = 0x4  // all or no argument
	,ACT_OPTONE    = 0x8  // all but the first argument are optional, any number
	// User flags.
	,SOCK_LOCAL    = 0x10
	,SOCK_NOTIFY   = 0x20
	,SOCK_POST     = 0x40
	,SOCK_IN       = 0x80
	,SOCK_OUT      = 0x100
	,SOCK_ANS      = 0x200
	,SOCK_PATH     = 0x400
	,SOCK_FINAL    = 0x800
	,SOCK_COMPANY  = 0x1000
	,INWORK_FILE   = 0x2000
};


//structure for definition of action syntax and storage
// There's fixed storage for argument types, with ACT_MAX_ARG entries.
// Crufty ... but that way, I can simply write down the action definitions without bothering for a mess on the heap constructing objects.
#define ACT_MAX_ARG 5
typedef struct def
{
	enum aid id; // numerical ID
	const char* name; // "outload"
	const char* info;
	unsigned int argc; // 3
	const char* arginfo[ACT_MAX_ARG];
	int argt[ACT_MAX_ARG]; // {INT, STRING, STRING}
	int flags; // ACT_CONT, ACT_NOSUB 
} def;

//general tools for action definitions

//find definition of action in world with given command name
const def* find_def(std::string& name, const def* world);
//append list of all command names in this worls to app
void list(const def* world, std::string& app);
//append action info and syntax help to app
void help(const def* act, std::string& app);
void allhelp(const def* world, std::vector<std::string*> &app);

} } // namespace, to be extended below

//the action class itself: representation, storage, parsing

/*
	there are two basic types of actions: single or container

	{ IN_EQ, "eq", "set all three eq factors", 4 , { "channel", "bass", "mid", "treble" }, { ACT_INT, ACT_FLOAT, ACT_FLOAT, ACT_FLOAT}, 0 }

	single action splits argument into argc pieces and puts them into strings, floats, ints, accordingly
	for container actions the last argument must be a string and this is not appended to strings but instead parsed again, creating the subaction.
	
	{ IN_SCRIPT, "script", "script an action triggered by an inchannel passing certain time/position, command is just any valid (and allowed) command with arguments", 3, { "channel", "time in seconds", "command" }, { ACT_INT, ACT_FLOAT, ACT_STRING }, ACT_CONTAINER|ACT_NOSUB }
	
	Hm, with this subaction pointer I guess I can endlessly contain actions in each other...
*/

class action
{
	public:
	bool disposable; // A flag to tell if the action should be deleted after processing. False by default.
	//unsigned char id; //type of action in some context
	const dmd::api::def* def; //the definition, including ID (subdef is filled for contained action)
	const dmd::api::def* def_world; //the world of definitions that is needed for parsing
	action* subaction; //the contained action
	bool first;
	bool good; // That one only indicates if the action has been parsed from text, successfully.
	int idmask;
	std::string fullname;

	//arguments for action
	std::vector<std::string> strings;
	// It is lame to have three different integer vectors.
	// But it would be hacky to cram them into one intmax_t, too.
	std::vector<long>   longs;
	std::vector<off_t>  offsets;
	std::vector<size_t> sizes;
	std::vector<float> floats;
	comm_data* comdat; //associated communication data (socketeer)
	dmd::socket_writer* swr; //associated socket_writer (output of resulting messages)

	//action(unsigned char i)
	action(const dmd::api::def* world);
	action(const dmd::api::def* world, const dmd::api::def* me);
	action();
	~action();

	//just clear data
	void clear_data()
	{
		strings.clear();
		longs.clear();
		sizes.clear();
		offsets.clear();
		floats.clear();
	}

	//really clear
	void clear()
	{
		clear_data();
		def = NULL;
		first = true;
		good = false;
		if(subaction != NULL)
		{
			delete subaction;
			subaction = NULL;
		}
		fullname.clear();
	};

	/*
		parse command and transform into the desired action,
		return true when definition found
	*/
	bool parse(const std::string &command);
	
	/*
		add command text that results in this action when parsed to given string
		idea is that this method should be useful when printing script actions...
	*/
	bool print(std::string &command);

	void success(std::string& app); //append error or success message(s) of myself and subactions to app
	void give_error(const std::string msg, std::string &app); //append given error message with [fullname] in front
	// Take a new error and stuff it into the error chain in comdat.
	// If there is no comdat, the error is silently deleted.
	void push_error(error *err);
	void msg(const std::string msg, std::string &app); //
	void begin(std::string &app);
	void error_begin(std::string &app);
	void end(std::string &app);

	void set_sub(action* newsub); //just set subaction without further precautions... for now
	bool valid(); // Check if the action has the correct amount of parameters for its definition. Also recureses down to subactions.
	// Shortcut to check if contained comm_data has errors noted.
	bool have_errors(){ if(comdat != NULL) return !comdat->errors.empty();
	else return false; };
	// Notify the waiting thread via communication semaphore, if present. 
	void notify(){ if(comdat != NULL && comdat->waitress != NULL) comdat->waitress->post(); };
};

#endif
