/*
	action: data structure for communicating orders and their execution

	part of DerMixD
	(c)2004-2014 Thomas Orgis, licensed under GPLv2
*/

#include "basics.hxx"

#include "action.hxx"
#include "comparse.hxx"
#include "parsenum.hxx"
#include "tstring.hxx"
#include "coms/comm_data.hxx"
#include "errorchain.hxx"

#include <stdio.h> // I don't get snprintf on Solaris otherwise:-/

#include "debug.hxx"

using namespace std;

namespace dmd {
namespace api {

const def* find_def(string &name, const def* world)
{
	const def* def = NULL;
	//find definition, slow, slow search
	int i = -1;
	if(world == NULL) return def;

	while((def == NULL) && (world[++i].id != NOTHING))
	{
		if(name == world[i].name) def = &world[i];
	}
#ifdef DEBUG
	if(def != NULL)
	{
		fprintf(stderr, "definition for %s", def->name);
		for(size_t i = 0; i < def->argc; ++i) fprintf(stderr, " %i", def->argt[i]);

		fprintf(stderr, "\n");
	}
#endif
	return def;
}

void list(const def* world, string &app)
{
	int i = -1;
	while(world[++i].id != NOTHING)
	{
		if(i != 0) app += " ";
		app += world[i].name;
	}
}

void allhelp(const def* world, vector<string*> &app)
{
	int i = -1;
	if(world == NULL)
	{
		app.push_back(new string("Error: No world to get action in!"));
		return;
	}
	app.push_back(new string("<name> parm1(type) parm2(type) ...; [ flags ]:\t<description>"));
	app.push_back(new string("meaning of flags:"));
	app.push_back(new string("\tcontainer -- this action can contain a subaction (given as parameter); read: this is the script command"));
	app.push_back(new string("\tnosub -- this action cannot be a subaction in a container; read: not allowed in script"));
	app.push_back(new string("\toptarg -- parameters are optional: either specify all or none"));
	app.push_back(new string("\toptone -- all but first parameter are optional: specify any number of them"));
	app.push_back(new string("\tin -- action on input channel"));
	app.push_back(new string("\tout -- action on output channel"));
	app.push_back(new string(""));
	while(world[++i].id != NOTHING)
	{
		app.push_back(new string);
		help(&world[i], *app.back());
	}
}

// append help 
void help(const def* act, string &app)
{
	if(act == NULL)
	{
		app += "unknown command";
		return;
	}

	app += act->name;
	for(unsigned int i = 0; i < act->argc; ++i)
	{
		app += " ";
		app += act->arginfo[i];
		app += "(";
		switch(act->argt[i])
		{
			case dmd::api::ACT_INT:
				app += "integer";
			break;
			case dmd::api::ACT_OFFSET:
				app += "offset-int";
			break;
			case dmd::api::ACT_SIZE:
				app += "size-int";
			break;
			case dmd::api::ACT_FLOAT:
				app += "float";
			break;
			case dmd::api::ACT_STRING:
				app += "string";
			break;
			default:
			MERROR("Bad action argument type %i! Most likely, this is a programming error that must be fixed.", act->argt[i]);
		}
		app += ")";
	}
	app += "; [";
	if(act->flags & dmd::api::ACT_CONTAINER) app += " container";
	if(act->flags & dmd::api::ACT_NOSUB) app += " nosub";
	if(act->flags & dmd::api::ACT_OPTARG) app += " optarg";
	if(act->flags & dmd::api::ACT_OPTONE) app += " optone";
	if(act->flags & dmd::api::SOCK_IN) app += " in";
	if(act->flags & dmd::api::SOCK_OUT) app += " out";
	app += " ]";
	app += ":\t";
	app += act->info;
}

}} // namespace

action::action(const dmd::api::def* world): disposable(false)
{
	CONSTRUCT("action");
	idmask = 0;
	subaction = NULL;
	comdat = NULL;
	swr = NULL;
	clear();
	def_world = world;
}

action::action(const dmd::api::def* world, const dmd::api::def* me): disposable(false)
{
	CONSTRUCT("action");
	idmask = 0;
	subaction = NULL;
	comdat = NULL;
	swr = NULL;
	clear();
	def = me;
	def_world = world;
}

action::action()
{
	CONSTRUCT("action");
	idmask = 0;
	subaction = NULL;
	comdat = NULL;
	swr = NULL;
	clear();
	def_world = NULL;
}

action::~action()
{
	DESTBEGIN("action");
	if(subaction != NULL) delete subaction;
	DESTEND("action");
}

bool action::parse(const string &command)
{
	bool bad = false;
	clear(); //redundant?
	MDEBUG("[action %p] Parsing: %s", this, command.c_str());

	if(commando(command, fullname))
	{
		MDEBUG("[action %p] Got name '%s'.", this, fullname.c_str());
		def = dmd::api::find_def(fullname, def_world);
		if(def != NULL)
		{
			/*
				definition found, try to parse command
				we have at def->argt def->argc integers stating the types
				so, split the command into the correct number of tokens and push them to the vectors
			*/
			vector<string> tokens;
			//optional arg means: either specify _all_ or _none_
			if(argo(command, tokens, def->argc) || def->flags & (dmd::api::ACT_OPTARG|dmd::api::ACT_OPTONE))
			{
				if(!tokens.empty()) //if we indeed have some arguments, convert/store them
				{
					size_t count = tokens.size(); // could be all, could be less
					MDEBUG("[action %p] Got %zu tokens.", this, count);
					for(size_t i = 0; i < count; ++i)
					{
						int err = strnum_good;
						MDEBUG("[action %p] Token %zu, type %i", this, i, def->argt[i]);
						switch(def->argt[i])
						{
							case dmd::api::ACT_INT:
								longs.push_back(str_to_long(tokens[i].c_str(), err));
							break;
							case dmd::api::ACT_OFFSET:
								offsets.push_back(str_to_offset(tokens[i].c_str(), err));
							break;
							case dmd::api::ACT_SIZE:
								sizes.push_back(str_to_size(tokens[i].c_str(), err));
							break;
							case dmd::api::ACT_FLOAT:
								floats.push_back(atof(tokens[i].c_str()));
							break;
							case dmd::api::ACT_STRING:
								strings.push_back(tokens[i]);
							break;
							default:
								strings.push_back("Bad action argument type - bug in command descriptions!");
								bad = true;
						}
						if(err != strnum_good)
						{
							strings.push_back("Parsing of numerical argument failed (over/underflow, invalid characters).");
							bad = true;
							break;
						}
					}
				}
				else if(def->flags & dmd::api::ACT_OPTONE)
				{ // We didn't get the one mandatory argument.
					strings.push_back("required parameter missing");
					bad = true;
				}

				//now there may be 
				if(!bad && def->flags & dmd::api::ACT_CONTAINER)
				{
					if(!strings.empty()) //at least one string argument is required: the command for subaction
					{
						subaction = new action(def_world);
						subaction->comdat = comdat;
						subaction->swr = swr;
						if(subaction->parse(strings.back()))
						{
							if(! (subaction->def->flags & dmd::api::ACT_NOSUB) ) good = true;
							else strings.push_back("supposed subaction is not allowed to be subaction (not for script...)");
						}
						else strings.push_back("parsing of subaction failed");
						subaction->first = false;
						fullname += "/" + subaction->fullname;
						//even if we had errors, the subactions stay there (for diagnosis) until clear();
					}
					else strings.push_back("no command for subaction (argument string missing)");
				}
				else good = !bad;
			}
			else strings.push_back("unable to get (all) needed arguments");
			
			//parsed string stays here...
		}
		else strings.push_back("unknown command");
	}
	else strings.push_back("unable extract command name");

	// Attach the channel number if demanded.
	if(good && def->flags & idmask)
	strprintf(fullname, " %"PRIiMAX, sizes.empty() ? (intmax_t) -1 : (intmax_t)sizes[0]);

	MDEBUG("[action %p] Parsing done and %s.", this, good ? "successful" : "failed");
	return good;
}

// An action is valid if it has a proper definition and all arguments are provided.
// This includes optional arguments -- those are optional to clients, but not further down inside DerMixD.
bool action::valid()
{
	if(def == NULL) return false;

	// Go through the argument type list and count...
	// Then check if the counts match the actual argument vector sizes.
	size_t icount = 0;
	size_t zcount = 0;
	size_t ocount = 0;
	size_t scount = 0;
	size_t fcount = 0;
	for(unsigned int i = 0; i < def->argc; ++i)
	{
		switch(def->argt[i])
		{
			case dmd::api::ACT_INT:
				++icount;
			break;
			case dmd::api::ACT_SIZE:
				++ zcount;
			break;
			case dmd::api::ACT_OFFSET:
				++ocount;
			break;
			case dmd::api::ACT_FLOAT:
				++fcount;
			break;
			case dmd::api::ACT_STRING:
				++scount;
			break;
			default:
				MERROR("Programming error! Bad action argument type %i", def->argt[i]);
				return false;
		}
	}

	// The last defined argument could be the command defining the subaction... thus, not stored here.
	if( (def->flags & dmd::api::ACT_CONTAINER) && (def->argc > 0) && (def->argt[def->argc-1] == dmd::api::ACT_STRING) )
	--scount; // Safe, because it has to be >= 1 since the last argument counted.

	// Now check...
	// Think about <= or == ... playing safe for strings (this vector can be misused to som degree...)
	// Decided for <= ... to allow some trickery to submit additional optional arguments.
	if(    icount <= longs.size()
	    && zcount <= sizes.size()
	    && ocount <= offsets.size()
	    && fcount <= floats.size()
	    && scount <= strings.size() )
	{
		if(def->flags & dmd::api::ACT_CONTAINER)
		return (subaction != NULL && subaction->valid());
		else
		return (subaction == NULL);
	}
	else return false;
}


/*
	reconstruct and print command string from action type and stored arguments
	
	this is kindof tricky with optional arguments...
	well, since I opted for all or none policy, this should apply
	
	note on subaction command string: when the last string of a container is not present, the subaction is called to append its command...
*/

bool action::print(string &command)
{
	char buffer[30];
	bool ret = true;
	if((def != NULL) && good)
	{
		command += def->name;
		//when this check doesn't guarantee that we won't try to access non-existent stuff, you messed too much with action internals!
		if(valid())
		{
			size_t icount = 0;
			size_t zcount = 0;
			size_t ocount = 0;
			size_t scount = 0;
			size_t fcount = 0;
			for(unsigned int i = 0; i < def->argc; ++i)
			{
				command += " ";
				switch(def->argt[i])
				{
					case dmd::api::ACT_INT:
						snprintf(buffer, 30, "%li", longs[icount++]);
						command += buffer;
					break;
					case dmd::api::ACT_SIZE:
						snprintf(buffer, 30, "%zu", sizes[zcount++]);
						command += buffer;
					break;
					case dmd::api::ACT_OFFSET:
						snprintf(buffer, 30, "%"OFF_P, (off_p)offsets[ocount++]);
						command += buffer;
					break;
					case dmd::api::ACT_FLOAT:
						snprintf(buffer, 30, "%0.3f", floats[fcount++]);
						command += buffer;
					break;
					case dmd::api::ACT_STRING:
						if((def->flags & dmd::api::ACT_CONTAINER) && (scount >= strings.size())) ret = subaction->print(command);
						else command += strings[scount];
						++scount;
					break;
					default:
					MERROR("bad action argument type %i of argument %u of command %s - most likely, this is a programming error that must be fixed", def->argt[i], i, def->name);
				}
			}
		}
		else ret=false;
	}
	else ret=false;

	return ret;
}

void action::success(std::string &app)
{
	if(first) app += "[" + fullname + "] " + (good ? "success" : "error: ");
	if(!good)
	{
		if(!strings.empty()) app += strings.back();
		if(subaction != NULL){ app += " <- "; subaction->success(app); }
	}
}

void action::push_error(error *err)
{
	if(err == NULL) return;

	if(comdat == NULL){ delete err; return; }

	comdat->errors.push(err);
}

void action::give_error(const string msg, string &app)
{
	app += "[" + fullname + "] " + "error: " + msg;
}

void action::msg(const string msg, string &app)
{
	app += "[" + fullname + "] " + msg;
}

void action::set_sub(action* newsub)
{
	subaction = newsub;
}

void action::begin(string &app)
{
	msg("+begin", app);
}

void action::end(string &app)
{
	msg("-end", app);
}

void action::error_begin(string &app)
{
	msg("+error: Backtrace follows, beginning at lowest level.", app);
}
