/*
	piperead: reading (audio) data from a pipe
	
	part of DerMixD
	(c)2004/5 Thomas Orgis, licensed under GPLv2
*/

#include "common.hxx"
#include "secsleep.hxx"

#include <sys/time.h>
#include <errno.h>

/*
	unsigned long int just_read(): try to read from a specified fd a specified number of bytes; abort if no data there and some volatile flag set

	input:
		int fd: file descriptor to read from
		char* buffy: pointer to the memory block where the bytes are to be read into; must be preallocated to contain the wanted number of bytes
		unsigned long int bytes_wanted: the number of bytes to read
		volatile bool* endboolA, endboolB: two (addresses of) flags that are checked periodically; read aborts when no data there and one of these set (this has multithreading in mind - the flags can be changed  by another thread after starting just_read()!)
		volatile bool* intbool: just another bool flag for ending the read attempts... I should reduce them to one and handle the diffs externally
		int& state: is for output (referenced variable is changed on special events)

	output:
		return  value: the number of bytes actually read
		state: is set to -1 if some pipe error occurs, to 1 if end is reached (no data available and one of the flags set);


	In DerMixD this is used for reading data from a decoder process over a pipe into a buffer of a certain size.
	The methodology has seen quite a lot of painful work and thinking going into it; this is _the_ part were I actually realized that modifying mixplayd wouln'd do it and I needed a fully multithreaded mixer for effective parralel... waiting for input.

	Crucial parts in the loop are:
		- select() with a small timeout on the pipe to see if there is data for reading
		- of course read() to get data (partially); with error check
		- sched_yield() to give cpu time to the decoder to fill the pipe (before next select()/read())
		- looping until
			a) got all data we wanted
			b) didn't get all data but an end flag was set (idea: we know from somethere else that the track finished decoding; one cannot see that on the pipe since it will be reused for next track)
			c) read()/select() error occured

	Optimization: One could monitor the number of runs the loop needs to fill the buffer normally and tune the select() timeout or some accordingly to minimize this number... probably pack this code into a class that implements this tuning dynamically as learning in itself? This may be complete nonsense... just a thought (for actually be in better sync with the decoder one may have to introduce a sleep of some s).
*/


unsigned long int just_read(int fd, char* buffy, unsigned long int bytes_wanted, volatile bool* endboolA, volatile bool* endboolB, volatile bool* intbool, volatile int& state)
{
	xdebug("just_read(%i, %p, %lu, %p, %p, %i)", fd, (void*)buffy, bytes_wanted, (void*) endboolA, (void*) endboolB, state);

	//preparation for select()
	struct timeval tv;
	fd_set fds;
	tv.tv_sec = 0;
	tv.tv_usec = 0;
	long int i;
	
	//we count down from bytes_wanted to zero
	char* endpoint = buffy + bytes_wanted;
	unsigned long int bytes_to_read = bytes_wanted;
	
	//the fancy loop 
	
	do
	{
		xdebug("[%i] loop near end: %s paused: %s bytes_to_read %lu", fd, ((*endboolA) ? "yes" : "no"), ((*endboolB) ? "yes" : "no"), bytes_to_read);

		FD_ZERO(&fds);
		FD_SET(fd, &fds);
		tv.tv_sec = 0;
		tv.tv_usec = 1000; //giv'em some time... but not too much (one could tune this one)!

		i = select(fd + 1, &fds , NULL, NULL, &tv); //something there?

		if(i > 0) //yes
		{
			i = read(fd, endpoint - bytes_to_read, bytes_to_read); //then read it

			xdebug("[%i] read %li bytes", fd, i);

			if(i > 0) bytes_to_read -= i; //good: we got some bytes; decrease the numbers of bytes still wanted
			else //bad: select indiacted that there is data, but there isn't
			{
				if(i == 0)
				{
					error("[%i] EOF occured - this must be some kind of error since we are talking pipes here!", fd);
					state = -1;
					break;
				}
				else
				{
					error("[%i] read error %i (EAGAIN: %i, EBADF:%i, EINTR:%i, EIO:%i)", fd, errno, EAGAIN, EBADF, EINTR, EIO);
					if(errno == EAGAIN) error("[%i] it's EAGAIN, just trying again", fd);
					else
					{
						state = -1;
						break;
					}
				}
			}
		}
		else if(i == -1) //no, nothing there, plus an error!
		{
			error("[%i] select error", fd);
			state = -1; break;
		}
		else xdebug("[%i]: nothing there", fd);

		//let the other wnd of the pipe do sth. and look then if we can get some more
		//if(bytes_to_read > 0)
		//{
		//	sched_yield();
		//	sched_yield(); //two sched_yield()s kindof arbitrary
		//}
	}
	while( (bytes_to_read > 0) && !( (i == 0) && ( (*endboolA) || (*endboolB) || (*intbool) ) && (state = 1) ) ); //I really mean state = 1 here!!!
	return(bytes_wanted - bytes_to_read);
}
