/*
	in_mpg123: mpg123 input device; the initial reason for DerMixD to exist at all
	
	part of DerMixD
	(c)2004-8 Thomas Orgis, licensed under GPLv2
	
	Much sweat and lost sleep has gone into this code...
	I should abstract more of this and try to adapt the techniques to other decoders.
	Actually, this has been the source of quite some generalized functionality already.
*/

#include <pthread.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <signal.h>
#include "common.hxx"
#include "comparse.hxx"
#include "in_mpg123.hxx"

//action codes
#define SEEK 0

#define JUMP 1
#define LOAD 2
#define PAUSE 3
#define UNPAUSE 4 
#define SEQ 5
#define INIT 9
#define LOADPAUSED 13
#define HALT 11

//error codes
#define SEEK_OUT_OF_RANGE 19
#define NO_SEEKBACK 23
#define THE_END 100 // an issuer get this when mpg123 encountered the end before responding
#define NOT_ONLINE 101

#define NOT_MPG123_THOR 20

#define ME "mpg123_input"
#define MEF "mpg123_file"

const string mpg123_input::t = "mpeg";

void* backwatch(void *p); //the thread listening for mpg123's responses

//big fat init
mpg123_input::mpg123_input(input_data* chd)
{
	construct(ME);
	data = chd;
	type = &t;
	problem = "";
	listener_there = false;
	decoder_pid = 0;
	build(chd, &mf); /* Creates the reader thread, too. */
	if(!on_line) return;

	unsigned int buffact = (unsigned int)(param.as_double("mpg123","prebuffer")*data->format.rate);

	if(buffact == 0) problem = "I need _some_ prebuffer (your choice evaluated to zero samples)";
	else
	{
		//spawn stream here...

		string decoder;
		param.get("mpg123","decoder",decoder);

		if(!decoder.empty())
		{

			int strm[2];
			int dec_stdin[2], dec_err[2];
			int pid;

			//create pipes
			debug("[%i] creating pipes", data->id);

			pipe(dec_stdin);
			pipe(strm);
			pipe(dec_err);

			pid=fork();
			if (pid == 0) //decoder process
			{
				debug("[%i] forked", data->id);

				//close the other side
				close(dec_stdin[1]);
				close(strm[0]);
				close(dec_err[0]);

				//duplicate for getting connected
				dup2(dec_stdin[0], STDIN_FILENO);
				dup2(strm[1],STDOUT_FILENO);
				dup2(dec_err[1], STDERR_FILENO);

				debug("[%i] now really starting decoder...", data->id);
				string gapless = param.as_bool("mpg123","gapless") ? "--gapless" : "";
				if(param.as_int("mpg123","nice") == 0)
				{
					if(audio_channels == 2)
					execlp(decoder.c_str(),decoder.c_str(),"-q","-s","-R","--remote-err", gapless.c_str(), "-",NULL); //normal stereo/mono mode
					else
					execlp(decoder.c_str(),decoder.c_str(),"-q","-s","-m","-R","--remote-err", gapless.c_str(), "-",NULL); //downmix in mpg123 (faster? at least less data in pipe)
				}
				else
				{
					string nice_level;
					param.get("mpg123","nice",nice_level);
					if(audio_channels == 2)
					execlp("nice","nice","-n",nice_level.c_str(),decoder.c_str(),"-q","-s","-R","--remote-err", gapless.c_str(), "-",NULL); //normal stereo/mono mode
					else
					execlp("nice","nice","-n",nice_level.c_str(),decoder.c_str(),"-q","-s","-m","-R","--remote-err", gapless.c_str(), "-",NULL); //downmix in mpg123 (faster? at least less data in pipe)
				}
				error("[%i] Start of decoder failed... You should stop me and check your setup.", data->id);
				exit(1);
			}

			//back again at main process

			debug("[%i] forker", data->id);

			//only work when the decoder process is still there

			int retval = 0;
			sched_yield();

			if( waitpid(pid, &retval, WNOHANG) == 0 )
			{
				decoder_pid=pid;

				//I want non-blocking read... was this really disabled before?
				fcntl(strm[0], F_SETFL, O_NONBLOCK);
				mf.audio_fd = strm[0];
				close(dec_stdin[0]);
				close(strm[1]);
				close(dec_err[1]);

				fcntl(dec_stdin[1], F_SETFD, FD_CLOEXEC); //close on exec (after forking)
				fcntl(strm[0], F_SETFD, FD_CLOEXEC); //close on exec (after forking)
				fcntl(dec_err[0], F_SETFD, FD_CLOEXEC); //close on exec (after forking)

				mf.status  = fdopen(dec_err[0],"r"); /* the responses or errors */
				mf.control = fdopen(dec_stdin[1],"a"); /* here we can dump orders */

				setlinebuf(mf.control);
				setlinebuf(mf.status);

				debug("[%i] spawned stream (running pid %u)", data->id, decoder_pid);

				fprintf(mf.control, "SILENCE\n");

				//clumsy setup of backwatch environment
				bd.mf = &mf;
				bd.data = data;

				//data->format must be initialized!
				mf.prebuffer = new audio_fifo(strm[0], data->format.channels, buffact, &data->near_end, &mf.decoder_paused);
				if(!mf.prebuffer || !mf.prebuffer->reader_there)
				{
					problem = "prebuffer failed to initialize";
					return;
				}
				add_thread(mf.prebuffer->rthread, "mpg123 fifo reader", mf.prebuffer);
				if(param.as_bool("mpg123", "zeroscan")) mf.prebuffer->set_zero(param.as_int("mpg123","zerolevel"), param.as_uint("mpg123","zerorange"));
				//going to wait for hello from mpg123
				comm_data initcd;
				mf.waiters.add(&initcd, INIT);

				//Finally, the crew is arriving!
				pthread_create(&listener, NULL, backwatch, (void*) &bd);
				add_thread(listener, "mpg123 backwatch", &bd);
				listener_there = true;
				//now wait for decoder getting ready
				debug("[%i] waiting for decoder init... I may starve here...", data->id);
				safe_sem_wait(initcd.waitress);

				//for what error should I check?
				if(initcd.errcode == 0) mf.on_line = true;
				else{ on_line = false; mf.on_line = false; }
				if(on_line)
				{
					debug("[%i] luck, it is there!", data->id);
					mf.waiters.add(&initcd, SEQ);
					fprintf(mf.control, "SEQ %f %f %f\n", data->bass, data->mid, data->treble);
					safe_sem_wait(initcd.waitress);
					debug("[%i] first eq setting succeded.", data->id);
				}
				else
				{
					error("[%i] failed to start (the correct) decoder, I'm wasted memory now.", data->id);
					problem = "don't like / cannot start the decoder program " + decoder + " because " + (initcd.errcode == NOT_MPG123_THOR ? "it has not the -thor control interface" : "of some reason...");
				}
			}
			else
			{
				error("[%i] failed to start decoder: retval (waitpid status) %i. I'm a zombie object until you rip me...", data->id, retval);
				problem = "failed to start decoder: " + decoder;
			}
		}
		else problem = "empty decoder command";
	}
}

mpg123_input::~mpg123_input()
{
	destbegin(ME);
	destroy(); // Do not forget the mandatory pendant to build() in the constructor!
	if(decoder_pid > 0)
	{
		if(listener_there)
		{
			pthread_cancel(listener);
			pthread_join(listener, NULL);
		}
		if(mf.prebuffer != NULL) delete mf.prebuffer;
		close(mf.audio_fd);
		fclose(mf.control);
		fclose(mf.status);
		// I _could_ kindly ask mpg123 to quit instead...
		kill (decoder_pid,15);
		waitpid(decoder_pid, NULL, 0); // check?
	}
	destend(ME);
}

mpg123_file::mpg123_file()
{
	construct(MEF);
	decoder_paused  = false;
	decoder_stopped = false;
	new_pos         = false;
	eq_pending      = false;
	new_eq          = false;
	on_line         = false;
	audio_fd = 0;
	status  = NULL;
	control = NULL;
	prebuffer = NULL;
	zeroscan = false;
	eq_val.resize(3);
	eq_val[0] = eq_val[1] = eq_val[2] = 0;
}

mpg123_file::~mpg123_file()
{
	destbegin(MEF);
	// Nothing to do... prebuffer is handled in mpg123_input.
}

// Just keep a note for worker tread to apply setting on next play.
bool mpg123_file::do_eq(const vector<double> &val)
{
	if(val.size()==3)
	{
		eq_val = val;
		new_eq = true;
		return true;
	}
	else return false;
}

void mpg123_file::let_me_die(unsigned char code, const string reason)
{
	if(on_line)
	{
		errcode = code;
		errtext = reason;
		on_line = false;
	}
}

///Only issue seq setting when there is going to be some reading action on the pipe!
void mpg123_file::check_eq()
{
	//If we have new eq settings, we'll apply them if we are sure that previous settings got their answers.
	if(new_eq && !eq_pending && on_line)
	{
		eq_pending = true;
		new_eq = false;
		fprintf(control, "SEQ %f %f %f\n", eq_val[0], eq_val[1], eq_val[2]);
	}
}

bool mpg123_file::do_read(audio_buffer *b, unsigned int wanted_samples)
{
	check_eq();
	while( wanted_samples )
	{
		on_line = prebuffer->gimme(*b,wanted_samples);
		if(!on_line) break;
#ifdef XDEBUG
		prebuffer->status();
#endif
		if(prebuffer->readstate == -1)
		{
			let_me_die(16, "read error on refill");
			break;
		}
		if(prebuffer->got_end && prebuffer->fill == 0) break;
	}
	return on_line;
}

bool mpg123_file::pause()
{
	if(!on_line) return false;

	if(decoder_paused) waiters.add(&cd, UNPAUSE);
	else waiters.add(&cd, PAUSE);

	fprintf(control, "PAUSE\n");
	safe_sem_wait(cd.waitress);
	if(cd.errcode)
	{
		serror("(un)pause decoding!");
		let_me_die(20, "could not issue pause");
		return false;
	}
	else return true;
}

bool mpg123_file::load(input_data *data)
{
	if(!on_line) return false;

	errcode = 0;
	bool pausing = !decoder_paused; 
	if(pausing)
	{
		debug("[inch %i] load: issuing a pause", data->id);
		waiters.add(&cd, HALT);
		fprintf(control, "PAUSE\n");
		sched_yield();
		sched_yield();
	}
	// now empty the pipe
	// prebuffer will read until it doesn't get anything _and_ decoder is either stopped or paused
	prebuffer->clear();
	prebuffer->start();
	prebuffer->drain();

	if(pausing) safe_sem_wait(cd.waitress); //shouldn't take long...

	debug("[inch %i] load: here again... with at_end %i", data->id, (int)data->at_end);
	if(cd.errcode == THE_END)
	serror("I got surprised by the end of the track while pausing the decoder, well, it's stopped now anyway. No problem...");
	else if(cd.errcode)
	{
		error("pause command got error %i", cd.errcode);
		let_me_die(20, "cannot pause for load");
	}
	if(on_line)
	{
		waiters.finalize(99); // If anyone was waiting, it was waiting for the old track.
		check_eq();
		waiters.add(&cd, LOADPAUSED);
		fprintf(control, "LP %s\n",data->filename.c_str());
		safe_sem_wait(cd.waitress);
		debug("[inch %i] load: load action finished... at_end now %i (This MUST be 0 now!)", data->id, (int)data->at_end);
		if(cd.errcode)
		{
			// That's a bit harsh... I should be able to recover from this situation now.
			let_me_die(21, "error loading stream/file");
			error("[inch %i] load: error on load (code %i)! Bye.", data->id, (int) cd.errcode);
		}
	}
	if(on_line)
	{
		prebuffer->clear(); // hm... clearing stops it...
		debug("[readily %i] load: finished; at_end: %i got_end: %i", data->id, (int)data->at_end, (int)prebuffer->got_end);
	}
	return on_line;
}

// I expect a new format on the first open() of a track.
// An automatic reload during seek is not expected to change format for prebuffer!
bool mpg123_file::do_open(input_data *data)
{
	if(!on_line) return false;

	load(data);
	// We need to catch stream info, which comes on first played frame.
	waiters.add(&cd, LOAD);
	pause(); // un-pause
	safe_sem_wait(cd.waitress);
	if(cd.errcode)
	{
		// That's a bit harsh... I should be able to recover from this situation now.
		let_me_die(23, "error starting stream/file");
		error("[inch %i] open: error on starting (code %i)! Bye.", data->id, (int) cd.errcode);
	}
	else
	{
		// number of channels may have changed
		if(prebuffer->resize(data->format.channels, prebuffer->size) == 0)
		{
			error("[inch %i] load: OUT OF MEMORY??? Cannot resize pebuffer! Bye.", data->id);
			let_me_die(25,"loader prebuffer out of mem");
		}
	}
	prebuffer->fresh();
	prebuffer->start(); // only on success?
	return on_line; // Success of loading is coupled with overall success for now.
}

bool mpg123_file::do_seek(input_data *data, unsigned long aim)
{
	if(!on_line) return false;

	// nothing bad, really... just one of my favourite Metallica songs;-)
	debug("[inch %i] seek: Searching... seek and destroy!", data->id);
	errcode = 0;
	unsigned long oldpos = data->position;
	// check if our big prebuffer has the wanted position handy already
	if(aim >= oldpos && aim < (oldpos + prebuffer->fill))
	{
		debug("[inch %i] seek: Yay! In-buffer seek!", data->id);
		unsigned int off = prebuffer->skip(aim-oldpos);
		// off could be != 0 if the backzeroscan chewed samples off the fill
		data->position = aim-off;
		return true;
	}

	// ok, we will have to do some real seeking
	// this _will_ return; perhaps after reading some bits from pipe
	unsigned int skip = prebuffer->get_zeroskip();

	bool pausing = !decoder_paused; 
	if(pausing)
	{
		waiters.add(&cd, PAUSE);
		fprintf(control, "PAUSE\n");
		debug("[inch %i] seek: pausing for clearance", data->id);
		//this is probably senseless here
		sched_yield();
		sched_yield();
	}

	debug("[inch %i] seek: cleaning pipe", data->id);
	// now empty the pipe
	// the pausing should get effective sometime in between
	// even if decoder stops while waiting for pause, drain shall get to the end
	prebuffer->clear();
	prebuffer->start();
	prebuffer->drain();

	if(pausing) safe_sem_wait(cd.waitress); //shouldn't take long...

	check_eq();
	// now the decoder can be stopped already... just reload track and start from beginning
	// this catches THE_END error for the pause
	if(decoder_stopped)
	{
		debug("[readily %i] seek: decoder stopped, reloading track", data->id);
		load(data);
	}

	if(!on_line)
	{
		error("[inch %i] seek: error on waiting for pause or reloading!", data->id);
		return false;
	}

	// decoder _is_ paused now, position is undefined, pipe is clean
	// so jump to a frame, taking the saved zeroskip into account
	waiters.add(&cd, JUMP);
	fprintf(control, "JUMP %ld\n",(aim+skip) / data->specials[0]);
	debug("[inch %i] seek: sent jump", data->id);
	sched_yield();
	safe_sem_wait(cd.waitress);

	if(cd.errcode)
	{
		error("[inch %i] seek: error %i from jump -- this really sucks!", data->id, cd.errcode);
		let_me_die(30,"some seek error");
		return false;
	}

	unsigned long decoder_pos = cd.retval_ulong * data->specials[0];
	debug("[inch %i] seek: restarting decoding", data->id);
	if(!pause()) return false;

	prebuffer->clear();
	prebuffer->start();
	// What about accurate seek? New mpg123 can do that itself, even.
	if(decoder_pos < skip)
	{
		prebuffer->skip(skip - decoder_pos);
		// I really cannot handle skip() not fully skipping here... I'd have to set negative position
		data->position = decoder_pos;
	}
	else data->position = decoder_pos - skip;

	return true;
}

bool mpg123_file::do_length(input_data *data)
{
	if(!on_line) return false;

	unsigned long oldpos = data->position;
	check_eq();
	// see how much data we get ...
	data->length   = data->position + prebuffer->drain();
	data->position = data->length;
	if(!do_seek(data, oldpos))
	{
		let_me_die(40, "seekback failed");
		return false;
	}
	else return true;
}

/*
	the thread that listens to the decoder's stderr
	sets stream variables and recognizes track playback state / errors
	does waiting for finishing of operations such as equalizer setting
*/
void* backwatch(void *p)
{
	backdata* bd = (backdata*) p;
	waiter_group *waiters = &bd->mf->waiters;

	int msglen = 0;
	int i;

	int fd = fileno(bd->mf->status);

	char buf[MSG_BUF_SIZE];
	int pos = 0;
	int fill = 0;
	buf[MSG_BUF_SIZE-1] = '\0';
	bool heardnothing = true;
	bd->thread_workout(2);

	while(1)
	{
		pthread_testcancel();
		bd->msg = "";
		msglen = 0;
		//fetch msg
		while(1)
		{
			pthread_testcancel();
			//cout << "fill: " << fill << endl;
			if(fill)
			{
				//search for next line end
				for(i=pos;i<fill;++i)
				{
					if(buf[i] == '\n'){ buf[i] = '\0'; break; }
					if(buf[i] == '\r'){ buf[i] = '\0'; }
				}
				//cout << "i: " << i << endl;
				if( (msglen += i - pos) > MAX_MSG_LENGTH )
				{
					error("msglen of %i is greater than %i", msglen, MAX_MSG_LENGTH);
					fill = 0;
					pos = 0;
					continue;
				}
				//copy line starting at pos and ending at i
				bd->msg += (buf + pos);
				if(i < fill) //really found a \n
				{
					//the line was the buffer: i is last character
					if(i == fill-1)
					{
						fill = 0;
						pos = 0;
					}
					//there is sth. left: i is smaller
					else
					{
						pos = i + 1;
					}
					break; //done, interpret line
				}
				//not at end: reading sth. next time...
				else{ fill = 0; pos = 0; } //pos=0 here or a few lines below?
			}
			else
			{
				pos = 0;
				pthread_testcancel();
				fill = read(fd, buf, MSG_BUF_SIZE-1);
				pthread_testcancel();
				if(fill < 1)
				{
					//better nothing than bad command
					//On such an error I must somehow get the decoder up again... most likely by destroying this damaged instance at all and later creating a new one... But how to tell? Mixer must do that.
					//if(fill < 0){ 
					fill=0; bd->msg = "";
					error("[backwatch %i] on reading from mpg123 stderr (%i)", bd->data->id, fd);
					//let me die...
					bd->data->at_end = true;
					bd->mf->on_line = false;
					waiters->finalize(THE_END); // And finish all waiters!!!
					while(1){	sleep(100); } // Give the others time to kill me properly
				}
			}
		}

		if(bd->msg[0] == '[')
		debug("[backwatch %i] got debugging message: %s", bd->data->id, bd->msg.c_str());
		else
		if(bd->msg != "")
		{
			debug("[backwatch %i] got string from decoder: %s", bd->data->id, bd->msg.c_str());
			string command;
			commando(bd->msg,command);
			vector<string> toks;

			if(heardnothing) //the first message...
			{
				heardnothing = false;
				if(bd->msg.substr(0,16) == "@R MPG123 (ThOr)")
				{
					waiters->finalize(INIT, 0);
					bd->mf->decoder_paused  = true;
					bd->mf->decoder_stopped = true;
				}
				else
				{
					error("[backwatch %i] Didn't get expected version string - most probably you have a wrong mpg123 version there! I absolutely need a version that behaves like 0.59r-thor.", bd->data->id);
					waiters->finalize(INIT, NOT_MPG123_THOR);
				}
			} 
			else if(command == "@P")
			{
				if(argo(bd->msg,toks,1))
				{
					switch(atoi(toks[0].c_str()))
					{
						case 1:
							bd->data->at_end = false; //sense?
							debug("[backwatch %i] paused... having set at_end to %i", bd->data->id, (int)bd->data->at_end);
							bd->data->near_end = false;
							bd->mf->decoder_paused  = true;
							bd->mf->decoder_stopped = false;
							//when someone wanted pause, he got it
							waiters->finalize(PAUSE, 0);
							waiters->finalize(HALT, 0);
							waiters->finalize(LOADPAUSED, 0);
						break;
						case 2:
							bd->mf->decoder_paused  = false;
							bd->mf->decoder_stopped = false;
							bd->data->at_end = false;
							debug("[backwatch %i] playback started again... having set at_end to %i", bd->data->id, (int)bd->data->at_end);
							bd->data->near_end = false;
							waiters->finalize(UNPAUSE, 0); //formerly LOAD was served here, too
						break;
						case 0:
							bd->data->near_end = true;
							bd->mf->decoder_paused  = true;
							bd->mf->decoder_stopped = true;
							//say goodbye to all pending waitresses
							//all standard error 100 for now...
							waiters->finalize(HALT, 0);
							if(waiters->finalize(waiter_group::anyone, THE_END))
							sdebug("some waiters got surprised by sudden end");
						break;
					}
					
				}
			}
			else if(command == "@J")
			{
				if(argo(bd->msg,toks,1)) waiters->finalize(JUMP, 0, 0, strtoul(toks[0].c_str(), NULL, 10));
				else waiters->finalize(JUMP, 1, 0, 0);
			}
			else if(command == "@bass:")
			{
				// There's danger here if you mix up attended and unattended EQ!
				if(bd->mf->eq_pending) bd->mf->eq_pending = false;
				else waiters->finalize(SEQ, 0);
			}
			else if(command == "@S")
			{
				if(argo(bd->msg,toks,12)) //@S 1.0 3 44100 Joint-Stereo 2 417 2 0 0 0 128 0 ... I only want "1.0"; later I'll be interested in stereo/mono, too
				{
					debug("[backwatch %i] got stream info: MPEG version %s layer %s", bd->data->id, toks[0].c_str(),toks[1].c_str());
					if(bd->data->specials.size() < 1) bd->data->specials.push_back(0);
					bd->data->specials[0] = 0; // we only trust mpg123 on this
					//     MPEG 1  MPEG 2 (LSF)  MPEG 2.5 (LSF)
					// I   384     384           384
					// II  1152    1152          1152
					// III 1152    576           576
					if(toks[0] == "1.0" || toks[0] == "2.0" || toks[0] == "2.5")
					{
						if(toks[1] == "1") bd->data->specials[0] = 384;
						else if(toks[1] == "2") bd->data->specials[0] = 1152;
						else if(toks[1] == "3")
						{
							if(toks[0] == "1.0") bd->data->specials[0] = 1152;
							else bd->data->specials[0] = 576;
						}
					}
					if(bd->data->specials[0] == 0)
					{
						debug("[backwatch %i] Encountered unknown MPEG format, letting this load fail (I would do bogus seeks)", bd->data->id);
						waiters->finalize(LOAD, 1); //error 1: unsupported format (probably);
					}
					else
					{
						bd->data->format.rate = strtoul(toks[2].c_str(), NULL, 10);
						bd->data->format.channels = (audio_channels == 2) ? ((unsigned int) strtoul(toks[6].c_str(), NULL, 10)) : 1;
						debug("[backwatch %i] stream data now: layer %s, %luHz, %u channels, %lu MP3SamplesPerFrame", bd->data->id, toks[1].c_str(), bd->data->format.rate, bd->data->format.channels, bd->data->specials[0]);
						waiters->finalize(LOAD, 0); //success
					}
				}
				else waiters->finalize(LOAD, 10); //some unknown weird stuff?
			}
			debug("[backwatch %i] string interpretation done", bd->data->id);
		}
	}

	pthread_exit(NULL);
}
