/*
	audio_fifo: audio data FIFO buffer on a file (pipe)
	
	part of DerMixD
	(c)2004-8 Thomas Orgis, licensed under GPLv2
	
	This code is only tested with two channels... but it should work with any channel count.
	Only basic restriction is that input channels == output channels; No fancy operations here!
	
	This class now features the background reading thread that makes such a FIFO really work as a buffer.
	Initially designed for having some space to do the zeroscan in, this thing now actually makes sense in the parallel world of DerMixD and hopefully helps preventing any unnecessary skip even durnig heavy system/disk load as long as the basic priorities (nice levels) are set accordingly.
	
	Usage:
	
	fd = open(...);
	audio_fifo* ff = new audio_fifo(fd, channels, size, ...);
	audio_buffer target(channels, chunk_size);
 	ff->start()	
	while(in_the_mood)
	{
		target.fill=0
		unsigned int wanted_samples = chunk_size;
		while(wanted_samples)
		{
			ff->gimme(target, wanted_samples);
		}
		do_with_audio(target);
	}
	ff->stop()
	
	
	AAAARRRRRG! the reader just doesn't get readstate=1 at end for some reason!!!
	That's sometimes on the latptop, reproducable with the alpha (miss tinitus, ifthat matters)
*/


#include "common.hxx"
#include "piperead.hxx"
#include "audio_buffer.hxx"
#include "audio_fifo.hxx"
#include "param.hxx"
#include "secsleep.hxx"
#include <sys/time.h>

void* fifo_reader(void* p)
{
	audio_fifo* af = (audio_fifo*) p;
	af->thread_workout(1);
	while(1)
	{

		xdebug("[%i] fifo_reader waiting", af->fd);
		sem_post(&af->wsem); // notify that I am waiting now
		safe_sem_wait(&af->rsem);
		xdebug("[%i] fifo_reader summoned", af->fd);
		sem_zero(&af->rsem); // make sure that we awake only once
		sem_zero(&af->wsem); // make sure the waiting sem is cleared to indicate non-waiting

		// trying to do it in smaller chunks so that we have the possibility to switch threads in between
		// seems to work for extremely big buffers on my laptop, but not when doing the big fill while crossfading (2 channels playing)
		// this event seems able to trigger a situation only resolved by pausing and letting the buffer fill before restarting playback
		// how is this on other boxen (with NPTL?)
		af->readstate = 0;
		while(!af->got_end && af->readstate == 0)
		{
			// do I trust the memory deallocation here?
			struct timeval tv;
			fd_set fds;
			int i;
			/*
				FIFO fun part 3

				Make it real ... FIFO

				We have a fill, we want an aimed_fill, fitting this into size.
			*/

			//wait for data
			FD_ZERO(&fds);
			FD_SET(af->fd, &fds);
			tv.tv_sec = 0;
			tv.tv_usec = af->wait_usec; //giv'em some time... but not too much (one could tune this one)!
			if(af->reader_pause)
			{
				if(af->zeroscan && !af->front_scanned)
				{
					if(af->fill >= af->zero_range)
					{
						af->frontzeroscan();
						debug("[%i] fifo_reader pausing on demand after scan", af->fd);
						break;
					}
					else debug("[%i] fifo_reader: No, Mommy, I want some more time to scan!", af->fd);
				}
				else
				{
					xdebug("[%i] fifo_reader pausing on demand", af->fd);
					break;
				}
			}
			i = select(af->fd + 1, &fds , NULL, NULL, &tv); //something there?
			if(i > 0) //yes, shove the stuff in
			{
				//read what I want/get in one go, max 16K
				int freespace = (af->size - af->fill) < 16384/(sizeof(audio_type)*af->channels) ?
				                (af->size - af->fill) :
				                16384/(sizeof(audio_type)*af->channels);
				unsigned int wpos = af->readerpos + af->fill;
				if(wpos >= af->size) wpos -= af->size; // wrap the write pointer
				if(wpos+freespace > af->size) freespace = af->size-wpos; // only read up to end
				
				xdebug("[%i] reading %u samples (%u/%u/%u)", af->fd, freespace, af->fill, af->aimed_fill, af->size);
				if(freespace > 0)
				{
					ssize_t read_bytes = read(af->fd, (char*) (const_cast<audio_type*> (af->ptr + af->channels*wpos)), freespace*sizeof(audio_type)*af->channels);
					if(read_bytes > 0)
					{
						unsigned int read_samples = read_bytes/(sizeof(audio_type)*af->channels);
						af->fill += read_samples;
						xdebug("[%i] read state %i, fill: %u / aim: %u", af->fd, af->readstate, af->fill, af->aimed_fill);
					}
					else
					{
						error("[%i] read error %i (EAGAIN: %i, EBADF:%i, EINTR:%i, EIO:%i)!", af->fd, errno, EAGAIN, EBADF, EINTR, EIO);
						if(errno == EAGAIN) error("[%i] it's EAGAIN, so hopefully temporary", af->fd);
						else af->readstate = -1;
					}
				}
			}
			else if(i == 0) // just nothing there yet
			{
				if(*af->endboolA || *af->endboolB)
				{
					debug("[%i] ******* got end! *******", af->fd);
					af->got_end = true;
					// do scan, but only on first-class end
					if(af->zeroscan && *af->endboolA)
					{
						sdebug("initiating zeroscans");
						if(!af->front_scanned) af->frontzeroscan();
						af->backzeroscan();
					}
					break;
				}
			}
			else //no, nothing there, plus an error!
			{
				error("[%i] select error (errno: %i)!", af->fd, errno);
				af->readstate = -1; break;
			}

			if(af->fill >= af->aimed_fill)
			{
				if(af->zeroscan && !af->front_scanned)
				{
					if(af->fill >= af->zero_range) af->frontzeroscan();
					// else: do another loop to read enough data to do zeroscan
				}
				else break;
			}
		}
		xdebug("[%i] fifo_reader done for now, readstate=%i, got_end=%i", af->fd, af->readstate, (int)af->got_end);
	}
	pthread_exit(NULL);
}


//the audio_fifo is an audio_buffer connected with a data source (file descripor, usually for a pipe) and FIFO access wrapped around...
audio_fifo::audio_fifo(int file, unsigned int c, unsigned int s, volatile bool* ebA, volatile bool* ebB) : audio_buffer(c,s)
{
	construct("audio_fifo");
	endboolA = ebA;
	endboolB = ebB;
	reader_pause = false;
	readerpos = 0;
	zeroscan = false;
	front_scanned = true;
	zero_level = 0;
	zero_range = 3000;
	skipped = 0;
	got_end = false;
	fd = file;
	reader_there = false;
	sem_there = false;
	aimed_fill = size;
	readstate = 0;
	active = false;
	if(size == 0)
	{
		serror("won't create a zero-sized fifo buffer");
		return;
	}
	wait_usec = param.as_int("fifo","wait");
	if(wait_usec < 1)
	{
		error("negative wait timeout given (%i)!", wait_usec);
		return;
	}
	if((sem_init(&rsem, 0 , 0) == 0)  && (sem_init(&wsem, 0 , 0) == 0))
	{
		sem_there = true;
		if(pthread_create(&rthread, NULL, fifo_reader, (void*) this) == 0)
		reader_there = true;
		else
		serror("with starting reader thread for audio FIFO!");
	}
}

audio_fifo::~audio_fifo()
{
	destbegin("audio_fifo");
	if(reader_there)
	{
		sdebug("going to stop");
		stop();
		sdebug("going to cancel reader");
		if(sem_cancel(rthread,&rsem) != 0) serror("error cancelling writer thread / thread not there");
		else pthread_join(rthread, NULL);
	}
	sdebug("reader thread gone (or never was there)");
	if(sem_there)
	{
		sem_destroy(&rsem);
		sem_destroy(&wsem);
	}
	destend("audio_fifo");
}

bool audio_fifo::waiting()
{
	int w;
	sem_getvalue(&wsem, &w);
	return (w > 0);
}

void audio_fifo::stop()
{
	if(!waiting())
	{
		xdebug("[%i] stop: suspending thread", fd);
		reader_pause = true;
		safe_sem_wait(&wsem); // wait for thread to see the flag and react
		sem_post(&wsem); // put the value back
		reader_pause = false;
	}
	else xdebug("[%i] stop: thread already waiting", fd);
}

void audio_fifo::start()
{
	if(waiting())
	{
		xdebug("[%i] start: summoning thread", fd);
		sem_post(&rsem);
	}
	else xdebug("[%i] start: thread already working", fd);
	active = true;
}

unsigned int audio_fifo::get_zeroskip()
{
	if(!zeroscan) return 0;
	if(front_scanned) return skipped;
	else
	{
		// the reader still needs to do it
		aimed_fill = zero_range;
		// if we didn't start the reader yet...
		if(waiting())
		{
			// one run to get the scan
			start();
			stop();
		}
		else
		{
			// tell it to finish as soon as frontzeroscan is done, then continue
			stop();
			start();
		}
		aimed_fill = size;
		if(!front_scanned) error("[%i] I really should have the scan by now!", fd);
		return skipped;
	}
}

//Go back to start and don't take the 4000 Mark!
void audio_fifo::clear()
{
	debug("[%i] clear", fd);
	stop();
	readerpos = 0;
	fill = 0;
	active=false;
	got_end = false;
}

void audio_fifo::fresh()
{
	if(zeroscan) front_scanned = false;
	else front_scanned = true;
	skipped = 0;
}

/* resizing analog to audio_buffer */

unsigned int audio_fifo::resize(unsigned int s)
{
	clear();
	return aimed_fill = reallocate(s);
}

unsigned int audio_fifo::resize(unsigned int c, unsigned int s)
{
	clear();
	return aimed_fill = reallocate(c,s);
}

unsigned int audio_fifo::skip(const unsigned int skip_samples)
{
	unsigned int rest_samples = skip_samples;
	gimme(skipper, rest_samples);
	return rest_samples; // possibly non-zero, not enough data there to ignore
}

unsigned long audio_fifo::drain()
{
	debug("[%i] draining", fd);
	unsigned long count = 0;
	if(active)
	{
		while(!(fill == 0 && got_end))
		{
			sched_yield();
			count += 4096-skip(4096); // skip() is the key action here!
		}
		active = false;
	}
	return count;
}

/*
	gimme() copy wanted_samples audio samples from (filled) fifo buffer into target
	If you don't get enough samples though there is no end reached, just try again later...

	FIFO fun part 1
*/

bool audio_fifo::gimme(audio_buffer& target, unsigned int& wanted_samples)
{
	if(active)
	{
		// A bit of safety...
		bool skipping = &target == &skipper;
		if(!skipping && wanted_samples*channels > (target.size-target.fill)*target.channels)
		{
			error("gimme: trying to stuff %u*%u samples into space of %u*%u!", wanted_samples, channels, (target.size-target.fill), target.channels);
			return false;
		}
		// that should be avoided by external forces!
		// give the reader time 
		// it's a dilemma: I want the reader not to mess with time critical threads but sometimes, when the buffer is empty, I _have_ to wait for it
		// hm, what about read-only check of fill and flags?
		unsigned int gonna_get = size > wanted_samples ? wanted_samples : size;
		aimed_fill = gonna_get;
		if(readstate == 0 && !got_end && fill < gonna_get)
		{
			debug("[%i] fill: %u / aim %u", fd, fill, aimed_fill);
			while(readstate == 0 && !got_end && fill < gonna_get)
			{
				xdebug("[%i] gimme waiting (fill: %u, need %u ... reader_pause=%i)", fd, fill, gonna_get, (int)reader_pause);
				//secsleep(1);
				sched_yield();
			}
		}
		// interrupt reader, going to mess with the buffer
		stop();
		gonna_get = fill > wanted_samples ? wanted_samples : fill;
		// we may have to wrap
		// so, in general there are two steps in reading
		// first from readerpos up to the end
		// second from the new readerpos up to the fill
		// first one only when we have to wrap... so let's check that first
		if(readerpos+gonna_get >= size) // if exactly up to the edge or over it
		{
			// first read up to the edge
			unsigned int gonna_get_now = size-readerpos;
			if(!skipping) // only if we have a real taker
			memcpy((void*) (const_cast<audio_type*> (target.ptr + target.fill * channels)),
		      	 (void*) (const_cast<audio_type*> (ptr + channels * readerpos)),
		      	 gonna_get_now * channels * sizeof(audio_type));
			target.fill += gonna_get_now;
			fill -= gonna_get_now;
			wanted_samples -= gonna_get_now;
			gonna_get -= gonna_get_now;
			readerpos = 0;
		}
		// now simply read from readerpos on... the numbers are settled
		if(gonna_get)
		{
			if(!skipping) // only if we have a real taker
			memcpy((void*) (const_cast<audio_type*> (target.ptr + target.fill * channels)),
		      	 (void*) (const_cast<audio_type*> (ptr + channels * readerpos)),
		      	 gonna_get * channels * sizeof(audio_type));
			target.fill += gonna_get;
			fill -= gonna_get;
			wanted_samples -= gonna_get;
			readerpos += gonna_get;
		}
		aimed_fill = size;
		if(!got_end) start();

		return true;
	}
	else
	{
		error("[%i] We didn't get end yet and still the reader thread is not active, scary!", fd);
		return false;
	}
}

/*
	frontzeroscan() and backzeroscan()
	
	trying to dismiss some data that we consider to be leading/trailing silence by advancing the reader pointer and/or decreasing the fill
	
	FIFO fun part 2
*/

// This works only once!!
// Make sure nobody else touches the buffer (ensured when inside fifo_reader, since everybody else calls stop() before)
void audio_fifo::frontzeroscan()
{
	skipped = 0;
	unsigned int realrange = zero_range;
	int my_zero = zero_level; // drop the volatility
	unsigned int my_pos = 0;
	// if we got less, we have less
	if(fill < realrange) realrange = fill;
	//search for non-zero in buffer (non-zero meaning absolute value being above a threshold)
	unsigned int rawoff = 0;
	for(; rawoff < realrange*channels; ++rawoff)
	{
		xdebug("front: abs(ptr[%u]) = %i > %i?", rawoff, abs(ptr[rawoff]), my_zero);
		if(abs(ptr[rawoff]) > my_zero) break;
	}
	my_pos = rawoff/channels;
	if(my_pos == realrange) my_pos = 0; //scan not successful

	fill -= my_pos;
	if(!got_end) start();
	readerpos = skipped = my_pos;
	debug("[%i] frontzeroscan: skipped %u samples", fd, skipped);
	front_scanned = true;
}

/*
	after having done this here, a following refill will most likely do crappy stuff... do a clear() in between!
	this really still valid? What bad thing can happen?
					debug("[%i] gimme waiting (fill: %u, need %u ... reader_pause=%i)", fd, fill, gonna_get, (int)reader_pause);
we had wrap; not anymore (zeroes)
	possible problem: normally we don't have free space before _and_ after readerpos! One never knows...
	here also: zero with threshold

	Make sure nobody else touches the buffer (ensured when inside fifo_reader, since everybody else calls stop() before)
*/
void audio_fifo::backzeroscan()
{
	//In principle I want to be larger that uint... not possible on a 32bit system and strict ISO C++
	long int i;
	unsigned int my_pos = readerpos;
	int my_zero = zero_level; // drop the volatility
	unsigned int my_fill = fill;
	//beware of order and type range!
	long int preend = ((long int) my_pos + my_fill - 1 - size) * channels; //position (0 ... size-1) of last sample if we have wrap, negative otherwise
	if(preend > -1) //we have a wrap, end stuff is _before_ my_pos
	{
		//this should be the left part
		for(i = preend; i != -1; --i) if( abs(ptr[i]) > my_zero ) break;
		
		if(i > -1) my_fill -= (preend - i)/channels; //i>=0 -> partial substraction of the left part
		//all was zero, so look on the right side, at least preend+1 samples are zero now
		else
		{
			for(i = (long int) channels * size - 1; i >= (long int) (channels * my_pos); --i)
			{
				xdebug("back: abs(ptr[%li]) = %i > %i?", i, abs(ptr[i]), my_zero);
				if(abs(ptr[i]) > my_zero) break;
			}
			if(i < (long int) (channels * my_pos) ) my_fill = 0; // worst case: nothing at all
			else my_fill = (unsigned int) (i/channels + 1 - my_pos);
		}
	}
	else //just look in the right part
	{
		for(i = (long int) channels * (my_pos + my_fill) - 1; i >= (long int) (channels * my_pos); --i)
		{
			xdebug("back: abs(ptr[%li]) = %i > %i?", i, abs(ptr[i]), my_zero);
			if(abs(ptr[i]) > my_zero) break;
		}
		if(i < (long int) (channels * my_pos) ) my_fill = 0; // worst case: nothing at all
		else
		{
			debug("my_fill = %li/%i + 1 - %u", i, channels, my_pos);
			my_fill = (unsigned int) (i/channels + 1 - my_pos);
		}
	}
	debug("[%i] backzeroscan: %u of %u remaining", fd, my_fill, fill);
	fill = my_fill;
}


//blah...
void audio_fifo::status()
{
	fprintf(stderr, "[audio_fifo %i at %p, size %u samples with %i channels, fill: %u pos: %u, got end: %i\n",
	        fd, (void*) ptr, size, channels, fill, readerpos, got_end);
}

void audio_fifo::set_zero(int zero, unsigned int range)
{
	debug("activating zeroscan with level %i and range %u", zero, range);
	zeroscan = true;
	zero_level = abs(zero);
	// ensure that we stay inside the buffer
	zero_range = range > size ? size : range;
	fresh();
}
