#part of ThOrMA (but not necessarily depending upon)
#DerMixD::Control: interface to dermixd with some added logic

#Copyright (c) 2005-2014 Thomas Orgis (thomas@orgis.org)
#This is Open Source, Distribution and Modification under the terms of the Artistic License
#see the file LICENSE in the ThOrMA package root or at a place some steps upwards directory-wise if you got this file in a separate package


package DerMixD::Control;

use strict;
use DerMixD::Base;
use List::Util qw(reduce);

our @ISA = ('DerMixD::Base');
my $me = 'DerMixD::Control';

sub new
{
	my $class = shift;
	my $self = $class->SUPER::new(@_);
	# Add macro action for load and play.
	$self->{macroact} = ['play'];
	$self->{macrohelp} = {play=>'combination of load with given arguments and subsequent start on an input channel'};
	$self->{channelact}{play} = [{pos=>1, map=>$self->{inchannelnum}}];
	return $self;
}

# Plain function, call as DerMixD::Control::params().
sub params
{
	return DerMixD::Base::params(@_);
}

# input: command array, possibly with channel name instead of number
# output: replacing channel name in-place with ID
# TODO: ensure we always got a current channel list (via pushed updates).
sub map_channel
{
	my $self = shift;
	my $coms = shift;

	return unless @{$coms};

	# Parse help output for given command to find position of channel number.
	# We treat input and output channels without distinction.
	# The channel map further down
	my $entry = $self->{channelact}{$coms->[0]};
	unless(defined $entry)
	{
		$self->{channelact}{$coms->[0]} = [];
		$entry = $self->{channelact}{$coms->[0]};
		my $help = $self->Command(['help',$coms->[0]]);
		if($help->[0] =~ /^$coms->[0]\s*(.*); \[(.+)\]:/)
		{
			my $pos = 1;
			my $flags = $2;
			my $pars = $1;
			my $map = undef;
			$map = $self->{inchannelnum}  if $flags =~ / in /;
			$map = $self->{outchannelnum} if $flags =~ / out /;
			while($pars =~ /^\s*([^\(\)]+)\(([^\(\)]+)\)(.*)$/)
			{
				$pars = $3;
				my $name = $1;
				my $type = $2;
				# There could be multiple channel specifications, in or out.
				if($name eq 'inchannel')
				{
					push(@{$entry}, {pos=>$pos, map=>$self->{inchannelnum}});
				}
				elsif($name eq 'outchannel')
				{
					push(@{$entry}, {pos=>$pos, map=>$self->{outchannelnum}});
				}
				elsif($name eq 'channel') # Just the one and only channel
				{
					push(@{$entry}, {pos=>$pos, map=>$map});
					last;
				}
				++$pos;
			}
		}
	}

	return unless defined $entry;

	for(@{$entry})
	{
		my $pos = $_->{pos};
		my $map = $_->{map};
		unless($coms->[$pos] =~ /^\d+$/)
		{
			print STDERR "$me trying to translate channel argument at position $pos with map $map\n"
				if $self->{debug};
			my $num = $map->{$coms->[$pos]};
			if(defined $num)
			{
				print STDERR "$me mapped number: $num\n" if $self->{debug};
				$coms->[$pos] = $num;
			}
			else
			{
				print STDERR "Warning: Failed to map channel name $coms->[$pos] to number for action $coms->[0].\n";
			}
		}
	}
}

sub Control
{
	#example: $pal->Control([seek,20])
	my $self = shift;
	my $coms = shift;
	my $ret = '';
	#the intelligent part of control is the insertion of active channel when needed...
	#needed when: single start/pause
	#or seek/load/play/bass/mid/treble/speed <arg>
	#or eq <arg1> <arg2> <arg3>
	my $ac;
	my $channelquest;
	my $numberquest;
	if($#{$coms} == 0) #really single command or just channel missing
	{
		if(grep($_ eq $coms->[0], ('pause','start')))
		{
			if(defined ($ac = $self->ActiveChannel()))
			{
				#more convenient, but not illogical
				$coms->[0] = 'start' if($coms->[0] eq 'pause' and $ac->{status} =~ 'pause');
				#just append channel number
				$coms->[1] = $ac->{num};
			}
			else{ $channelquest = 1; }
		}
	}
	elsif($#{$coms} == 1 or ($coms->[0] eq 'eq' and $#{$coms} == 3)) #insert channel
	{
		if(grep($_ eq $coms->[0], ('seek','rseek','eq','bass','mid','treble','speed','pitch','load','play','vol','volume')))
		{
			if(defined ($ac = $self->ActiveChannel()))
			{
				splice(@{$coms}, 1, 0, $ac->{num}); #put channel in place, interpret numbers later
			}
			else{ $channelquest = 1; }
		}
	}
	#mangle numbers
	if($#{$coms} == 2 and grep($_ eq $coms->[0],('seek','bass','mid','treble','speed','volume','vol')))
	{
		#interpret numbers
		if($coms->[0] eq 'seek')
		{
			$numberquest = 1 unless defined ($coms->[2] = $self->AbsoluteNumber($coms->[2],$ac->{position},$ac->{length}));
		}
		else
		{
			if($coms->[0] eq 'vol'){ $coms->[2] /= 100 unless $coms->[2] =~ '%'; $coms->[0] = 'volume'; }
			$numberquest = 1 unless defined ($coms->[2] = $self->AbsoluteNumber($coms->[2],$ac->{$coms->[0]},$ac->{$coms->[0]}));
		}
	}
	$self->map_channel($coms) unless $channelquest;

	return $channelquest ? ['error: What is the active channel?'] : ($numberquest ? ['error: Unable to interpret number!'] : $self->ControlCommand($coms));
}

# Wrap over macro commands.
# Also, hack to get current status after certain commands.
# In future, we should get overall notifications for that.
sub ControlCommand
{
	my $self = shift;
	my $coms = shift;
	my $ret = [];

	if($coms->[0] eq 'play')
	{
		$coms->[0] = 'load';
		$ret = $self->Command($coms);
		$self->Command(['start', $coms->[1]]);
	}
	elsif($coms->[0] eq 'help' and grep {$_ eq $coms->[1]} @{$self->{macroact}})
	{
		$ret = [$self->{macrohelp}{$coms->[1]}];
	}
	else{ $ret = $self->Command($coms); }

	# Be aware of changed channels.
	$self->FullStatus()
		if grep {$_ eq $coms->[0]} (qw(addin addout remin remout));
	return $ret;
}

#guess the channel that is the current active one
#again with some respect for apparent fading...
#still not smooth...
sub ActiveChannel
{
	my $self = shift;
	$self->FullStatus();
	if(@{$self->{in}})
	{
		my $actor = $self->{in}->[0]; #default is first channel
		#find all channels that have a file loaded
		if(my @candidates = grep( ($_->{file} ne '' and $_->{length} > 0) , @{$self->{in}}))
		{
			#now find players or if pausing one if there are none...
			my @list = grep( $_->{status} =~ 'play', @candidates); #playing and seek-play
			my $fading = $#list > 0 ? 1 : 0;
			@list = grep( $_->{status} =~ 'pause', @candidates) unless @list; #paused and seek-pause
			@list = grep( $_->{status} eq 'stopped', @candidates) unless @list; #...
			$actor = reduce { $b->{position}/$b->{length} < $a->{position}/$a->{length} ? $b : $a } @list ;
			$actor->{fading} = $fading;
		}
		return $actor;
	}
	return undef;
}

sub AbsoluteNumber
{
	my ($self, $rel, $now, $ref) = @_;
	my $num = undef;
	if($rel =~ /^([+-]{0,1})(\d*\.\d+|\d+)(%{0,1})$/)
	{
		if($3){ $num = $2/100*$ref; } #percents
		else{ $num = $2; }  #normal units
		if($1) #offset, relative
		{
			my $factor = ($1 eq '+') ? 1 : -1;
			$num = $now + $factor*$num;
		}
	}
	elsif($rel =~ /^([+-]{0,1})(\d+):((\d*\.\d+|\d+)|(\d+):(\d*\.\d+|\d+))$/)
	{
		if($6 ne ''){ $num = $2*3600+$5*60+$6; }
		else{ $num = $2*60+$4; }
		if($1) #offset, relative
		{
			my $factor = ($1 eq '+') ? 1 : -1;
			$num = $now + $factor*$num;
		}
	}
	return $num;
}

sub Wait
{
	my $self = shift;
	my $arg = shift;
	return 0 unless($#{$arg} >= 1);
	$arg->[2] = 'wakeup' unless defined $arg->[2];
	my $res = $self->Command(['script',$arg->[0],$arg->[1],'say',$arg->[2]]);
	return $res->[0] =~ /^\s*error\:/ ? 0 : $self->WaitForSay($arg->[2]);
}

return 1;
