#part of ThOrMA (but not necessarily depending upon)
#DerMixD::Base: basic interface to dermixd

#Copyright (c) 2005-2014 Thomas Orgis (thomas@orgis.org)
#This is Open Source, Distribution and Modification under the terms of the Artistic License
#see the file LICENSE in the ThOrMA package root or at a place some steps upwards directory-wise if you got this file in a separate package

# This used to be "the bad" basic module... it used old-style Socket with old-style filehandle...
# That's not good when opening multiple connections... I strongly suppose so.
# But: The code using Socket and plain handle compiles way faster than IO::Socket stuff.
# On a Pentium-M Banias at 600Mhz I see user time of `dermixd-control id` at around 0.27s using IO::Socket compared to 0.14 with plain handle.
# That makes one-shot usage of dermixd-control quite inconvenient.
# Guess for one-shot usage I'd better write a dedicated C client anyway.

package DerMixD::Base;

use strict;
use IO::Socket;

our %long =
(
  'f', 'fullstat'
, 'S', 'shutdown'
, 'p', 'pause'
, 's', 'start'
, 'l', 'load'
, 'c', 'close'
);

our %default =
(
	'remote', 'localhost',
	'port', '8888',
	'socket', '/tmp/dermixd.socket',
	'dermixd-debug', 0, # Prevent all-to-easy name clash.
	'echo', 0
);

# Plain function, call as DerMixD::Base::params().
sub params
{
	my $ref = shift;
	my @p = 
	(
		 'socket', $default{socket}, 'k','dermixd: UNIX domain socket to use instead of TCP port when connecting to localhost (fallback to TCP if not there)',
		,'port',   $default{port},   'p','dermixd: port number to connect to (where dermixd is listening)'
		,'remote', $default{remote}, 'R','dermixd: the (remote) machine to connect to'
		,'echo',   $default{echo},   'e','dermixd: echo server responses verbatim to STDOUT (without this you will never see "[saying] Hello")'
		,'dermixd-debug', $default{'dermixd-debug'}, '', 'dermixd: debugging'
	);
	if(defined $ref)
	{
		push(@{$ref}, @p);
	}
	else
	{
		return @p;
	}
}

#hack for my old OSF1 Perl:
#don't like to give up quotes!
my $q = '\Q';
my $e = '\E';
if(`uname` =~ /^OSF1/){ $q = ''; $e = ''; }
#damn! that never works! ... There is something wrong with my understanding of regex building here
$q = ''; $e = '';

my $me = '[DerMixD::Base]';
#talk with some dermixd-1.0+ 
my $major = 1;
my $miniminor = 0;

sub new
{
	my $class = shift;
	my $config = shift; #remote, <host>, port, <port>, ..
	my $self = {};
	for(keys %default)
	{
		$self->{$_} = defined $config->{$_} ? $config->{$_} : $default{$_};
	}
	$self->{debug} = $self->{'dermixd-debug'};
	$self->{online} = 0;
	$self->{events} = []; # array, of ['update', $inchannel ] or ['peer', $from, $to, $message ] or ['say', $message]
	bless $self, $class;
	$self->Open();
	return $self;
}

#gets full status into internal structured hash
sub FullStatus
{
	my $self = shift;
	@{$self->{in}} = ();
	@{$self->{out}} = ();
	%{$self->{inchannelnum}} = ();
	%{$self->{outchannelnum}} = ();
	for(@{$self->Command(['fullstat'])})
	{
		#in 0:Lena -> (0) stopped at 0s/150.152s (buf: 0) vol: 1 speed: 1 eq: 1|1|1, dev: mp3 file: /home/thomas-data/thorma/var/music/phantoms_of_futur
		#in 1:Elena -> (0) idle at 0s/0s (buf: 0) vol: 1 speed: 1 eq: 1|1|1
		#out 0:Helena <- (0,1) playing dev: oss file: /dev/dsp
			if(/^in\s+(\d+):(\S+)\s*->\s*(\([\d,]*\))\s+(\S+)\s+at\s+([\d\.]+)s\/([\d\.]+)s\s+\([^\(]+\)\s+vol:\s*([\d\.]+)\s+speed:\s*([\d\.]+)\s+eq:\s*([\d\.]+)\|([\d\.]+)\|([\d\.]+)\s*(\,\s*following\s+(\d+:[^,]*)|)\s*(\,\s*(\d+)\s*actions|)\s*(\,?\s*dev:\s*(\S+)\s+file:\s*(.*)|)$/)
		{
			next if $1 < 0;
			$self->{in}->[$1] = 
			{
				'num', $1,
				'nickname', $2,
				'status', $4,
				'position', $5,
				'length', $6,
				'volume', $7,
				'vol', $7*100,
				'speed', $8,
				'bass', $9,
				'mid', $10,
				'treble', $11,
				'actions', defined $15 ? 1*$15 : 0
			};
			# Store mapping from nickname to channel ID. First one rules.
			$self->{inchannelnum}{$2} = $1
				unless defined $self->{inchannelnum}{$2};
			#targets from $3 (better reference list?)
			@{$self->{in}->[$1]->{targets}} = split(',', $3);
			
			if($16)
			{
				$self->{in}->[$1]->{device} = $17;
				$self->{in}->[$1]->{file} = $18;
			}
		}
		elsif(/^out\s+(\d+):(\S+)\s*<-\s*(\([\d,]*\))\s+(\S+)(\s+dev:\s*(\S+)\s+file:\s*(.*)|)$/)
		{
			next if $1 < 0;
			$self->{out}->[$1] = 
			{
				'num', $1,
				'nickname', $2,
				'status', $4
			};
			# Store mapping from nickname to channel ID. First one rules.
			$self->{outchannelnum}{$2} = $1
				unless defined $self->{outchannelnum}{$2};
			#sources from $3 (better reference list?)
			@{$self->{out}->[$1]->{sources}} = split(',', $3);
			if($5)
			{
				$self->{out}->[$1]->{device} = $6;
				$self->{out}->[$1]->{file} = $7;
			}
		}
		else{ print STDERR "$me not understanding this fullstat line: $_\n"; }
	}
}

# Just because $line = <$self->{handle}> doesn't work!
sub receive_line
{
	my $self = shift;
	my $sockhandle = $self->{handle};
	my $line = <$sockhandle>;
	return $line;
}

# get one atomic response or event block
sub pull_atom
{
	my $self = shift;
	my %atom = ('command'=>'', 'sub'=>'', 'messages'=>[]);
	my $in = 0;
	while(defined (my $line = $self->receive_line()))
	{
		print $line if $self->{echo};
		$line =~ s/\r*\n*$//; #really chomp!
		unless($in)
		{
			if($line =~ /^\[([\w\-]+)(|\/[\w\-]+|\s.+)\]\s*(.*)$/)
			{
				$atom{command} = $1;
				$atom{sub} = $2;
				my $mess = $3;
				if(length($mess) > 0 and substr($mess,0,1) eq '+')
				{ # it is the beginning of some block
					$in = 1;
				}
				else
				{ # it is a one-line message/response
					$atom{messages} = [$mess];
					return \%atom;
				}
			}
			else
			{
				print STDERR "$me non-understood line (perhaps old-style [say]?): $line\n";
			}
		}
		else
		{ # There is either stuff part of a block or the end.
			if($line =~ /^\[\Q$atom{command}\E(|\/.+|\s.+)\]\s*-/)
			{
				return \%atom;
			}
			else{ push(@{$atom{messages}}, $line); }
		}
	}
	# When we did not leave the loop via return, there is trouble reading from the socket.
	print STDERR "$me Trouble reading from socket?\n";
	return undef;
}

# issue a command and wait for / parse response
# between issuing the command and getting the answer, there could be pushed info from dermixd
# watch out for channel status updates and messages
sub Command
{
	my $self = shift;
	my $com = shift;
	my @l; #answer lines
	$com->[0] = $long{$com->[0]} if defined $long{$com->[0]};
	my $atom = undef;
	my $in = 0;

	print STDERR "$me >@{$com}\n" if $self->{debug};
	print {$self->{handle}} "@{$com}\n";

	while(defined ($atom = $self->pull_atom()))
	{
		print STDERR "got atom with command $atom->{command}\n" if $self->{debug};
		# A close command response is always the end, also if not asked for.
		# Otherwise, we wait until the right response arrived.
		if($atom->{command} eq $com->[0]){ last; }
		elsif($atom->{command} eq 'close')
		{
			$self->{online} = 0;
			last;
		}
		else
		{	# could be event
			$self->handle_event($atom);
			print STDERR "$me still waiting for $com->[0]\n" if $self->{debug};
		}
	}
	if($com->[0] eq 'shutdown' or $com->[0] eq 'close'){ $self->{online} = 0; }

	return [] unless defined $atom;

	#unless($l){die "Somehow the command $com did not get a response... I'm scared. ($!)\n";}
	print STDERR "$me <".join("\n$me <",@{$atom->{messages}})."\n" if $self->{debug};

	return $atom->{messages};
}

sub dump_messages
{
	my $self = shift;
	while(defined (my $line = $self->receive_line()))
	{
		chomp($line);
		print $line,"\n";
	}
	$self->{online} = 0;
}

#wait for a specific string as produced by a timed say <string> command
sub WaitForSay
{
	my $self = shift;
	my $word = shift;
	return 0 unless $self->{online};

	while(defined (my $atom = $self->pull_atom()))
	{
		if($atom->{command} eq 'saying' and $atom->{messages}[0] eq $word)
		{
			return 1;
		}
		else
		{
			$self->handle_event($atom);
		}
	}

	return 0;
}

# Just waits for the next line(s) from dermixd... interprets channel status updates
# ...also receives peer messages...
sub WaitForSomething
{
	my $self = shift;

	# Old news is still news.
	if(@{$self->{events}}){ return 1; }

	# This is the new news.
	if($self->{online})
	{
		my $a = $self->pull_atom();
		if(defined $a)
		{
			$self->handle_event($a);
		}
		else
		{
			return 0; # got trouble
		}
	}
	return 1; # got something
}

#array, of ['update', $inchannel ] or ['peer', $from, $to, $message ] or ['say', $message]
sub handle_event
{
	my $self = shift;
	my $atom = shift;
	if($atom->{command} =~ /ch(\d+)/)
	{
		# input channel update
		my $ch = $1;
		# When it is a new channel, I should have known... well, guess I should just trigger a fullstat when something fresh comes.
		$self->{in}[$ch] = {} unless defined $self->{in}[$ch];
		my $m = $atom->{messages}[0];
		if($m =~ /(\d+)s,\s+(\S+)/)
		{	# playback info, time and status
			$self->{in}[$ch]{status} = $2;
			$self->{in}[$ch]{position} = $1;
		}
		push(@{$self->{events}}, [ 'update', $ch ]);
	}
	elsif($atom->{command} eq 'peer')
	{
		my ($from, $to) = split(/\s+/, $atom->{sub});
		push(@{$self->{events}}, [ 'peer', $from, $to, $atom->{messages}[0] ]);
	}
	elsif($atom->{command} eq 'saying')
	{
		push(@{$self->{events}}, [ 'say', $atom->{messages}[0] ]);
	}
}

#Make connection to the daemon
#attention: I think $self->{handle} is treated as class handle here... so no several connections at once, thus OpenSocket closes an existing connection before opening new...(?)
sub Open
{
	my $self = shift;
	
	$self->Close();
	#get the network on
	if($self->{remote} eq 'localhost' and -S $self->{socket})
	{
		print STDERR "$me want to connect to socket $self->{socket}\n" if $self->{debug};
		$self->{handle} = new IO::Socket('Domain' => PF_UNIX, 'Type' => SOCK_STREAM, 'Peer' => $self->{socket});
	}
	else
	{
		print STDERR "$me want to connect to host $self->{remote}:$self->{port}\n" if $self->{debug};
		$self->{handle} = new IO::Socket( 'Domain' => PF_INET, 'Type' => SOCK_STREAM
		                                 ,'Proto' => 'tcp', 'PeerHost' => $self->{remote}
		                                 ,'PeerPort' => $self->{port} );
	}
	unless(defined $self->{handle} and defined $self->{handle}->connected()){ $self->{handle} = undef; return 0; }
	# I hear that autoflush is default nowadays... still:
	my $oldhandle = select($self->{handle}); $| = 1; select($oldhandle);
	
	#read and interpret greeting
	my $greeting = $self->receive_line();
	if($greeting =~ /^\[connect\]\s*DerMixD\s*v(\d+)\.(\d+)/)
	{
		$self->{interface} = "$1.$2";
		$self->{major} = $1;
		$self->{minor} = $2;
		# For the future: Check behaviour with newer DerMixD; there are some API changes in 2.x .
		if($self->{major} >= $major)
		{
			if($self->{major} > $major or $self->{minor} >= $miniminor)
			{
				$self->{online} = 1;
				print STDERR "$me connected to host $self->{remote}\n" if $self->{debug};
				$self->FullStatus();
			}
			else{	print STDERR "$me need at least DerMixD v$major.$miniminor!\n"; }
		}
		else{	print STDERR "$me need DerMixD v$major.x interface (not $self->{interface})!\n"; }
	}
	else
	{
		print STDERR "$me want to know who's there\n" if $self->{debug};
	}
	return $self->{online};
}

#end communication with the current dermixd
sub Close
{
	my $self = shift;
	if($self->{online})
	{
		if(defined $self->{handle}) # For some reason that one is already undef on destruction??
		{
			print {$self->{handle}} "close\n";
			#to extreme?
			close ($self->{handle}) or die "$me close $!";
		}
		$self->{online} = 0;
	}
	$self->{handle} = undef;
}

sub DESTROY
{
	my $self = shift;
	$self->Close();
}

1;
