// A test that reads an input audio file and plays it on an audio output.
// Parameters: output type, list of input files.

#include "basics.hxx"
#include "in/input_file.hxx"
#include "out/output_file.hxx"
#include "param_init.hxx"
#include "file_type.hxx"

#include <string>
#include "tstring.hxx"

input::file_type typer;

void play_file(const std::string location, const std::string outtype);

int main(int argc, char **argv)
{
	for(int i=2; i<argc; ++i)
	{
		play_file(argv[i], argv[1]);
	}
	return 0;
}


void play_file(const std::string location, const std::string outtype)
{
	input_file *in = NULL;
	output_file *out = NULL;
	audio_buffer buf;

	fprintf(stderr, " ===== Playing: %s =====\n", location.c_str());
	fprintf(stderr, "Creating input file of matching type.\n");
	input::switch_file(in, typer.guess(location));

	if(in == NULL)
	{
		fprintf(stderr, "That did not work, for whatever reason.");
		return;
	}

	fprintf(stderr, "Opening input.\n");
	if(!in->do_open_safe(location))
	{
		std::string *msg = in->err.as_string();
		fprintf(stderr, "Failed: %s\n", msg->c_str());
		delete msg;
	}

	buf.reallocate(in->format.channels, 2048);

	fprintf(stderr, "Preparing general format for proper output (rate %lu, channels %u).\n", in->format.rate, in->format.channels);
	param("audio_rate") = "";
	strprintf(param("audio_rate"), "%lu", in->format.rate);
	param("channels") = "";
	strprintf(param("channels"), "%u", in->format.channels);

	output::type_id otype = output::name_to_id(outtype);
	if(!output::type_available(otype))
	{
		fprintf(stderr, "Desired output type is not available.\n");
		goto end;
	}

	output::switch_file(out, otype);

	if(out == NULL)
	{
		fprintf(stderr, "Failed to create output file of type %s.\n", output::id_to_name(otype));
		goto end;
	}

	if(!out->do_open_safe())
	{
		std::string *msg = out->err.as_string();		
		fprintf(stderr, "Cannot open default output: %s\n", msg->c_str());
		delete msg;
		goto end;
	}

	while(1)
	{
		buf.fill = 0;
		if(!in->buf_read(buf))
		{
			std::string *msg = in->err.as_string();
			fprintf(stderr, "Reading failed: %s\n.", msg->c_str());
			delete msg;
			break;
		}
		//fprintf(stderr, "Read %zu samples.\n", buf.fill);
		if(buf.fill == 0) break;

		if(!out->buf_write(buf))
		{
			std::string *msg = out->err.as_string();
			fprintf(stderr, "Trouble writing: %s\n", msg->c_str());
			delete msg;
			break;
		}
	}

end:
	if(in  != NULL) delete in;
	if(out != NULL) delete out;
}
