#include <cstdlib>
#include <cstdio>
#include <dlfcn.h>
#include <errno.h>
#include <ladspa.h>

using namespace std;

class module
{
private:
	void *dlhandle;
	int dlflags;
	const char* errmsg;
public:
	module(): dlhandle(NULL), dlflags(RTLD_NOW) {};
	~module();
	bool open(const char* name);
	virtual bool special_open();
	void close();
	virtual void special_close();
	void *get_symbol(const char* name);
	const char* error();
	bool check_error();
	void clear_error();
};

// a module, a set of modules... that has to be cleared up
class ladspa_module: public module
{
	private:
	// module handles... states...
	LADSPA_Descriptor_Function descr;

	public:
	ladspa_module(): descr(NULL) {};
	~ladspa_module(){};
	bool special_open();
	void special_close();
	void list();
};


module::~module()
{
	close();
}

bool module::open(const char* name)
{
	close();
	dlhandle = dlopen(name,dlflags);
	if(dlhandle == NULL)
	{
		check_error();
		fprintf(stderr, "error(%s) ", error());
		return false;
	}
	return special_open();
}

bool module::special_open(){ return true; }


void module::close()
{
	if(dlhandle != NULL)
	{
		special_close();
		dlclose(dlhandle);
	}
}

void module::special_close(){}

void* module::get_symbol(const char* name)
{
	void *location;
	clear_error();
	location = dlsym(dlhandle, name);
	if(check_error())
	{
		fprintf(stderr, "error(%s) ", error());
	}
	return location;
}

bool module::check_error()
{
	errmsg = dlerror();
	return (errmsg != NULL);
}

void module::clear_error()
{
	dlerror();
	errmsg = NULL;
}

const char* module::error()
{
	return errmsg != NULL ? errmsg : "no error";
}

// LADSPA specific stuff

bool ladspa_module::special_open()
{
	descr = (LADSPA_Descriptor_Function) get_symbol("ladspa_descriptor");
	if(descr == NULL)
	{
		fprintf(stderr, "error(ladspa_descriptor not found) ");
		return false;
	}
	return true;
}

void ladspa_module::list()
{
	if(descr == NULL) return;

	const LADSPA_Descriptor* mod;
	unsigned long idx = 0;
	while((mod = descr(idx++)) != NULL)
	{
		// I don't like it that there is no safety for the portdescripor array.
		// But then, I rely on a binary blob anyway.
		int iports = 0;
		int oports = 0;
		int cports = 0;
		for(int p=0; p<mod->PortCount; ++p)
		{
			LADSPA_PortDescriptor pd = mod->PortDescriptors[p];
			if(pd & LADSPA_PORT_CONTROL) ++cports;
			if(pd & LADSPA_PORT_INPUT)   ++iports;
			if(pd & LADSPA_PORT_OUTPUT)  ++oports;
		}
		printf("%lu: %s %i,%i,%i (%s)\n", mod->UniqueID, mod->Label, iports, oports, cports, mod->Name);
		printf("\tPorts:\n");
		for(int p=0; p<mod->PortCount; ++p)
		{
			LADSPA_PortDescriptor pd = mod->PortDescriptors[p];
			const char* name = mod->PortNames[p];
			const char* iotype = "?";
			const char* ctype  = "?";
			if(pd & LADSPA_PORT_INPUT)   iotype = "in";
			if(pd & LADSPA_PORT_OUTPUT)  iotype = "out";
			if(pd & LADSPA_PORT_AUDIO)    ctype = "audio";
			if(pd & LADSPA_PORT_CONTROL)  ctype = "ctl";

			printf("port %i: %s/%s \"%s\"\n", p, iotype, ctype, name);
		}
	}
}

void ladspa_module::special_close()
{
	descr = NULL;
}

int main(int argc, char** argv)
{
	ladspa_module m;
	for(int i=1; i<argc; ++i)
	{
		bool success;
		fprintf(stderr, "Accessing module %s: ", argv[i]);
		success = m.open(argv[i]);
		fprintf(stderr, "%s\n", success ? "success" : "failure");
		if(success) m.list();
	}
	return 0;
}
