
/*************************************************************************
 *
 * Copyright (C) 2005-2012,2013 by Inlab Software GmbH, Gruenwald, Germany
 *             All Rights Reserved / Alle Rechte vorbehalten
 *
 *    Use of and access to this source code requires either a written 
 *    permission of Inlab Software GmbH (source code license) or has 
 *    to be covered by a valid written contractual agreement. 
 * 
 * $RCSfile: bngagent.c,v $
 * $Revision: 3.13 $
 * $Date: 2013/01/03 21:10:39 $
 *
 *************************************************************************/

#include <stdio.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/termios.h>
#include <netinet/in.h>
#include <arpa/nameser.h>
#include <net/if.h>
#include <stdlib.h>
#include <unistd.h>
#include <resolv.h>
#include <fcntl.h>
#include <netdb.h>
#include <string.h>
#include <strings.h>
#include <signal.h>
#include <sysexits.h>
#include <syslog.h>

#define  MAXBYTES 64 
#define  MAXLINE  64 

static int foreground=0;
static int debugflag=0;
static int testflag=0;
static int ipv6=0;

#ifdef __hpux
#include <sys/pstat.h>

getloadavg (double *ave,int na) {
  struct pst_dynamic pstd;
  int i;

  if (pstat_getdynamic(&pstd,sizeof(struct pst_dynamic),(size_t)1,0)<0) {
    for (i=0; i<na; i++) {
      *(ave + i) = 0.0;
    }
    if(debugflag) {
      fprintf(stderr, "pstat_getdynamic() failed, assuming 0.0\n"); 
    }
    return (-1);
  }

  switch (na) {
    case 3:
      *(ave + 2) = pstd.psd_avg_15_min;
    case 2:
      *(ave + 1) = pstd.psd_avg_5_min;
    case 1:
      *(ave) = pstd.psd_avg_1_min;
    default:
      break;
  }
  return (3);
}
#else
int getloadavg(double*, int);
#endif

void usage() {
  fprintf(stderr,"\n");
  fprintf(stderr,"  $Revision: 3.13 $\n");
  fprintf(stderr,"  bngagent is an open source part of the BalanceNG product\n");
  fprintf(stderr,"  Copyright (C) 2005-2009,2010 by Inlab Software GmbH, Gruenwald, Germany\n");
  fprintf(stderr,"  All rights reserved - more infos at: http://www.BalanceNG.net\n");
  fprintf(stderr,"\n");
  fprintf(stderr,"  usage:\n");
  fprintf(stderr,"    server            bngagent <options>    port\n");
  fprintf(stderr,"    request (test)    bngagent <options> -r address port\n");
  fprintf(stderr,"  options:\n");
  fprintf(stderr,"    -0                return 1 minute load avg    (server,default)\n");
  fprintf(stderr,"    -1                return 5 minute load avg    (server)\n");
  fprintf(stderr,"    -2                return 15 minute load avg   (server)\n");
  fprintf(stderr,"    -6                use IPv6 instead of IPv4\n");
  fprintf(stderr,"    -f                stay in foreground          (server)\n");
  fprintf(stderr,"    -c <command>      specify command             (server)\n");
  fprintf(stderr,"    -d                enable debug and foreground (both)\n");
  fprintf(stderr,"    -t <targetid>     specify targetid            (request)\n");
  fprintf(stderr,"\n");
  exit(EX_USAGE);
}

static void daemonize() {
  int fd,childpid;

  if ((childpid = fork()) < 0) {
    fprintf(stderr,"cannot fork\n");
    exit(EX_OSERR);
  } else {
    if (childpid > 0) {
      exit(EX_OK); 
    }
  }
  #ifdef __hpux 
  (void) setsid();
  #else
  #ifdef BSD 
    setpgid(getpid(), 0);
  #else 
    setpgrp();
  #endif
  {
    int fd;
    if((fd = open("/dev/tty", O_RDWR)) != -1) {
      if(ioctl(fd, TIOCNOTTY) < 0) {
        fprintf(stderr, "error: cannot ioctl(IOCNOTTY)");
        exit(EX_OSERR);
      }
      close(fd);
    }
  }
  #endif

  openlog("bngagent", LOG_ODELAY | LOG_PID | LOG_CONS, LOG_DAEMON);

  if((fd=open("/dev/null", O_RDWR, 0)) < 0) {
    syslog(LOG_CRIT, "unable to open /dev/null");
    exit(EX_OSERR);
  }

  if(dup2(fd, 0) < 0)
    syslog(LOG_ERR, "Unable to set /dev/null as stdin");
  if(dup2(fd, 1) < 0)
    syslog(LOG_ERR, "Unable to set /dev/null as stdout");
  if(dup2(fd, 2) < 0)
    syslog(LOG_ERR, "Unable to set /dev/null as stderr");
  close(fd);

  if(chdir("/") < 0)
    syslog(LOG_ERR, "Unable to chadir to /");
}

static char*  	command=NULL;
static char*  	targetid=NULL;
static unsigned int sd;
static int      laindex=0;	

static void testrequest_timeout() {
  printf("*timeout*\n");
  exit(EX_OK);
}

static void perform_testrequest(char* destination, char* port) {
  int rc;
  static unsigned char buffer[MAXBYTES];
  unsigned char intbuffer[4];
  struct addrinfo hints;
  struct addrinfo *results;
  int status;

  socklen_t salen;
  struct sockaddr *sa;

  bzero(&hints, sizeof(hints));

  if(ipv6 == 1) {
    hints.ai_family = AF_INET6;
  } else {
    hints.ai_family = AF_INET;
  }

  hints.ai_socktype = SOCK_DGRAM;
  hints.ai_protocol = IPPROTO_UDP;

  status = getaddrinfo(destination, port, &hints, &results);

  if(status != 0) {
    fprintf(stderr,"error at getaddrinfo: %s\n", gai_strerror(status));
    fprintf(stderr,"exiting.\n");
    exit(EX_OSERR);
  }

  if(results == NULL) {
    fprintf(stderr,"no matching results at getaddrinfo\n");
    fprintf(stderr,"exiting.\n");
    exit(EX_OSERR);
  }

  do { 
    sd=socket(results->ai_family, results->ai_socktype, results->ai_protocol);
    if(sd >= 0)
      break; 
  } while ((results=results->ai_next) != NULL);

  if(sd == 0) {
    fprintf(stderr,"cannot create client socket\n");
    fprintf(stderr,"exiting.\n");
    exit(EX_OSERR);
  }

  sa=malloc(results->ai_addrlen);
  memcpy(sa, results->ai_addr, results->ai_addrlen);
  salen=results->ai_addrlen;

  rc=sendto(sd, buffer, sizeof(unsigned short), 0, sa, salen);
  if(rc < 0) { 
    perror("sendto");
    exit(EX_OSERR);
  }

  signal(SIGALRM,testrequest_timeout);
  alarm(1);
  rc = recvfrom(sd, buffer, sizeof(buffer), 0, NULL, NULL);

  if(rc == 6) {
    memcpy(intbuffer, buffer+sizeof(unsigned short), sizeof(unsigned int));
    printf("target id: %u value: %u\n", ntohs(*((unsigned short*) buffer)), 
           ntohl(*((unsigned int*) intbuffer)));  
  } else {
    printf("invalid length %d returned\n", rc);
  }
  exit(EX_OK);
}

int main(int argc, char** argv) {
  int c; 

  unsigned char buffer[MAXBYTES];
  unsigned char intbuffer[4];

  int sockfd;
  int n;
  socklen_t addrlen=0, len;
  struct sockaddr *cliaddr;
  char* port;
  struct addrinfo hints, *res, *ressave;

  while((c = getopt(argc,argv,"0126fdc:rt:")) != EOF) {
    switch(c) {
    case 'f':
      foreground=1;
      break;
    case 'd':
      debugflag=1;
      foreground=1;
      break;
    case 'c':
      command=strdup(optarg);
      break;
    case 't':
      targetid=strdup(optarg);
      break;
    case 'r':
      testflag=1;
      break;
    case '0':
      laindex=0;
      break;
    case '1':
      laindex=1;
      break;
    case '2':
      laindex=2;
      break;
    case '6':
      ipv6=1;
      break;
    case '?':
    default:
      usage();
    }
  }
  argc -= optind;
  argv += optind;

  if(testflag) {
    if(argc != 2) {
      usage();
    }     
    perform_testrequest(argv[0],argv[1]);
    exit(EX_OK);
  }

  if(argc != 1) {
    usage();
  }     

  port = argv[0];
 
  bzero(&hints, sizeof(struct addrinfo));
  hints.ai_flags=AI_PASSIVE;

  if(ipv6 == 1) {
    hints. ai_family=AF_INET6;
  } else {
    hints. ai_family=AF_INET;
  }

  hints. ai_socktype=SOCK_DGRAM;
  hints.ai_protocol=IPPROTO_UDP;

  if((n = getaddrinfo(NULL, port, &hints, &res)) !=0) {
    fprintf(stderr,"bngagent error for %s",port);
    exit(EX_OSERR);
  }

  ressave=res;

  do {
    sockfd=socket(res->ai_family, res->ai_socktype, res->ai_protocol);
    if (sockfd < 0)
      continue;
   
    if (bind(sockfd, res->ai_addr, res->ai_addrlen) == 0)
      break;
    
    close(sockfd);
    sockfd=0;

 } while ((res=res->ai_next)!= NULL); 
 
 if(res)
   addrlen=res->ai_addrlen;

 freeaddrinfo(ressave);
 cliaddr=malloc(addrlen);
 len=addrlen;

  if(sockfd<=0) {
    fprintf(stderr,"cannot create server socket\n");
    exit(EX_OSERR);
  }

  if(!foreground)
    daemonize();

  for(;;) {	
    unsigned int returnvalue;
    FILE *pipe;
    char line[MAXLINE];

    recvfrom(sockfd, buffer, sizeof(buffer), 0, cliaddr, &len);

    if(command != NULL) {
      if(debugflag) 
        fprintf(stderr,"executing script or command [%s]\n", command);
      if((pipe=popen(command,"r")) == NULL) {
        fprintf(stderr,"error popen()\n");
        exit(EX_OSERR);
      }
      if(fgets(line,MAXLINE,pipe) == NULL) {
        fprintf(stderr,"error fgets()\n"); 
        exit(EX_OSERR);
      }
      pclose(pipe);
      returnvalue=atoi(line);
    } else {
      double values[3];
      if(getloadavg(values, 3) != 3) {
	values[0]=0.0;
	values[1]=0.0;
	values[2]=0.0;
      }
      if(debugflag) {
	if(laindex==0)
          fprintf(stderr,"1-minute system-load: %f\n",values[0]);
	if(laindex==1)
          fprintf(stderr,"5-minute system-load: %f\n",values[1]);
	if(laindex==2)
          fprintf(stderr,"15-minute system-load: %f\n",values[2]);
      }
      if(values[laindex] < 0.0) 
	values[laindex]=0.0;
      returnvalue=(unsigned int) (values[laindex]*100.0);

      /* 0 would mean service unavailable, so we return at least 1 */
      if(returnvalue == 0) 
        returnvalue++; 
    }

    if(debugflag) 
      fprintf(stderr,"about to return %d\n", returnvalue);

    /* avoiding alignment problems */ 
    returnvalue = htonl(returnvalue);
    *((unsigned int*) intbuffer)=returnvalue;	

    memcpy(buffer+sizeof(unsigned short), intbuffer, sizeof(unsigned int));

    sendto(sockfd, buffer, sizeof(unsigned short)+sizeof(unsigned int), 0, 
           cliaddr, len);

  }
}

