//<c>   Copyright 1986-2000 by Gerry J. Danen; all rights reserved.

/*
<t>     File Dump

  Description   This program dumps a file in the DOS DEBUG format.  The output
                includes the byte counter in Hex for the first byte of each
                line, 16 bytes in Hex, the same 16 bytes in ASCII.
                Non-printable characters are shown as dots.  FDUMP will read
                hidden and system files also!

  Syntax        FDUMP filename [/p] [/o=outfile]
                Full pathnames are supported for the filename.

  Switches      /p      Pause at end of screen
                /o=     Redirect output to a disk file or to the printer.
*/
#include"gd_tools.h"
#include"gd_extrn.h"
#include"gd_kb.h"
#include<fcntl.h>
#include<stat.h>


#define pgVERS_ "V2.20"
#define pgTITL_ "File Dump"
#define pgCOPR_ "1986"


INT16   pause, lcnt, lmax ;
UCHAR   oFile[FS_LEN] ;
BOOL    out2file = FALSE ;
FILE    *oFh ;


void    NoSuchFile( STRING filename )
{
    NewScreen();
    fprintf( stderr, "\nFile %s not found\a\n", filename );
}


void    help( void )
{
    if ( vid.first_scr )
        NewScreen();
    fprintf( stderr, "\n\nUsage:\t  %s filename [/p] [/o=outfile]\n\n", get_global_program_name() );
    fprintf( stderr, "\t\tfilename - name of file to be dumped, full pathname supported\n" );
    fprintf( stderr, "\t\tswitches - /p = pause at bottom of screen\n" );
    fprintf( stderr, "\t\t\t   /o = direct output to file or printer\n\n" );
    fprintf( stderr, "\t  %s will read hidden and system files\n\n", get_global_program_name() );
    fprintf( stderr, "\t  Hit any key to pause screen output\n" );
    gd_exit( 1, "" );
}


void    CheckSwitch( STRING s )
{
    if ( strlen(s) == 0 )
        return;
    if (*s == SLASH)
    {
        switch (*(++s))
        {
            case 'p':
            case 'P':
                pause = TRUE ;
                break;
            case 'o':
            case 'O':
                out2file = TRUE ;
                s++ ;
                if ( *s == '=' )
                {
                    s++ ;
                    strcpy( oFile, s );
                    _strupr( oFile );
                }
                if ( ! strlen(oFile) )
                    strcpy( oFile, "ZZZ.ZZZ" );
                pause = FALSE ;
                break;
        }
    }
}


void    next_line( void )
{
    INT16       row, col ;

    if ( ! out2file )
    {
        LocateCursor( &row, &col );
        if ( row > (vid.rows-1) )
            sc_scroll( 1, vid.normal_attr, 3,1, vid.rows, vid.cols, 0 );
        else
            row += 1;
        MoveCursor( row, 1 );
    }
}


void    pause_it( void )
{
    if ( ! out2file )
    {
        HoldMsg( FALSE, FALSE );
        lcnt = 2;
    }
}


void    dump_line( INT16 l, STRING work, UINT32 bytecount )
{
    INT16       key, row, col ;
    UINT16      cnt ;
    UCHAR       ch, temp[ 17 ], msg[ STR_LEN ], pr_line[ STR_LEN ] ;

    if ( l != 0 )
    {
        CursOff();
        StrFill( temp, SPACE, 17 );             //  Initialize string
        if ( pause )
        {
            lcnt++;
            if ( lcnt > lmax )
                pause_it();
        }
        sprintf( pr_line, "%06X ", bytecount );
        for ( cnt = 0 ; cnt < (UINT16)l ; )
        {
            if ( cnt == 8 )
                st_cat( pr_line, " -", vid.cols+1 );
            ch = *work++;
            sprintf( msg, " %02X", ch );
            st_cat( pr_line, msg, vid.cols+1 );
            if ( (ch < 32) || (ch > 126) )
                temp[ cnt ] = '.' ;
            else
                temp[ cnt ] = ch ;
            cnt += 1 ;
        }
        for ( ; cnt < 16 ; cnt++ )
        {
            if ( cnt == 8 )
                st_cat( pr_line, "  ", vid.cols+1 );
            st_cat( pr_line, "   ", vid.cols+1 );
        }
        sprintf( msg, "%s  %s", pr_line, temp );
        if ( out2file )
        {
            fprintf( oFh, "%s %10lu\n", msg, bytecount );
        }
        else
        {
            LocateCursor( &row, &col );
            DspMessage( row, 2, vid.normal_fg, vid.normal_bg, "%s", msg );  // may contain formats!
            if ( vid.cols > 89 )
                DspMessage( row, 81, vid.normal_fg, vid.normal_bg, "%10lu", bytecount );
            next_line();
        }
        key = kb_getkey_nowait();
        if ( BreakPressed() )
            gd_exit( 1, "Aborted as requested" );
        if ( key != 0 )
            pause_it();
    }
}


INT16   read_16_bytes( FILE *iFh, UCHAR *s, INT16 *endfile )
/*
        Read 16 bytes from "iFh".

        Get string s from iFh, maximum 16 characters.
        When EOF is reached, endfile becomes TRUE.
        Does NOT flush characters if maxchar is reached.
*/
{
    INT16       len, maxchar=17 ;
    UCHAR       c;

    maxchar--;                          //  leave room for the EOS terminator
    for ( len=0 ; (len < maxchar) && (!feof(iFh)) ; len++ )
    {
        c = getc(iFh);
        *s++ = c;
    }
    *s = EOS;
    if ( feof( iFh ) )
    {
        len--;
        *endfile=TRUE;
    }
    else
        *endfile=FALSE;
    return (len);
}


void    main( int argc, char *argv[] )
{
    FILE        *iFh ;
    UCHAR       iFile[ MAX_PATH_LEN ], work[ STR_LEN ], msg[ STR_LEN ] ;
    INT16       l, i, endfile;
    UINT32      bytecount=0;

    fs_init( FALSE, SHAREWARE_VERSION, 0, pgVERS_, pgTITL_, pgCOPR_ );
    Allow_Break();

    lmax  = vid.rows - 2 ;
    lcnt  = 1 ;
    pause = FALSE ;
    oFile[0] = EOS ;

    if( argc < 2 )
        help();

    st_cpy( iFile, argv[1], 64 );
    _strupr( iFile );
    if ((iFh = fopen(iFile,"rb")) == NULL)
    {
        NoSuchFile(iFile);
        help();
    }

    argv++;
    argv++;
    for ( i=2 ; i < argc ; i++, argv++ )        // Loop for possible switches
    {
        if ( strlen(*argv) != 0 )
        {
            CheckSwitch( *argv );
        }
    }

    sprintf( msg, "Dump of %s", iFile );
    SetGlobalString( PGM_TITLE, msg );
    NewScreen();
    if ( out2file )
    {
        MoveCursor( 5, 1 );
        oFh = fl_create_or_replace_tty( oFh, oFile, "w", YES, YES );
        fprintf( oFh, "File: %s\n\n", iFile );
    }
    FOREVER
    {
        l = read_16_bytes( iFh, work, &endfile );
        dump_line( l, work, bytecount );
        bytecount = bytecount + l;
        if ( endfile )
        {
            DspChar( vid.err_fg, vid.err_bg, (UCHAR)'', vid.cols );
            next_line();
            sprintf( msg, " --Done--   %s   %lu bytes", iFile, bytecount );
            if ( out2file )
            {
                fprintf( stderr, "\n%s\n\n", msg );
            }
            else
            {
                PutString( msg );
                next_line();
                if ( bytecount > 9999L )
                    sn_beep();
                next_line();
                next_line();
                CursorUp( 2 );
            }
            _fcloseall();
            gd_exit( 0, "" );
        }
    }
}
