#!/usr/bin/perl
#
# $Id: subst.pl,v 1.1.2.3 2003/02/17 21:46:50 dyff Exp $
#
# Substitute directory names for FIDOGATE scripts
#

use strict;


if($#ARGV < 0) {
    print STDERR
      "usage:   subst.pl [-p] [-t TOPDIR] [-c CONFIG.MAKE] [-v] [-h] [-HC] source ...\n",
      "\n",
      "options:   -d              dump config parameters from config.make\n",
      "           -p              replace 1st line with \#!PERL\n",
      "           -o OPTS         PERL -OPTS\n",
      "           -c CONFIG.MAKE  use specified config.make\n",
      "           -i INCLUDEDIR   set include dir\n",
      "           -s COMMONDIR    set common dir\n",
      "           -t TOPDIR       set top dir (containing config.*)\n",
      "           -v              verbose\n",
      "           -h              generate headers\n";
    exit 1;
}


use vars qw($opt_d $opt_p $opt_o $opt_c $opt_v $opt_h
	    $opt_H $opt_C $opt_A $opt_E $opt_t $opt_i $opt_s);
use Getopt::Std;

getopts('dpo:c:vhHCAEt:i:s:');


my $topdir = $opt_t ? $opt_t : ".";
my $config = $opt_c ? $opt_c : "build/config.make";
my $common_dir = $opt_s ? $opt_s : "src/common";
my $include_dir = $opt_i ? $opt_i : "src/include";


my %macros;
my %macros_v;
my %subst;
my %macros;
my $perl;
my $shell;
my %abbrevs;
my $FP;



##### Write cf_funcs.h #######################################################

sub cf_funcs_h {
    my($key) = @_;

    my $lkey = $key;
    $lkey =~ tr/A-Z/a-z/;

    print FP
	"char *cf_p_$lkey(void);\n",
	"char *cf_s_$lkey(char *);\n";
}



##### Write cf_funcs.c #######################################################

sub cf_funcs_c {
    my($key) = @_;

    my $lkey = $key;
    $lkey =~ tr/A-Z/a-z/;

    print FP
	"/***** $key *****/\n";
    print FP
	"static char *cf_p_s_$lkey(char *s)\n",
	"{\n",
	"    static char *pval = NULL;\n",
	"    if(s)\n",
	"        pval = strsave(s);\n",
	"    if(! pval)\n",
	"    {\n",
	"	if( ! (pval = cf_get_string(\"$key\", TRUE)) )\n",
	"	    pval = DEFAULT_$key;\n",
	"	debug(8, \"config: $key %s\", pval);\n",
	"    }\n",
	"    return pval;\n",
	"}\n",
	"char *cf_p_$lkey(void)\n",
	"{\n",
	"    return cf_p_s_$lkey(NULL);\n",
	"}\n",
	"char *cf_s_$lkey(char *s)\n",
	"{\n",
	"    return cf_p_s_$lkey(s);\n",
	"}\n\n";
}



##### Write cf_env.c #########################################################

sub cf_env_c {
    my($key) = @_;

    my $lkey = $key;
    $lkey =~ tr/A-Z/a-z/;

    print FP
	"    if( (p = getenv(\"FIDOGATE_$key\")) )\n",
	"        cf_s_$lkey(p);\n";
}



##### Generate code ##########################################################
sub gen_code {
    my($name) = @_;

    my($s, $r);

    if($name eq "perl-default") {
	print "##Automatically generated by subst.pl, DO NOT EDIT!!!##\n";
	for $s (keys %macros) {
	    $r = $macros{$s};
	    $s =~ tr/A-Z/a-z/;
	    print "\t\"$s\", \"$r\",\n";
	}
    }

    if($name eq "perl-abbrev") {
	print "##Automatically generated by subst.pl, DO NOT EDIT!!!##\n";
	for $s (keys %abbrevs) {
	    $r = $abbrevs{$s};
	    $s =~ tr/A-Z/a-z/;
	    print "\t\"$r\", \"$s\",\n";
	}
    }
}


##### Read file, substitute, write to STDOUT #################################

my $first_file = 1;

sub do_file {
    my($name) = @_;

    local(*F);
    my($s, $r, $text);
    my $PERL5 = $perl;
    my $SHELL = $shell;

    open(F,"$name") || die "subst: can't open $name\n";

    while(<F>) {
	# Generate code
	if( /^\s*<GEN ([A-Za-z_-]+)>\s*$/ ) {
	    &gen_code($1);
	    next;
	}

	# Include
	if( /^\s*<INCLUDE (.*)>\s*$/ ) {
	    &do_file("$topdir/$1");
	    next;
	}

	# Substitute
	for $s (keys %macros) {
	    $r = $macros{$s};
	    s/<$s>/$r/g;
	}
	for $s (keys %subst) {
	    $r = $subst{$s};
	    s/<$s>/$r/g;
	}
	
	print;
    }
    close(F);
}



##### Main ###################################################################

my ($s, $r, $o, $m, $ls, $f);

# Read config.make
open(P, "$config") || die "subst: can't open config.make\n";

while(<P>) {
    chop;
    next if( /^\s\#/ );
    s/\#.*$//;

    ~ s{\$\{(\w+)\}} {
	no strict `refs`;
	if(defined ${$1}) {${$1}}
    }egx;


    if( /^\s*([A-Za-z0-9_]*prefix)\s*=\s*(.*)/ ) {
	print "subst: $1=$2\n" if($opt_v);
	no strict `refs`;
	${$1} = $2;
    }

    if( /^\s*DEFAULT_F_([A-Z0-9_]+)\s*=\s*(.*)\s*$/ ) {
	
	print "subst: $1=$2\n" if($opt_v);
	$macros{$1} = $2;
    }
    elsif( /^\s*DEFAULT_V_([A-Z0-9_]+)\s*=\s*(.*)\s*$/ ) {
	print "subst: $1=$2\n" if($opt_v);
	$macros{$1} = $2;
	$macros_v{$1} = 1;
    }
    elsif( /^\s*DEFAULT_A_([A-Z0-9_]+)\s*=\s*%(.)\s*$/ ) {
	print "subst: $1=%$2\n" if($opt_v);
	$abbrevs{$1} = $2;
    }
    elsif( /^\s*([A-Z0-9_]+)\s*=\s*(.*)\s*$/ ) {
	# all other macros in config.make
	$subst{$1} = $2 if(! $macros{$1});
    }
}
close(P);

# Resolve macros (only 1 level allowed)
for $s (keys %macros) {
    $r = $macros{$s};
    if( $r =~ /\$\((.*)\)/ ) {
	$o = $1;
	$m = $1;
	$m =~ s/^DEFAULT_(._)?//;
	$r =~ s/\$\($o\)/$macros{$m}/;
	$macros{$s} = $r;
	print "subst: $s=$r\n" if($opt_v);
    }
}



# Dump parameters
if($opt_d) {
    for $s (keys %macros) {
	print
	    "$s ", ($macros_v{$s} ? "(variable)" : "(fixed)"),
	    " = $macros{$s}\n";
    }
    for $s (keys %abbrevs) {
	print
	    "%$abbrevs{$s} = $s\n";
    }
    print "perl = $perl\n";

    exit 0;
}



# Write cf_funcs.h
if($opt_h) {
    open(FP, ">$include_dir/cf_funcs.mak");

    print FP
	"/* Automatically generated by subst.pl --- DO NOT EDIT!!! */\n\n",
	"/*\n",
	" * FIDOGATE\n",
	" *\n",
	" * \$Id\$\n",
	" */\n\n";
    
    for $s (keys %macros) {
	&cf_funcs_h($s) if($macros_v{$s});
    }
    close(FP);

# Write cf_funcs.c
    open(FP, ">$common_dir/cf_funcs.mak");

    print FP
	"/* Automatically generated by subst.pl --- DO NOT EDIT!!! */\n\n",
	"/*\n",
	" * FIDOGATE\n",
	" *\n",
	" * \$Id\$\n",
	" */\n\n",
	"\#include \"fidogate.h\"\n\n";
    
    for $s (keys %macros) {
	&cf_funcs_c($s) if($macros_v{$s});
    }
    close(FP);

# Write paths.h

    open(FP, ">$include_dir/paths.mak");
    print FP
	"/* Automatically generated by subst.pl --- DO NOT EDIT!!! */\n\n",
	"/*\n",
	" * FIDOGATE\n",
	" *\n",
	" * \$Id\$\n",
	" */\n\n";

    for $s (keys %macros) {
	$r = $macros{$s};
	printf FP "#define DEFAULT_%-23s \"%s\"\n", $s, $r;
    }
    close(FP);

# Write cf_abbrev.c
    open(FP, ">$common_dir/cf_abbrev.mak");
    print FP
	"/* Automatically generated by subst.pl --- DO NOT EDIT!!! */\n\n",
	"/*\n",
	" * FIDOGATE\n",
	" *\n",
	" * \$Id\$\n",
	" */\n\n";

    for $s (keys %abbrevs) {
	$ls = $s;
	$ls =~ tr/A-Z/a-z/;
	$r = $abbrevs{$s};
	printf FP "{ '$r', cf_p_$ls },\n", $s, $r;
    }
    close(FP);


# Write cf_env.c
    open(FP, ">$common_dir/cf_env.mak");

    print FP
	"/* Automatically generated by subst.pl --- DO NOT EDIT!!! */\n\n",
	"/*\n",
	" * FIDOGATE\n",
	" *\n",
	" * \$Id\$\n",
	" */\n\n";

    for $s (keys %macros) {
	&cf_env_c($s) if($macros_v{$s});
    }
    close(FP);

    exit 0;
}


# Substitute
for $f (@ARGV) {
    &do_file($f);
}

exit 0;
