#include "StdAfx.h"
#include "Log.h"
#include "OutboundReader.h"


COutboundReader::COutboundReader(const string& anOutboundPath, const string& aTempPath,
								 bool toTemp)
								 throw (CIllegalArgumentException)
{
	// check outbound path
	outboundPath = getFixedPath( anOutboundPath );
	tempPath     = getFixedPath( aTempPath      );

	if (strcmpi( outboundPath.c_str(), tempPath.c_str() ) == 0) {
		throw CIllegalArgumentException( string("paths are equal: '") + anOutboundPath
										 + string("' and '") + aTempPath
										 + string("'") );
	}

	// determine whether it is same logical drive or not
	char	outboundDrive[3];
	char	tempDrive[3];
	char	curDrive[3] = " :";
	curDrive[0] = _getdrive() - 1 + 'A';

	// try to extract outbound drive
	if (outboundPath.size() > 1 && outboundPath[1] == ':') {
		strcpy( outboundDrive, outboundPath.substr( 0, 2 ).c_str() );
	} else {
		strcpy( outboundDrive, curDrive );
	}

	// try to extract temporary drive
	if (tempPath.size() > 1 && tempPath[1] == ':') {
		strcpy( tempDrive, tempPath.substr( 0, 2 ).c_str() );
	} else {
		strcpy( tempDrive, curDrive );
	}

	sameDrive = (strcmpi( outboundDrive, tempDrive ) == 0);

	// create destination (temp) directory
	createPath( tempPath );

	if (toTemp)
		save();
	else
		restore();
}


string COutboundReader::getFixedPath(const string& path) const {
	string	workingPath = path;
	if (workingPath.size() > 0) {
		// check last symbol (remove separator)
		if (workingPath[workingPath.size()-1] == PATH_SEPARATOR_CHAR) {
			workingPath = workingPath.substr( 0, workingPath.size()-1 );
		}
	}

	if (workingPath.size() == 0) {
		workingPath = ".";
	}

	return workingPath;
}


COutboundReader::~COutboundReader() {
}


void COutboundReader::save() throw (CIOException) {
	queue<string>	paths;
	paths.push( "" );

	while (paths.size() > 0) {
		string	path = paths.front();
		paths.pop();

		// prepare mask to search
		string	mask = outboundPath;
		mask += PATH_SEPARATOR_CHAR;
		mask += path + "*.*";

		bool	noTempSubdir = true;

		_finddata_t	findData;
		long		findHandle = _findfirst( mask.c_str(), &findData );
		if (findHandle != -1) {
			do {
				// skip "system", "hidden" and "read only" files
				if ((findData.attrib & (_A_SYSTEM | _A_HIDDEN | _A_RDONLY)) != 0) {
					continue;
				}
				// remember subdirectories (except for special "." and "..")
				if ((findData.attrib & _A_SUBDIR) != 0) {
					if ((strcmp( findData.name, "."  ) != 0) &&
						(strcmp( findData.name, ".." ) != 0))
					{
						string	newPath = path + findData.name;
						newPath += PATH_SEPARATOR_CHAR;
						paths.push( newPath );
					}
					continue;
				}
				// ingore non-tic files
				char*	atDot = strrchr( findData.name, '.' );
				if (atDot == NULL) {
					// no dot in name
					continue;
				}
				if (strcmpi( atDot, ".tic" ) != 0) {
					// extension is not "tic"
					continue;
				}

				// at this moment we have a tic!

				// ensure that we have destination subdirectory
				if (noTempSubdir) {
					string	tempSubdir = tempPath;
					tempSubdir += PATH_SEPARATOR_CHAR;
					tempSubdir += path;
					makeDir( tempSubdir.c_str() );
					noTempSubdir = false;
				}

				checkTic( path, findData.name );

			} while (_findnext( findHandle, &findData ) != -1);
			_findclose( findHandle );
		}
	}
}


void COutboundReader::checkTic(const string& path, const string& ticFileName) {
	string	fullTicFileName = outboundPath;
	fullTicFileName += PATH_SEPARATOR_CHAR;
	fullTicFileName += path + ticFileName;

	bool	found = false;
	string	dataFileName;

	// Tic line is limited by 256 symbols in length, so we use buffer size > 265
	char	line[512];

	ifstream	in( fullTicFileName.c_str() );
	try {
		while (!in.eof()) {
			in.getline( line, 257 );	// 256 symbols plus '\0'
			// check line
			if (strncmp( line, "File ", 5 ) == 0) {
				// we've found a line with "File" tag

				dataFileName = line + 5;
				found = true;
				break;
			}
		}

		in.close();
	} catch (...) {
		// close anyway
		in.close();
		throw;
	}

	if (found) {
		try {
			// move tic
			moveFileToTemp( path, ticFileName );
			try {
				// move data
				moveFileToTemp( path, dataFileName );
			} catch (CIOException& e) {
				CLog::newLine() << "failed to move data file: " << dataFileName << "\n";
				CLog::newLine() << e.getMessage() << "\n";
			}
		} catch (CIOException& e) {
			CLog::newLine() << "failed to move TIC file: " << ticFileName << "\n";
			CLog::newLine() << e.getMessage() << "\n";
		}
	}
}


void COutboundReader::createPath(const string& path) const throw (CIOException) {
	char*	buffer = new char[path.size() + 1];
	try {
		strcpy( buffer, path.c_str() );

		int	separator = -1;
		do {
			separator = path.find_first_of( PATH_SEPARATOR_CHAR, separator+1 );
			if (separator > 0) {
				// truncate buffer
				buffer[separator] = '\0';

				makeDir( buffer );

				// restore buffer
				buffer[separator] = PATH_SEPARATOR_CHAR;
			}
		} while (separator > 0);

		// create entire directory
		makeDir( buffer );

		delete[] buffer;
	} catch (...) {
		// always free buffer
		delete [] buffer;
		throw;
	}
}


void COutboundReader::makeDir(const char* dir) const throw (CIOException) {
	// check last character
	if (dir[strlen(dir) - 1] == ':') {
		// we don't create drives ;)
		return;
	}

	if (_mkdir( dir ) != 0) {
		if (errno != EEXIST) {
			throw CIOException( string("failed to create directory: ") + dir );
		}
	}
}


void COutboundReader::moveFile(const string& path, const string& name, bool toTemp) 
	const throw (CIOException)
{
	string	srcName = (toTemp ? outboundPath : tempPath);
	srcName += PATH_SEPARATOR_CHAR;
	srcName += path;
	srcName += name;

	string	destName = (toTemp ? tempPath : outboundPath);
	destName += PATH_SEPARATOR_CHAR;
	destName += path;
	destName += name;

	if (sameDrive) {
		if (rename( srcName.c_str(), destName.c_str() ) != 0) {
			throw CIOException( string("failed to rename file: ") + srcName + " -> " + destName );
		}
	} else {
		const	BUFFER_SIZE = 32*1024;
		char	buffer[BUFFER_SIZE];

		HANDLE	in = ::CreateFile( srcName.c_str(), GENERIC_READ,
								   FILE_SHARE_READ, NULL, OPEN_EXISTING, 
								   FILE_ATTRIBUTE_NORMAL | FILE_FLAG_SEQUENTIAL_SCAN, NULL );

		if (in == INVALID_HANDLE_VALUE) {
			throw CIOException( string("failed to open file for reading: ") + srcName );
		}

		HANDLE	out = ::CreateFile( destName.c_str(), GENERIC_WRITE,
									FILE_SHARE_READ, NULL, CREATE_ALWAYS,
									FILE_ATTRIBUTE_NORMAL, NULL );

		if (out == INVALID_HANDLE_VALUE) {
			::CloseHandle( in );
			throw CIOException( string("failed to open file for writing: ") + destName );
		}

		try {

			DWORD	haveBytes, savedBytes;
			do {
				haveBytes = 0;
				if (::ReadFile( in, buffer, BUFFER_SIZE, &haveBytes, NULL )) {
					if (haveBytes > 0) {
						if (!::WriteFile( out, buffer, haveBytes, &savedBytes, NULL )) {
							throw CIOException( string("failed to write data into file: ") + destName );
						}
					}
				} else {
					throw CIOException( string("failed to read data from file: ") + srcName );
				}
			} while (haveBytes > 0);

		} catch (...) {
			// ensure that we close files and unlink destination file (we failed to copy)
			::CloseHandle( out );
			::CloseHandle( in );
			_unlink( destName.c_str() );
			throw;
		}

		FILETIME	creationTime, lastAccessTime, lastWriteTime;

		if (::GetFileTime( in, &creationTime, &lastAccessTime, &lastWriteTime )) {
			if (!::SetFileTime( out, &creationTime, &lastAccessTime, &lastWriteTime )) {
				CLog::newLine() << "warning: failed to set file time for file: " << destName << "\n";
			}
		} else {
			CLog::newLine() << "warning: failed to get file time of file: " << srcName << "\n";
		}

		::CloseHandle( out );
		::CloseHandle( in );

		if (_unlink( srcName.c_str() ) != 0) {
			CLog::newLine() << "warning: failed to unlink source file: " << srcName << "\n";
		}
	}
}


void COutboundReader::restore() throw (CIOException) {
	queue<string>	paths;
	paths.push( "" );

	while (paths.size() > 0) {
		string	path = paths.front();
		paths.pop();

		// prepare mask to search
		string	mask = tempPath;
		mask += PATH_SEPARATOR_CHAR;
		mask += path + "*.*";

		bool	noOutboundSubdir = true;

		_finddata_t	findData;
		long		findHandle = _findfirst( mask.c_str(), &findData );
		if (findHandle != -1) {
			do {
				// skip "system", "hidden" and "read only" files
				if ((findData.attrib & (_A_SYSTEM | _A_HIDDEN | _A_RDONLY)) != 0) {
					continue;
				}
				// remember subdirectories (except for special "." and "..")
				if ((findData.attrib & _A_SUBDIR) != 0) {
					if ((strcmp( findData.name, "."  ) != 0) &&
						(strcmp( findData.name, ".." ) != 0))
					{
						string	newPath = path + findData.name;
						newPath += PATH_SEPARATOR_CHAR;
						paths.push( newPath );
					}
					continue;
				}

				// at this moment we have a file to move!

				// ensure that we have destination subdirectory
				if (noOutboundSubdir) {
					string	outboundSubdir = outboundPath;
					outboundSubdir += PATH_SEPARATOR_CHAR;
					outboundSubdir += path;
					makeDir( outboundSubdir.c_str() );
					noOutboundSubdir = false;
				}

				try {
					moveFileToOutbound( path, findData.name );
				} catch (CIOException& e) {
					CLog::newLine() << "failed to move file from temporary storage: "
									<< tempPath << PATH_SEPARATOR_CHAR << path << findData.name << "\n";
					CLog::newLine() << e.getMessage() << "\n";
				}

			} while (_findnext( findHandle, &findData ) != -1);
			_findclose( findHandle );
		}
	}
}
