#ifndef _GRINGO_H
#define _GRINGO_H

// libgringo - @1998-2000 by Antonio Gallo <agx@linux.it> 
// http://www.badpenguin.org	http://www.linux.it/~agx/

#include <badpenguin-chain.h>
#ifdef USES_NCURSES
  #include <ncurses.h>
#endif
#ifdef USES_GTK
  #include <gtk.h>
  #include <gtkwidget.h>
#endif

// ------------------------------------------- The Object structure

// Type declaration
typedef struct gringo_Object    TObject;
typedef struct gringo_Window    TWindow;
typedef struct gringo_Label     TLabel;
typedef struct gringo_Button    TButton;
typedef struct gringo_Edit      TEdit;
typedef struct gringo_CheckBox  TCheckBox;
typedef struct gringo_ListBox   TListBox;
typedef struct gringo_Item      TItem;
typedef struct gringo_Grid      TGrid;
typedef struct gringo_Column    TColumn;

// Macros for type casting
#define TOBJECT(x)	((TObject*)(x))
#define TWINDOW(x)	((TWindow*)(x))
#define TLABEL(x)	((TLabel*)(x))
#define TBUTTON(x)	((TButton*)(x))
#define TEDIT(x)	((TEdit*)(x))
#define TCHECKBOX(x)	((TCheckBox*)(x))
#define TLISTBOX(x)	((TListBox*)(x))
#define TITEM(x)	((TItem*)(x))
#define TGRID(x)	((TGrid*)(x))

// Type of objects
#define TYPE_UNKNOWN		  0
#define TYPE_NOTSELECTABLE	100
#define TYPE_WINDOW		101
#define TYPE_LABEL		102
#define TYPE_LISTITEM		103
#define TYPE_SELECTABLE		200
#define TYPE_BUTTON		201
#define TYPE_EDITBOX		202
#define TYPE_CHECKBOX		203
#define TYPE_LISTBOX		204
#define TYPE_GRID		205


struct gringo_Object {
// The object structure is common to all the visible widgets
// It store information upon the object itself (name, type),
// some of the basical properties for visualization and
// field to double-link it with other objects
  // Intrinsic
  char *name;				// Object Name
  int   type;				// Tipe of the widget assigned by the xxxxx_Create function
  // Visual properties
  int x, y, w, h;			// Positions and Dimensions
  int active;				// TRUE if the object is the active object
  int enabled;				// TRUE if the object can become active
  int visible;				// TRUE if the object should be painted
  // Family Tree
  void *parent;				// Parent pointer
  TChainItem *me;	// Identify the TChainItem that hold the window into the TChain gringo_Windows
			// or a widget into a Window's widget TChain  
};



struct gringo_Window {
  TObject object;
  // Event Callback
  void	(* fOnClose)();
  // Childrens
  // objects of the same window are double-linked togheter
  // those are the pointers used to manage that list
  void *activeChild;		// current active object
  TChain *widgets;
#ifdef USES_NCURSES
  WINDOW *window;				// Ncurses window
  char   *title;				// Current title
  chtype  title_color, title_color_active;	// Title colors
  chtype  back_color, back_color_active;	// Back colors
  int     trasparent;				// If background should be painted
  chtype  border_color, border_color_active;	// Color Borders
  int     border_width;				// Border width (0=none,1=normal)
  int     shadow;				// If Window has a shadow
#endif
#ifdef USES_GTK
  GtkWidget *widget;		// Widget for Gtk
  GtkWidget *fixed;		// Container for fixed positions of Gtk objects
#endif
};



struct gringo_Label {
  TObject object;
#ifdef USES_NCURSES
  WINDOW *window;				// Ncurses window
  char   *caption;				// label's caption
  chtype  caption_color, caption_color_active;	// caption colors
#endif
#ifdef USES_GTK
  GtkWidget *widget;
#endif
};



struct gringo_Button {
  TObject object;
  int     status;				// 0=not clicked; 1=clicked
#ifdef USES_NCURSES
  WINDOW *window;				// Ncurses window
  char   *caption;				// label's caption
  chtype  caption_color, caption_color_active;	// caption colors
  void  (*fOnClick)();				// Button's OnClick Event
#endif
#ifdef USES_GTK
  GtkWidget *widget;
#endif
};



struct gringo_Edit {
  TObject object;
#ifdef USES_NCURSES
  WINDOW *window;				// Ncurses window
  char   *text;					// text in editing
  int     curpos;				// Cursor Editing Position
  chtype  text_color, text_color_active;	// text color
  chtype  back_color, back_color_active;	// background colors  
#endif
#ifdef USES_GTK
  GtkWidget *widget;
#endif
};



struct gringo_CheckBox {
  TObject object;
#ifdef USES_NCURSES
  WINDOW *window;				// Ncurses window
  char   *caption;				// label's caption
  chtype  caption_color, caption_color_active;	// caption colors
  int     status;				// 0=unchecked 1=checked
#endif
#ifdef USES_GTK
  GtkWidget *widget;
#endif
};


struct gringo_Item {
  char  *text;
  TItem *next;
  TItem *prev;
};


struct gringo_Column {
  char    *caption;
  int      width;
  int      height;  
  TColumn *next;
  TColumn *prev;
  TItem   *firstItem;
  TItem   *lastItem;
  TItem   *pageItem;		// first Item of the page
  TItem   *activeItem;		// current active object
  int      activeIndex;
#ifdef USES_NCURSES
  chtype  caption_color;
  chtype  item_color, item_color_active, item_color_inactive; // Text Color
#endif
};


struct gringo_ListBox {
  TObject object;
  int    activeIndex;
#ifdef USES_NCURSES
  WINDOW *window;				// Ncurses window
  char   *title;				// Current title
  chtype  title_color, title_color_active;	// Title colors
  chtype  back_color, back_color_active;	// Back colors
  chtype  border_color, border_color_active;	// Color Borders
  chtype  item_color, item_color_active;	// TExt Color 
  chtype  item_color_inactive; 
  int     border_width;				// Border width (0=none,1=normal)
  // Items
  TItem *firstItem;		// first object of the list
  TItem *lastItem;		// last object of the list
  TItem *pageItem;		// first Item of the page
  TItem *activeItem;		// current active object
#endif
#ifdef USES_GTK
  GtkWidget *widget;
#endif
};


struct gringo_Grid {
  TObject object;
#ifdef USES_NCURSES
  WINDOW *window;				// Ncurses window
  chtype  back_color, back_color_active;	// Back colors
  chtype  border_color, border_color_active;	// Color Borders
  int     border_width;				// Border width (0=none,1=normal)
  TColumn *firstCol;				// first object of the list
  TColumn *lastCol;				// last object of the list
  void  (*fOnClick)();				// Grids's OnClick Event
#endif
#ifdef USES_GTK
  GtkWidget *widget;
  int	     columns;
#endif
};


// ------------------------------------------- Return Codes
#define GRINGO_ERROR_NONE		0
#define GRINGO_ERROR_NOWINDOWS		97
#define GRINGO_ERROR_NOMOREWINDOWS	98
#define GRINGO_ERROR_NOPAINTMETHOD	99

// ------------------------------------------- PUBLICS

TChain  *gringo_Windows;

// ------------------------------------------- FUNCTIONS

// GRINGO
// Your program should start with a call to gringo_init
// then you can create any window and add to it any object
// last call gringo_Loop and put gringo_Quit at the end of the program
// use gringo_Terminate to quit the program inside of a callback
void gringo_Init(int *argc, char **argv[] );
int  gringo_Loop();
void gringo_Terminate(int sig);
void gringo_Quit();
int libgringo_version();

// WINDOW
// It is non necessary to call window_Destroy for any window, this is
// done automatically by libgringo. Use window_Destroy only when you
// want to kill a specified window (we suggest you to use Show/Hide
// instead).
TWindow* window_Create( int x, int y, int w, int h, char *title );
void window_Activate( TWindow *win );
void window_Deactivate( TWindow *win );
void window_Show( TWindow *win );
void window_Hide( TWindow *win );
void window_Close( TWindow *win );
void window_Enable( TWindow *win );
void window_Disable( TWindow *win );
void window_OnClose(TWindow *win, void *f );

/* WIDGETS */
void object_SetName( TObject *obj, char *newname );

// LABEL
TLabel* label_Create( TWindow* win, int x, int y, int w, char *title );
char* label_GetValue( TLabel *obj );

// BUTTON
TButton* button_Create( TWindow* win, int x, int y, int w, char *title );
void button_OnClick( TButton *obj, void * f );
int button_GetValue( TButton *obj );

// EDIT BOX
TEdit* edit_Create( TWindow* win, int x, int y, int w, char *text );
char* edit_GetValue( TEdit *obj );

// CHECK BOX
TCheckBox* checkbox_Create( TWindow* win, int x, int y, char *title, int checked );
int checkbox_GetValue( TCheckBox *obj );

// LIST BOX
TListBox* listbox_Create( TWindow* win, int x, int y, int w, int h );
void listbox_AddItem( TListBox* obj, char *s );
char *listbox_GetValue( TListBox *obj );
// GRID BOX
TGrid *grid_Create( TWindow* win, int x, int y, int w, int h );
TColumn *grid_AddColumn( TGrid *obj, char *title, int w );
void grid_AddText( TGrid *obj, ... );
void grid_OnClick( TGrid *obj, void * f );

#endif
