/*
 * chkimap - mail folder check
 * $Id: chkimap.c,v 1.3 2002/01/26 11:53:36 johans Exp $
 */

/*
 * Copyright (C) 1998 - 2001, Johan van Selst <johans@stack.nl>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <string.h>
#include <netdb.h>
#include <pwd.h>
#include <err.h>
#include <unistd.h>

#define MAXLEN 1024

int cmdcount = 0, folder = 0;
int spc = 5, hideold = 0, mlen = 4, caps = 0;
char buf[MAXLEN];
char *folders[MAXLEN];

void
usage()
{
	/* give clue */
	fputs("chkimap 2.10 - mail folder check\n"
		"Usage: chkimap [options] [user@]server\n\n"
		"-# <n>     Pad the number of messages with n spaces\n"
		"-c         Include folder names with capitals\n"
		"-h         Hide folders containing only old mail\n"
		"-i         Only parse standard incoming mailfolder\n"
		"-n         Only parse folders that received new mail\n"
		"Read the manual for more details.\n",
		stderr);
	exit(1);
}

int
parse_list(const char *line)
{
	int len = 0;
	char *file;

#ifdef		HAVE_STRCASESTR
	if (strncasecmp(line, "* LIST ", 7) ||
		strcasestr(line, "\\NoSelect"))
#else
	if (strncasecmp(line, "* LIST ", 7) ||
		strstr(line, "\\NoSelect"))
#endif		/* HAVE_STRCASESTR */
		return 1;

	file = (char *)strrchr(line, ' ') + 1;
	if ((len = strlen(file)) &&
		strcmp(file, "INBOX") &&
		(file[0] < 'a' || file[0] > 'z') &&
		(file[0] < 'A' || file[0] > 'Z' || !caps) &&
		(file[0] < '0' || file[0] > '9'))
		return 1;

	folders[folder++] = (char *)strdup(file);
	if (len > mlen)
		mlen = len;
	return 0;
}

int
parse_status(const char *line)
{
	int new, rnew, total;
	char folnam[MAXLEN];

	if (sscanf(line, "* STATUS %s (MESSAGES %d RECENT %d UNSEEN %d)",
		folnam, &total, &rnew, &new) == 4 && total)
	{
		if (!new)
		{
			if (!hideold)
				fprintf(stdout, "%-*s   - new of total%*d\n",
					mlen, folnam, spc, total);
		}
		else if (rnew || hideold < 2)
			fprintf(stdout, "%-*s %3d new of total%*d\n",
				mlen, folnam, new, spc, total);
	}
	return 0;
}

int
imapexec(int s, char *cmd, int (*parse)(const char *))
{
	int len, done;
	char *line, *next, *last, *uid = malloc(strlen(cmd) + 12);
	static char bigbuf[MAXLEN * MAXLEN];

	done = 0;
	bigbuf[0] = '\0';
	line = next = cmd;
	last = cmd + strlen(cmd) - 1;
	while (next < last &&
		((next = (char *)strchr(line, '\n')) ||
		 (next = (char *)strchr(line, '\0'))))
	{
		if (next < last)
			*next++ = '\0';
		snprintf(uid, MAXLEN, "chk%05d %s\r\n", cmdcount++, line);
		strcat(bigbuf, uid);
		line = next;
	}
	if (send(s, bigbuf, strlen(bigbuf), NULL) < 0)
		err(1, NULL);

	while ((len = recv(s, buf, MAXLEN, NULL)) > 0)
	{
		line = next = buf;
		while (*next && (next = (char *)strchr(line, '\r')))
		{
			*next++ = '\0';
			if (!strncmp(line, uid, 9))
			{
				line += 9;
				if (!strncmp(line, "OK", 2))
					return 0;
				else if (!strncmp(line, "NO", 2))
					return 1;
				else if (!strncmp(line, "BAD", 3))
					return 2;
				else
					return -1;
			}
			else if (parse)
				parse(line);
			line = next + 1;
		}
	}
	return 0;
}

int
main(int argc, char *argv[])
{
	int s, i, opt, error, done, inonly = 0;
	char buf[MAXLEN], bigbuf[MAXLEN*MAXLEN], *user, *pwd, *host, *port;
	struct addrinfo hints, *res;

	port = NULL;
	while ((opt = getopt(argc, argv, "#:chilnp:")) != EOF)
		switch (opt)
		{
		case '#':
			if ((spc = atoi(optarg)) <= 0)
				usage();
			break;
		case 'c':
			caps = 1;
			break;
		case 'h':
			hideold = 1;
			break;
		case 'i':
			inonly = 1;
			break;
		case 'n':
			hideold = 2;
			break;
		case 'p':
			port = optarg;
			break;
		default:
			usage();
		}

	if (optind >= argc)
		usage();

	if ((host = (char *)strchr(argv[optind], '@')))
	{
		user = argv[optind];
		*host++ = '\0';
	}
	else
	{
		user = getenv("USER");
		host = argv[optind];
	}
	snprintf(buf, MAXLEN, "IMAP password for %s@%s: ", user, host);
	pwd = getpass(buf);

	memset(&hints, 0, sizeof(hints));
	hints.ai_family = PF_UNSPEC;
	hints.ai_socktype = SOCK_STREAM;
	if ((error = getaddrinfo(host, port ? port : "imap", &hints, &res)))
		errx(1, "%s", gai_strerror(error));

	if ((s = socket(res->ai_family, res->ai_socktype, res->ai_protocol)) <  0)
		err(1, NULL);
	if (connect(s, res->ai_addr, res->ai_addrlen) < 0)
		err(1, NULL);

	done = folder = 0;
	snprintf(buf, MAXLEN, "login %s %s", getenv("USER"), pwd);
	if (imapexec(s, buf, NULL))
		errx(1, "Authentication failure");

	if (!inonly)
	{
		imapexec(s, "list \"\" %", parse_list);

		bigbuf[0] = '\0';
		for (i = 0; i < folder; i++)
		{
			snprintf(buf, MAXLEN, "status %s (messages recent unseen)\n",
				folders[i]);
			strcat(bigbuf, buf);
		}
		bigbuf[strlen(bigbuf)-1] = '\0';
		imapexec(s, bigbuf, parse_status);
	}
	else
		imapexec(s, "status inbox (messages recent unseen)", parse_status);
	imapexec(s, "logout", NULL);
	close(s);

	return 0;
}
