/*
 * chk4mail - mail folder check
 * $Id: chk4mail.c,v 1.31 2002/01/26 11:30:00 johans Exp $
 */

/*
 * Copyright (C) 1998 - 2001, Johan van Selst <johans@stack.nl>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#define __USE_BSD
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#undef __USE_BSD
#include <string.h>
#include <dirent.h>
#include <utime.h>
#include <sys/utsname.h>
#ifdef __linux__
#include <getopt.h>
#endif

#define MAXHEAD 256
#define MAXDIR 1024
#define MAXFILE 80

/* probably shouldn't have this bit hardcoded */
#define REMOTE_PATH "/usr/local/bin/chk4mail"

int spc = 5, hideold = 0, mlen = 4, quiet = 0;
static char rcsid[] = "$Id: chk4mail.c,v 1.31 2002/01/26 11:30:00 johans Exp $";

void
usage()
{
	/* give clue */
	fputs("chk4mail 2.10 - mail folder check\n"
		"Mbox, MMDF, MH and Maildir folders are recognized.\n\n"
		"-# <n>     Pad the number of messages with n spaces\n"
		"-c         Include folder names with capitals\n"
		"-d <dir>   Look for mailfolders in this directory instead\n"
		"-h         Hide folders containing only old mail\n"
		"-i         Only parse standard incoming mailfolder\n"
		"-l         Always follow symbolic links\n"
		"-n         Only parse folders that received new mail\n"
		"-q         Suppress errors about missing or unreadable files\n"
		"-r <host>  Execute the command on an other host (with rsh)\n\n"
		"Folders may be specified as arguments as well (filename or =folder)\n"
		"Read the manual for more details.\n",
		stderr);
	exit(1);
}

void
fail(char *msg, char *file)
{
	/* write friendly message */
	fputs(msg, stderr);
	fputs(": ", stderr);
	fputs(file, stderr);
	fputc('\n', stderr);
	exit(1);
}

int
parse_mbox_mmdf(FILE *file, int *new, int *total, int mmdf)
{
	int nrs, read, next, header = 1;
	long skip;
	char line[MAXHEAD+1];

	nrs = read = 0;
	while (!feof(file))
	{
		skip = next = 0;

		/* parse headers */
		while (fgets(line, MAXHEAD, file) && strlen(line) > 1)
		{
			if ((!mmdf && !strncmp(line, "From ", 5)) ||
				(mmdf && !strncmp(line, "\1\1\1\1", 4)))
				next = header = 1;
			else if (header)
			{
				if (strlen(line) < 2)
					header = 0;
				else if (!strncmp(line, "Content-Length: ", 16))
					skip = atol(line + 16);
				else if (!strncmp(line, "Status: RO", 10) ||
					 !strncmp(line, "Status: OR", 10))
					read++;
			}
		}

		/* skip body (hopefully) */
		if (next)
		{
			nrs++;
			fseek(file, skip, SEEK_CUR);
		}
	}

	if (mmdf)
		nrs--;

	*total = nrs;
	*new = nrs - read;

	return 0;
}

int
parse_mbox(FILE *file, int *new, int *total)
{
	return parse_mbox_mmdf(file, new, total, 0);
}

int
parse_mmdf(FILE *file, int *new, int *total)
{
	return parse_mbox_mmdf(file, new, total, 1);
}

int
parse_maildir(char *subdir, int *new, int *total)
{
	char *filename = malloc(strlen(subdir) + 5), *p;
	struct dirent *file;
	DIR *dir;
	int read;

	*new = -4;	/* 2x . and .. subdirs */
	*total = 0;

	strcpy(filename, subdir);
	strcat(filename, "/cur");
	if (!(dir = opendir(filename)))
		fail("Unable to open subdirectory", subdir);

	while ((file = readdir(dir)))
	{
		read = 0;
		if ((p = strrchr(file->d_name, ':')) && !strncmp(p, ":2,", 3))
		{
			p += 2;
			while (*(++p))
				if (*p == 'S')
					read = 1;
		}
		if (read)
			(*total)++;
		else
			(*new)++;
	}
	closedir(dir);

	strcpy(filename, subdir);
	strcat(filename, "/new");
	if (!(dir = opendir(filename)))
		fail("Unable to open subdirectory", subdir);

	while ((file = readdir(dir)))
		(*new)++;
	closedir(dir);

	*total += *new;

	return 0;
}

int
parse_mh(char *subdir, int *new, int *total)
{
	DIR *dir;
	struct dirent *file;
	struct stat statbuf;
	const int len = strlen(subdir) + 7;
	char *x;
	char *filename = malloc(len);
	char real;

	if (!(dir = opendir(subdir)))
		fail("Unable to open subdirectory", subdir);

	/* total = #filenames consisting of numbers */
	for (*total = 0; (file = readdir(dir)); )
	{
		for (real = 1, x = file->d_name; *x; x++)
			real &= *x >= '0' && *x <= '9';
		if (real)
		{
			(*total)++;
			snprintf(filename, len, "%s/%s", subdir, file->d_name);
			filename[len-1] = '\0';
			if (!stat(filename, &statbuf))
				if (statbuf.st_mtime >= statbuf.st_atime)
					(*new)++;
		}
	}

	closedir(dir);
	return 0;
}

int
parse_dir(char *dir, int *new, int *total)
{
	struct stat *filestat = malloc(sizeof(struct stat));
	char *filename = malloc(strlen(dir) + 5);

	strcpy(filename, dir);
	strcat(filename, "/cur");

	if (lstat(filename, filestat) < 0 || !S_ISDIR(filestat->st_mode))
		/* parse MH-style mailfolder */
		return parse_mh(dir, new, total);
	else
		/* parse cur/new/tmp-style mailfolder */
		return parse_maildir(dir, new, total);
}

int
display(char *folder, int new, int total)
{
	if (total > 0)
	{
		if (new < 0)
			fprintf(stdout, "%-*s   ? new of total%*d\n",
				mlen, folder, spc, total);
		else if (!new)
		{
			if (!hideold)
				fprintf(stdout, "%-*s   - new of total%*d\n",
					mlen, folder, spc, total);
		}
		else
			fprintf(stdout, "%-*s %3d new of total%*d\n",
				mlen, folder, new, spc, total);
	}

	return 0;
}


int
parse_folder(char *dir, char *filenm, int flink)
{
	char *filename = malloc(strlen(dir) + strlen(filenm) + 2), c;
	struct stat *filestat = malloc(sizeof(struct stat));
	struct utimbuf *filetime = malloc(sizeof(struct utimbuf));
	FILE *file;
	int new, total;

	strcpy(filename, dir);
	strcat(filename, "/");
	strcat(filename, filenm);

	if (lstat(filename, filestat) < 0)
	{
		if (quiet)
			return 2;
		else
			fail("Unable to stat file", filename);
	}

	/* Follow symlinks iff 'flink' */
	if (S_ISLNK(filestat->st_mode) && flink &&
		(stat(filename, filestat) < 0))
	{
		if (quiet)
			return 2;
		else
			fail("Unable to stat file", filename);
	}
	if (S_ISDIR(filestat->st_mode))
	{
#ifdef	SUBMAILDIR
		char *subsub = malloc(MAXFILE), *p;
		DIR *subdir;
		struct dirent *subfile;
#endif	/* SUBMAILDIR */

		/* parse directory mailfolder */
		new = total = 0;
		parse_dir(filename, &new, &total);
		display(filenm, new, total);

#ifdef	SUBMAILDIR
		if (!(subdir = opendir(filename)))
			fail("Unable to open subdirectory", filename);

		/* find maildir subdirectories */
		while ((subfile = readdir(subdir)))
		{
			if (!strcmp(subfile->d_name, ".") ||
				!strcmp(subfile->d_name, ".."))
				continue;
			/* subdirs 'new' and 'cur' should exist - prolly 'tmp' too */
			snprintf(subsub, MAXFILE, "%s/%s/new", filename, subfile->d_name);
			if ((stat(subsub, filestat) < 0) || !S_ISDIR(filestat->st_mode))
				continue;
			snprintf(subsub, MAXFILE, "%s/%s/cur", filename, subfile->d_name);
			if ((stat(subsub, filestat) < 0) || !S_ISDIR(filestat->st_mode))
				continue;
			/* looks like a maildir, let's check it out... */
			if (!(p = strrchr(subsub, '/')))
				continue;	/* silently ignore this error */
			*p = '\0';
			parse_maildir(subsub, &new, &total);

			snprintf(subsub, MAXFILE, "../%s", subfile->d_name);
			display(subsub, new, total);
		}
#endif	/* SUBMAILDIR */

		return 1;
	}
	if (!S_ISREG(filestat->st_mode))
		return 2;
	if (!filestat->st_size)
		return 3;

	/* open and parse mailfolder */
	filetime->actime = filestat->st_atime;
	filetime->modtime = filestat->st_mtime;

	if (hideold > 1 && filetime->modtime < filetime->actime)
		return 0;

	if (!(file = fopen(filename, "r")))
	{
		if (quiet)
			return 4;
		else
			fail("Unable to open file", filename);
	}

	if ((c = fgetc(file)) == '\1')
	{
		ungetc(c, file);
		parse_mmdf(file, &new, &total);
	}
	else
	{
		ungetc(c, file);
		parse_mbox(file, &new, &total);
	}
	display(filenm, new, total);

	/* close and preserve timestamp */
	fclose(file);
	utime(filename, filetime);

	return 0;
}

int
main(int argc, char **argv)
{
	char *maildir = malloc(MAXFILE+1);
	char *inbox = NULL;
	char *remote = NULL;
	DIR *dir;
	struct dirent *file;
	char *slash;
	char dirlist[MAXDIR][MAXFILE];
	struct utsname uts;
	int opt, max, len, inonly = 0, flink = 0, caps = 0;

	maildir[0] = '\0';
	len = strlen(argv[0]);
	if ((len > 5 && !strcmp(argv[0] + len - 5, "/nfrm")) ||
		(len == 4 && !strcmp(argv[0], "nfrm")))
		inonly = 1;

	while ((opt = getopt(argc, argv, "#:cd:hilnqr:")) != EOF)
		switch (opt)
		{
		case '#':
			if ((spc = atoi(optarg)) <= 0)
				usage();
			break;
		case 'c':
			caps = 1;
			break;
		case 'd':
			strncpy(maildir, optarg, MAXFILE);
			break;
		case 'h':
			hideold = 1;
			break;
		case 'i':
			inonly = 1;
			break;
		case 'l':
			flink = 1;
			break;
		case 'n':
			hideold = 2;
			break;
		case 'q':
			quiet = 1;
			break;
		case 'r':
			remote = strdup(optarg);
			break;
		default:
			usage();
		}

	if (remote)
	{
		if (uname(&uts) < 0)
			fail("uname() failed", "???");
		if ((slash = strchr(uts.nodename, '.')))
			slash[0] = '\0';
		if (strcmp(remote, uts.nodename))
		{
			while (--argc)
				*(argv[argc]-1) = ' ';
			/* execl() does not return on success */
			if (execl("/usr/bin/rsh", remote, *argv, NULL) < 0)
				fail("execl() failed", remote);
		}
	}
	/* find inbox */
	if ((inbox = getenv("MAIL")))
		inbox = strdup(inbox);

	/* Shortcut to the end */
	if (inonly)
		goto INBOX;

	/* find maildir */
	if (!*maildir)
	{
		strncpy(maildir, getenv("HOME"), MAXFILE-5);
		strcat(maildir, "/Mail");
		if (!(dir = opendir(maildir)))
		{
			strncpy(maildir, getenv("HOME"), MAXFILE-5);
			strcat(maildir, "/mail");
		}
	}
	if (!(dir = opendir(maildir)))
		fail("Unable to open maildirectory", maildir);

	/* only parse specified mailboxes and exit */
	if (optind < argc)
	{
		/* for a nice layout: find max filename length first */
		opt = optind;
		while (opt < argc)
		{
			if ((slash = strrchr(argv[opt], '/')))
				len = strlen(slash + 1);
			else
				len = strlen(argv[opt]);
			opt++;
			if (len > mlen)
				mlen = len;
		}
		opt = optind;
		while (opt < argc)
		{
			if (argv[opt][0] == '=' || argv[opt][0] == '+')
				parse_folder(maildir, argv[opt] + 1, 1);
			else if ((slash = strrchr(argv[opt], '/')))
			{
				slash[0] = '\0';
				parse_folder(argv[opt], slash+1, 1);
			}
			else
			{
				getcwd(maildir, MAXFILE);
				parse_folder(maildir, argv[opt], 1);
			}
			opt++;
		}
		closedir(dir);
		return 0;
	}

	/* sort maildir */
	max = 0;
	memset(dirlist, 0, MAXFILE*MAXDIR);
	while ((file = readdir(dir)))
	{
		/* check if this mailfolder is worth reading */
		if ((len = strlen(file->d_name)) &&
			(file->d_name[0] < 'a' || file->d_name[0] > 'z') &&
			(file->d_name[0] < 'A' || file->d_name[0] > 'Z' || !caps) &&
			(file->d_name[0] < '0' || file->d_name[0] > '9'))
			continue;

		if ((len > 4) && !strcmp((char *)(file->d_name + len-5), ".lock"))
			continue;

		if (len > mlen)
			mlen = len;
		strncpy(dirlist[max++], file->d_name, MAXFILE);
	}

	qsort(dirlist, MAXDIR, MAXFILE,
		(int (*)(const void *, const void *))strcmp);
	max = MAXDIR-max;

	/* parse maildir */
	while (max < MAXDIR)
		parse_folder(maildir, dirlist[max++], flink);
	closedir(dir);

INBOX:
	/* parse inbox */
	if (inbox && strlen(inbox))
	{
		slash = strrchr(inbox, '/');
		slash[0] = '\0';
		parse_folder(inbox, slash+1, 1);
	}

	return 0;
	/* It's no unused variable... See? Here it is... */
	(void)rcsid;
}
