/*
 * tf22.c
 * Copyright (C) 2002-2013, Ciprian Niculescu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <config.h>

#if defined HAVE_FEATURES_H
# include <features.h>
#endif				/* HAVE_FEATURES_H */

#include <getopt.h>
#include <signal.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/types.h>
#include <sys/wait.h>

#include <aime.h>
#if !defined HAVE_LIBx1f4i0
# include <integral.v-d.h>
#endif				/* !HAVE_LIBx1f4i0 */
#include <inter.h>

#define DATA_BITS \
    X1f4_LXDATA_EVERLINK

#define DATE_BITS \
    X1f4_LXDATE_EVERLINK

#define DECQ_BITS \
    X1f4_LXDECQ_EVERLINK | X1f4_LXDECQ_RESETNEW

#define FILE_BITS \
    X1f4_LXFILE_EVERLINK | X1f4_LXFILE_SETCLOSE

#define LINE_BITS \
    X1f4_LXLINE_EVERLINK | X1f4_LXLINE_RESETNEW

#define LIST_BITS \
    X1f4_LXLIST_EVERLINK | X1f4_LXLIST_RESETNEW

#define PORT_BITS \
    X1f4_LXPORT_EVERLINK | X1f4_LXPORT_SETCLOSE

#define A113_BITS \
    X1f4_A1_BCOLLECT | X1f4_A1_COMPOSER | X1f4_A1_DDACCESS | X1f4_A1_LEADCAST \
    | X1f4_A1_OPTIMIZE | X1f4_A1_SCOMMENT | X1f4_A1_SIDELIST		      \
    | X1f4_A1_TRANSFER | X1f4_A1_TYPELIST

#define HERE_13_OBJECT			(X1f4_E4_LAST + 256)

#define l_MODE(e, output) \
    {									      \
	X1f4_E4_C_MODE *l;						      \
									      \
	l = (void *) (output);						      \
	*l = (e);							      \
    }

typedef struct context_type {
    char **argv;
    const char *program, *self;
    int argc, class, error, exit, immediate,
	(*line) (struct context_type *, unsigned *);
    struct x1f4_indexset_type *indexset_data;
    unsigned call;
    void *logic;
    const void *data;
} context_type;

typedef struct library_type {
    int (*free) (void *, void *), sequence;
    struct x1f4_linetext_type functions[2];
    void *data;
} library_type;

static int copy_error(void *, const char *, unsigned);
static int copy_output(void *, const char *, unsigned);
static int ever_list(void *, void *, struct x1f4_variable_type *);
static int fini_sequence(void **);
static int fini_xset(struct context_type *);
static int flat_text(void *);
static int flush_error(struct context_type *, const char *);
static int fset_sequence(void *, const struct x1f4_linetext_type **);
static int init_xset(struct context_type *);
static int line_flat(void *);
static int list_text(void *, const void *);
static int mode_sequence(void *, void *, void **);
static int note_sequence(const struct x1f4_lxnote_type **);
static int pick_shuffle(struct context_type *, unsigned *);
static int post_flat(void *);
static int push_flat(void *, const char *, unsigned);
static int run_logic(struct context_type *, int, char **);
static int seek_line(struct context_type *, unsigned, unsigned *);
static int select_context(const void *, const struct x1f4_function_type *,
			  void **);
static int select_logic(const char *, unsigned, const void *,
			const struct x1f4_function_type **);
static int set_composer(struct context_type *, void **,
			int (**) (void *, int, int,
				  const struct x1f4_linetext_type **));
static int set_logic(struct context_type *, const char *, const void *);
static int test_cast(void *, const char *, unsigned);
static int tile_sequence(void **, unsigned, struct x1f4_lxwide_type *,
			 unsigned, const void *);
static int wait_some(struct context_type *);

static void SIGCHLD_handler(int);
static void list_0010(struct context_type *);
static void list_0220(struct context_type *);
static void usage(void);

static int *static_class;
static const int c_____m__[] = {
/* *INDENT-OFF* */
    X1f4_E4_MODE
/* *INDENT-ON* */
};
static sig_atomic_t SIGCHLD_received;
static struct x1f4_lxnear_type lxnear;
static const struct x1f4_lxnote_type sequence_note = {
/* *INDENT-OFF* */
    NULL,
    NULL,
    NULL,
    NULL,
    fini_sequence,
    fset_sequence,
    NULL,
    NULL,
    NULL,
    tile_sequence,
    NULL
/* *INDENT-ON* */
};
const static struct x1f4_lxtile_type tiles[] = {
/* *INDENT-OFF* */
    {	note_sequence,		0,		NULL		},
    {	x1f4_note_lxcast,	0,		NULL		},
    {	x1f4_note_lxdata,	DATA_BITS,	NULL		},
    {	x1f4_note_lxdate,	DATE_BITS,	NULL		},
    {	x1f4_note_lxdecq,	DECQ_BITS,	NULL		},
    {	x1f4_note_lxfile,	FILE_BITS,	NULL		},
    {	x1f4_note_lxline,	LINE_BITS,	NULL		},
    {	x1f4_note_lxlist,	LIST_BITS,	NULL		},
    {	x1f4_note_lxnear,	0,		&lxnear		},
    {	x1f4_note_lxport,	PORT_BITS,	NULL		},
    {	x1f4_note_lxtext,	0,		NULL		}
/* *INDENT-ON* */
};
static const struct x1f4_textpipe_type near_pipes[] = {
/* *INDENT-OFF* */
    {	"o",		copy_output,		1,	NULL	},
    {	"v",		copy_error,		1,	NULL	}
/* *INDENT-ON* */
};

static int
copy_error(void *none, const char *data, unsigned size)
{
    fwrite(data, size, 1, stderr);

    return 0;
}


static int
copy_output(void *none, const char *data, unsigned size)
{
    fwrite(data, size, 1, stdout);

    return 0;
}


static int
ever_list(void *none, void *data, struct x1f4_variable_type *variable_data)
{
    puts(variable_data->name);

    return 0;
}


static int
fini_sequence(void **sequence)
{
    int status;
    struct library_type *library_data;

    library_data = *sequence;
    if (library_data->free) {
	status = library_data->free(library_data->data, library_data);
    } else {
	free(library_data);
	status = 0;
    }

    return status;
}


static int
fini_xset(struct context_type *context_data)
{
    return x1f4_flat_indexset(context_data->indexset_data);
}


static int
flat_text(void *context)
{
    int status;
    struct context_type *context_data;
    struct x1f4_indexset_type *indexset_data;

    context_data = context;

    indexset_data = context_data->indexset_data;

    status = indexset_data->autodeck_set.deck
	(&indexset_data->autodeck_set.text);

    if (SIGCHLD_received) {
	int excess;

	excess = wait_some(context_data);
	if (excess) {
	    if (status) {
	    } else {
		status = excess;
	    }
	}
    }

    return status;
}


static int
flush_error(struct context_type *context_data, const char *program)
{
    if (context_data->error) {
    } else {
	fflush(stdout);
	fprintf(stderr, "%s:", context_data->self);
	if (program) {
	    fprintf(stderr, " %s:", program);
	}
	if (1) {
	    int (*line) (struct context_type *, unsigned *);

	    line = context_data->line;
	    if (line) {
		unsigned copy, miss;

		line(context_data, &copy);
		seek_line(context_data, copy, &miss);
		fprintf(stderr, " %u:", miss + 1);
	    }
	}

	fprintf(stderr, " cannot execute program\n");
    }

    return 17;
}


static int
fset_sequence(void *sequence, const struct x1f4_linetext_type **linetext)
{
    *linetext = ((struct library_type *) sequence)->functions;

    return 0;
}


static int
init_xset(struct context_type *context_data)
{
    struct x1f4_textport_type textport;
    unsigned textport_bits = 0;

    textport_bits = X1f4_TEXTPORT_CASTTYPE | X1f4_TEXTPORT_TEXTFLAT;

    textport.casttype_set.type = HERE_13_OBJECT;

    textport.lasttype_set.type = HERE_13_OBJECT + 256;

    textport.tilelong_set.lxtile_data = tiles;
    textport.tilelong_set.miss =
	sizeof(tiles) / sizeof(struct x1f4_lxtile_type);

    lxnear.nearpipe_set.textpipe_data = near_pipes;
    lxnear.nearpipe_set.miss = 2;

    textport.textflat_set.data = context_data;
    textport.textflat_set.line = line_flat;
    textport.textflat_set.post = post_flat;
    textport.textflat_set.push = push_flat;

    textport.autodeck_set.class = &context_data->class;

    return x1f4_fast_indexset
	(context_data->indexset_data, textport_bits, &textport);
}


static int
line_flat(void *context)
{
    struct context_type *context_data;

    context_data = context;

    fflush(stdout);
    fputs(context_data->self, stderr);
    fputs(":", stderr);
    if (context_data->program) {
	fputs(" ", stderr);
	fputs(context_data->program, stderr);
	fputs(":", stderr);
    }
    if (1) {
	int (*line) (struct context_type *, unsigned *);

	line = context_data->line;
	if (line) {
	    unsigned copy, miss;

	    line(context, &copy);
	    seek_line(context, copy, &miss);
	    fprintf(stderr, " %u:", miss + 1);
	}
    }

    fputs(" ", stderr);

    return 0;
}


static int
list_text(void *context, const void *text)
{
    x1f4_print_htfunction
	(stdout, text, "", ((struct context_type *) context)->indexset_data,
	 x1f4_look_indexset);

    return 0;
}


static int
mode_sequence(void *context, void *output, void **input)
{
    struct library_type *library_data;

    library_data = context;

    l_MODE(library_data->sequence, output);

    library_data->sequence++;

    return 0;
}


static int
note_sequence(const struct x1f4_lxnote_type **lxnote)
{
    *lxnote = &sequence_note;

    return 0;
}


static int
pick_shuffle(struct context_type *context_data, unsigned *copy)
{
    return x1f4_seek_shuffle(context_data->logic, copy);
}


static int
post_flat(void *context)
{
    ((struct context_type *) context)->error = 1;

    fputs("\n", stderr);

    return 0;
}


static int
push_flat(void *context, const char *data, unsigned size)
{
    fwrite(data, size, 1, stderr);

    return 0;
}


static int
run_logic(struct context_type *context_data, int argc, char **argv)
{
    int status;
    struct x1f4_indexset_type indexset;

    context_data->indexset_data = &indexset;

    status = init_xset(context_data);
    if (status) {
    } else {
	const char *program;

	program = argv[optind];
	optind++;

	SIGCHLD_received = 0;

	signal(SIGCHLD, SIGCHLD_handler);

	if (context_data->immediate) {
	    if (0) {
	    } else {
		context_data->data = program;

		context_data->program = NULL;

		status = set_logic(context_data, NULL, program);
	    }
	} else {
	    unsigned size;
	    void *data;

	    status = libx1f4i0_read_file(&data, &size, program, 1);
	    if (status) {
		libx1f4i0_stat_failure(argv[0], status, program);
	    } else {
		if (1) {
		    char *mine;

		    mine = data;
		    mine[size] = 0;
		}

		context_data->data = data;

		context_data->program = program;

		status = set_logic(context_data, program, data);

		free(data);
	    }
	}

	signal(SIGCHLD, SIG_DFL);

	if (1) {
	    int excess;

	    excess = fini_xset(context_data);
	    if (excess) {
		if (status) {
		} else {
		    status = excess;
		}
	    }
	}
    }

    return status;
}


static int
seek_line(struct context_type *context_data, unsigned copy, unsigned *miss)
{
    const char *data;
    unsigned line = 0;

    data = context_data->data;
    while (copy) {
	if (*data == 10) {
	    line++;
	}

	copy--;
	data++;
    }

    *miss = line;

    return 0;
}


static int
select_context(const void *context, const struct x1f4_function_type *miss,
	       void **text)
{
    *text = (void *)
	((const struct x1f4_linetext_type *)
	 ((const char *) miss
	  - offsetof(struct x1f4_linetext_type, function)))->context;

    return 0;
}


static int
select_logic(const char *f, unsigned length, const void *context,
	     const struct x1f4_function_type **function)
{
    int status;
    struct context_type *context_data;
    const void *mind;

    context_data = (void *) context;

    status = x1f4_find_mxpath
	(context_data->indexset_data->function_set.text, f, length, &mind);
    if (status) {
	status = 0;

	*function = mind;
    } else {
	status = 1;
    }

    return status;
}


static int
set_composer(struct context_type *context_data, void **text,
	     int (**lock) (void *, int, int,
			   const struct x1f4_linetext_type **))
{
    struct x1f4_lxslip_type *lxslip_data;

    lxslip_data = context_data->indexset_data->sliplong_set.lxslip_data;
    while (lxslip_data->note != x1f4_note_lxcast) {
	lxslip_data++;
    }

    *lock = x1f4_lock_lxcast;

    *text = lxslip_data->slip;

    return 0;
}


static int
set_logic(struct context_type *context_data, const char *program,
	  const void *data)
{
    int status;
    struct x1f4_a1_type a1;
    struct x1f4_a1record_type a1record;
    void *proGram;

    a1.autolead_set.context = context_data->indexset_data->autodeck_set.less;
    a1.autolead_set.link = x1f4_link_e4lf;
    a1.autolead_set.pick = x1f4_pick_e4lf;
    a1.bcollect_set.a1record_data = &a1record;
    set_composer(context_data, &a1.composer_set.context, &a1.composer_set.get);
    a1.datatype_set.miss =
	context_data->indexset_data->datatype_set.datatype_data;
    a1.function_set.fix = select_context;
    a1.function_set.get = select_logic;
    a1.function_set.context = context_data;
    x1f4_llink_operator1s(&a1.operator1s);
    a1.operator2s = context_data->indexset_data->operator_set.operator2s;
    x1f4_pset_lxcast
	((void *) a1.composer_set.context, (void *) &a1.sidetype_set.miss);
    a1.transfer_set.fine = context_data->indexset_data->transfer_set.fine;
    a1.variable_set.context = context_data->indexset_data->variable_set.text;

    status = x1f4_init_shuffle(&proGram, data, A113_BITS, &a1);
    if (status) {
	if (status == X1f4_A1_ALLOC_ERROR) {
	    perror(context_data->self);
	} else {
	    struct x1f4_eelookup_type eelookup;

	    fprintf(stderr, "%s: cannot parse ", context_data->self);
	    if (program) {
		fprintf(stderr, "`%s'", program);
	    } else {
		fprintf(stderr, "program");
	    }
	    fprintf(stderr, "\n");
	    fprintf(stderr, "%s: ", context_data->self);
	    eelookup.type_l.context = context_data->indexset_data;
	    eelookup.type_l.fix = x1f4_look_indexset;
	    eelookup.type_q.e4fine =
		context_data->indexset_data->transfer_set.fine;
	    a1record.pick = 1;
	    x1f4_stat_shuffle(stderr, test_cast, data, &a1record, &eelookup);
	    fprintf(stderr, "\n");
	}
    } else {
	status = x1f4_pipe_e4ll
	    (context_data->indexset_data->autodeck_set.less, proGram);
	if (status) {
	    perror(context_data->self);
	} else {
	    X1f4_E4_C_MODE degree;

	    context_data->class = 0;
	    context_data->error = 0;

	    context_data->exit = 0;

	    context_data->line = pick_shuffle;

	    context_data->logic = proGram;

	    static_class = &context_data->class;

	    status = x1f4_long_shuffle
		(proGram, &context_data->class, context_data,
		 flat_text, &degree);
	    if (status) {
		if (context_data->exit) {
		    status = context_data->exit >> 1;
		} else {
		    flush_error(context_data, program);
		}
	    } else {
		status = degree;
	    }
	}

	x1f4_fini_shuffle(&proGram);
    }

    return status;
}


static int
test_cast(void *cast, const char *name, unsigned size)
{
    return size ^ fwrite(name, 1, size, cast);
}


static int
tile_sequence(void **sequence, unsigned wide_bits,
	      struct x1f4_lxwide_type *lxwide_data, unsigned bits,
	      const void *data)
{
    int status;
    struct library_type *library_data;

    if (wide_bits & X1f4_LXWIDE_CODELINK) {
	void *library;

	status = lxwide_data->codelink_set.link
	    (lxwide_data->codelink_set.data, &library,
	     sizeof(struct library_type));
	if (status) {
	} else {
	    library_data = library;
	    library_data->data = lxwide_data->codelink_set.data;
	    library_data->free = lxwide_data->codelink_set.free;
	}
    } else {
	library_data = malloc(sizeof(struct library_type));
	if (library_data) {
	    status = 0;
	    library_data->free = NULL;
	} else {
	    status = -1;
	}
    }
    if (status) {
    } else {
	library_data->sequence = 0;

	library_data->functions[0].context = library_data;
	library_data->functions[0].function.args = NULL;
	library_data->functions[0].function.count = 0;
	library_data->functions[0].function.flags =
	    X1f4_E4_KEEP_CALL | X1f4_E4_TEXT_LINK;
	library_data->functions[0].function.function = mode_sequence;
	library_data->functions[0].function.length = 8;
	library_data->functions[0].function.name = "sequence";
	library_data->functions[0].function.type = X1f4_E4_MODE;

	library_data->functions[1].function.name = NULL;

	*sequence = library_data;
    }

    return status;
}


static int
wait_some(struct context_type *context_data)
{
    int status;
    struct x1f4_lxslip_type *lxslip_data;
    void *lxport;

    SIGCHLD_received = 0;

    lxslip_data = context_data->indexset_data->sliplong_set.lxslip_data;
    while (lxslip_data->note != x1f4_note_lxport) {
	lxslip_data++;
    }

    lxport = lxslip_data->slip;

    while (1) {
	pid_t wait;

	wait = waitpid(-1, NULL, WNOHANG);
	if (wait == -1) {
	    status = 0;
	    if (1) {
		break;
	    }
	} else {
	    if (wait) {
		x1f4_wait_lxport(lxport, wait);
	    } else {
		status = 0;
		if (1) {
		    break;
		}
	    }
	}
    }

    return status;
}


static void
SIGCHLD_handler(int signal)
{
    SIGCHLD_received = 1;
    *static_class = 1;
}


static void
list_0010(struct context_type *context_data)
{
    struct x1f4_indexset_type indexset;

    context_data->indexset_data = &indexset;

    context_data->argv = (void *) context_data;

    if (init_xset(context_data)) {
    } else {
	x1f4_lime_mxdeck(indexset.function_set.text, context_data, list_text);

	fini_xset(context_data);
    }
}


static void
list_0220(struct context_type *context_data)
{
    struct x1f4_indexset_type indexset;

    context_data->indexset_data = &indexset;

    context_data->argv = (void *) context_data;

    if (init_xset(context_data)) {
    } else {
	x1f4_list_state(indexset.variable_set.text, NULL, ever_list);

	fini_xset(context_data);
    }
}


static void
usage(void)
{
    puts("Usage: tf22 [OPTIONS] PROGRAM\n\
Execute PROGRAM.\n\
\n\
Options:\n\
  -c				execute the PROGRAM program instead program\n\
				read from the PROGRAM file\n\
      --data			list defined constants and exit\n\
      --list			list available functions and exit\n\
      --help			display this help and exit\n\
      --version			output version information and exit");
}


int
main(int argc, char **argv)
{
    int list_functions = 0, status = 0;
    struct context_type context;

    context.immediate = 0;

    unsetenv("POSIXLY_CORRECT");

    {
	int fast = ~0;

	while (1) {
	    char c;
	    static struct option long_options[] = {
/* *INDENT-OFF* */
		{   "data",	    0x00,   NULL,   0x02    },
		{   "help",	    0x00,   NULL,   0x68    },
		{   "list",	    0x00,   NULL,   0x01    },
		{   "print",	    0x00,   NULL,   0x50    },
		{   "version",	    0x00,   NULL,   0x76    },
		{   NULL,	    0x00,   NULL,   0x00    }
/* *INDENT-ON* */
	    };

	    c = getopt_long(argc, argv, "c", long_options, NULL);

	    if (!~c) {
		break;
	    }

	    switch (c) {
	    case 001:
		list_functions = 1;
		break;
	    case 002:
		list_functions = 2;
		break;
	    case 'c':
		context.immediate = 1;
		break;
	    case 'h':
	    case 'v':
		if (!~fast) {
		    fast = c;
		}
		break;
	    case '?':
		return 1;
	    }
	}

	if (~fast) {
	    switch (fast) {
	    case 'h':
		usage();
		break;
	    case 'v':
		printf("%s (%s) %s\n", argv[0], PACKAGE, VERSION);
	    }

	    return 0;
	}
    }

    context.argc = argc - optind;
    context.argv = argv + optind;

    if (list_functions) {
	if (list_functions == 1) {
	    list_0010(&context);
	} else {
	    if (list_functions == 2) {
		list_0220(&context);
	    }
	}
    } else {
	if (context.argv ? !(argc - optind) : (argc - optind) % 3 != 1) {
	    fprintf(stderr, "%s: wrong number of arguments\nTry `%s --help' "
		    "for more information.\n", argv[0], argv[0]);

	    return 1;
	} else {
	    context.self = argv[0];

	    status = run_logic(&context, argc, argv);
	}
    }

    return status;
}
