/*
 * tf17.c
 * Copyright (C) 2002-2013, Ciprian Niculescu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <config.h>

#if defined HAVE_FEATURES_H
# include <features.h>
#endif				/* HAVE_FEATURES_H */

#include <getopt.h>
#if defined HAVE_LIBx1f4l0
# include <libx1f4l0.h>
#endif				/* HAVE_LIBx1f4l0 */
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <aime.h>
#include <inter.h>
#include <sf.h>
#include <types.h>

typedef struct context_type {
    const char *self;
    unsigned slip;
} context_type;

extern const struct x1f4_function_type _libx1f4i0_e4_io[];

static int fast_program(void *, const struct x1f4_function_type *);
static int link_program(const char *, void *, const char *);
static int lose_program(void *, const struct x1f4_function_type *);
static int merge_sets(struct x1f4_function_type **);
static int test_cast(void *, const char *, unsigned);

static void usage(void);

static int
fast_program(void *text, const struct x1f4_function_type *function_data)
{
    int delete;
    struct context_type *context_data;

    context_data = text;
    context_data->slip++;

    delete = context_data->slip < 1024 ? 0 : 1;

    if (delete) {
	fprintf(stderr, "%s: terminating execution for the function call dep"
		"th hit the 1024 limit\n", context_data->self);
    }

    return delete;
}


static int
link_program(const char *self, void *proGram, const char *program)
{
    int status;
    struct context_type context;
    struct x1f4_a1screen_type screen;

    context.self = self;
    context.slip = 0;

    screen.data = &context;

    screen.back = lose_program;
    screen.fast = fast_program;
    screen.lose = lose_program;

    status = x1f4_pipe_shuffle(proGram, &screen);
    if (status) {
    } else {
	status = x1f4_link_shuffle(proGram);
    }

    return status;
}


static int
lose_program(void *text, const struct x1f4_function_type *function_data)
{
    struct context_type *context_data;

    context_data = text;
    context_data->slip--;

    return 0;
}


static int
merge_sets(struct x1f4_function_type **set)
{
    int status = 0;
    struct x1f4_function_type *function_data;
    unsigned count, trans;

    x1f4_count_functions(x1f4_e4_defaults, &count);
    x1f4_count_functions(_libx1f4i0_e4_io, &trans);
    function_data = (struct x1f4_function_type *)
	malloc((count + trans + 1) * sizeof(struct x1f4_function_type));
    if (!function_data) {
	status = 1;
    } else {
	*set = function_data;
	memcpy(function_data, x1f4_e4_defaults,
	       count * sizeof(struct x1f4_function_type));
	function_data += count;
	memcpy(function_data, _libx1f4i0_e4_io,
	       (trans + 1) * sizeof(struct x1f4_function_type));
    }

    return status;
}


static int
test_cast(void *cast, const char *name, unsigned size)
{
    return 1 ^ fwrite(name, size, 1, cast);
}


static void
usage(void)
{
    puts("Usage: tf14 [OPTIONS] PATH [TYPE NAME VALUE]\n\
Execute program read from PATH.\n\
\n\
Options:\n\
  -M, --stat-storage		stat program memory storage requirements\n\
  -m, --stat-memory		stat memory operations\n\
  -o, --optimize		enable optimizations\n\
      --help			display this help and exit\n\
      --version			output version information and exit");
}


int
main(int argc, char **argv)
{
    int do_memory = 0, do_optimize = 0, do_storage = 0, status = 0;

    unsetenv("POSIXLY_CORRECT");

    {
	int fast = ~0;

	while (1) {
	    char c;
	    static struct option long_options[] = {
/* *INDENT-OFF* */
		{   "help",	    0x00,   NULL,   0x68    },
		{   "optimize",	    0x00,   NULL,   0x6f    },
		{   "stat-memory",  0x00,   NULL,   0x6d    },
		{   "stat-storage", 0x00,   NULL,   0x4d    },
		{   "version",	    0x00,   NULL,   0x76    },
		{   NULL,	    0x00,   NULL,   0x00    }
/* *INDENT-ON* */
	    };

	    c = getopt_long(argc, argv, "Mmo", long_options, NULL);

	    if (!~c) {
		break;
	    }

	    switch (c) {
	    case 'M':
		do_storage = 1;
		break;
	    case 'h':
	    case 'v':
		if (!~fast) {
		    fast = c;
		}
		break;
	    case 'm':
		do_memory = 1;
		break;
	    case 'o':
		do_optimize = 1;
		break;
	    case '?':
		return 1;
	    }
	}

	if (~fast) {
	    switch (fast) {
	    case 'h':
		usage();
		break;
	    case 'v':
		printf("%s (%s) %s\n", argv[0], PACKAGE, VERSION);
	    }

	    return 0;
	}
    }

    if ((argc - optind) % 3 != 1) {
	fprintf(stderr, "%s: wrong number of arguments\nTry `%s --help' for "
		"more information.\n", argv[0], argv[0]);

	return 1;
    } else {
	unsigned size;
	void *context = NULL, *data, *state_data;

	do {
	    const char *program;
	    unsigned count;

	    program = argv[optind];
	    optind++;

	    status = libx1f4i0_init_fine
		(optind, argc, argv, &count, &state_data, &context);
	    if (status) {
		break;
	    }

	    status = libx1f4i0_read_file(&data, &size, program, 1);
	    if (status) {
		libx1f4i0_stat_failure(argv[0], status, program);
	    } else {
		struct x1f4_function_type *function_data;

		if (merge_sets(&function_data)) {
		    status = 1;
		    perror(argv[0]);
		} else {
#if defined HAVE_LIBx1f4l0
		    struct list_type dlist;
#endif				/* HAVE_LIBx1f4l0 */
		    struct x1f4_a1_type a1;
		    struct x1f4_a1record_type a1record;
		    unsigned flags = X1f4_A1_BCOLLECT | X1f4_A1_DDACCESS;
		    void *proGram;
#if defined HAVE_LIBx1f4l0
		    void *valist = NULL;
#endif				/* HAVE_LIBx1f4l0 */

		    {
			char *mine;

			mine = data;
			mine[size] = 0;
		    }

		    if (do_optimize) {
			flags |= X1f4_A1_OPTIMIZE;
		    }

#if defined HAVE_LIBx1f4l0
		    libx1f4i0_init_valist(&valist, do_memory, do_storage);
#endif				/* HAVE_LIBx1f4l0 */

		    a1.function_set.get = libx1f4i0_select_function;
		    a1.function_set.context = function_data;
		    x1f4_llink_operator1s(&a1.operator1s);
		    x1f4_llink_operator2s(&a1.operator2s);
		    a1.variable_set.context = context;

		    a1.bcollect_set.a1record_data = &a1record;

#if defined HAVE_LIBx1f4l0
		    if (valist) {
			flags |= X1f4_A1_RESOURCE;
			libx1f4i0_line_valist
			    (valist, do_memory, do_storage,
			     &a1.resource_set.free, &a1.resource_set.link,
			     &a1.resource_set.mode, &a1.resource_set.context,
			     &dlist);
		    }
#endif				/* HAVE_LIBx1f4l0 */

		    status = x1f4_init_shuffle(&proGram, data, flags, &a1);
		    if (status) {
			if (status == X1f4_A1_ALLOC_ERROR) {
			    perror(argv[0]);
			} else {
			    fprintf(stderr, "%s: cannot parse `%s'\n", argv[0],
				    program);
			    fprintf(stderr, "%s: ", argv[0]);
			    a1record.pick = 1;
			    x1f4_stat_shuffle
				(stderr, test_cast, data, &a1record, NULL);
			    fprintf(stderr, "\n");
			}
		    } else {
			status = link_program(argv[0], proGram, program);

#if defined HAVE_LIBx1f4l0
			if (valist) {
			    if (do_storage) {
				unsigned size;

				x1f4_size_xalist(valist, &size);
				fprintf(stderr, " %7u\n", size);
			    }
			}
#endif				/* HAVE_LIBx1f4l0 */

			x1f4_fini_shuffle(&proGram);
		    }

#if defined HAVE_LIBx1f4l0
		    if (valist) {
			libx1f4i0_fini_valist
			    (argv[0], valist, do_memory, do_storage, &status);
		    }
#endif				/* HAVE_LIBx1f4l0 */

		    free(function_data);
		}

		free(data);
	    }
	} while (0);

	if (context) {
	    x1f4_air_state(context);
	}

	if (state_data) {
	    free(state_data);
	}
    }

    return status;
}
