/*
 * text.0.c
 * Copyright (C) 2011-2012, Ciprian Niculescu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <errno.h>
#include <fcntl.h>
#if defined HAVE_LIBx1f4l0
# include <libx1f4l0.h>
#endif				/* HAVE_LIBx1f4l0 */
#include <signal.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <sys/select.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>

#if !defined HAVE_LIBx1f4i0
# include <command-e.h>
#endif				/* !HAVE_LIBx1f4i0 */
#include <xshell.h>

#define PROGRESS			4096

#define PORT_BITS \
    X1f4_XSHELL_ARGV_MASK | X1f4_XSHELL_HEAD_MASK | X1f4_XSHELL_LOOK_MASK     \
    | X1f4_XSHELL_PATH_MASK | X1f4_XSHELL_TAIL_MASK | X1f4_XSHELL_TYPE_MASK

static int head_shell(void *);
static int land_shell(const char *, const char *, int *, int *, pid_t *);
static int lead_shell(const char *, const char **, int *, int *, pid_t *);
static int link_frame(const char *, void **);
static int lock_shell(int);
static int lose_shell(const char *, int);
static int miss_shell(const char *, int, int);
static int tail_shell(void *);
static int text_shell(const char *, int, int *, const char *, void **,
		      pid_t *, int);
static int type_shell(void *, const char *, unsigned);
static int wait_shell(pid_t *);

static void SIGCHLD_handler(int);

static sig_atomic_t SIGCHLD_received;

static int
head_shell(void *text)
{
    fprintf(stderr, "%s: ", (char *) text);

    return 0;
}


static int
land_shell(const char *self, const char *preprocessor, int *fd_0, int *fd_1,
	   pid_t *wait)
{
    int status;
    char **items;
    unsigned count;

    status = x1f4_esplit_command(preprocessor, 0, 1, &count, &items);
    if (status) {
	if (status < 0) {
	    perror(self);
	} else {
	    fprintf(stderr, "%s: cannot parse command: `%s'\n", self,
		    preprocessor);
	}
    } else {
	items[count] = NULL;

	status = lead_shell(self, (const char **) items, fd_0, fd_1, wait);

	free(items[0]);
	free(items);
    }

    return status;
}


static int
lead_shell(const char *self, const char **argv, int *fd_0, int *fd_1,
	   pid_t *wait)
{
    int status;
    void *xshell;

    status = x1f4_init_xshell(&xshell);
    if (status) {
    } else {
	const char *static_argv[2];
	int excess;
	struct x1f4_xshell_type Xshell;
	unsigned bits = PORT_BITS;

	if (argv) {
	    Xshell.command.argv = argv;
	    Xshell.command.path = argv[0];
	} else {
	    Xshell.command.argv = static_argv;
	    Xshell.command.path = "cpp";

	    static_argv[0] = "cpp";
	    static_argv[1] = NULL;
	}

	Xshell.printer.head = head_shell;
	Xshell.printer.look = (void *) self;
	Xshell.printer.tail = tail_shell;
	Xshell.printer.type = type_shell;

	status = x1f4_mode_xshell(xshell, &Xshell, bits);
	if (status) {
	} else {
	    status = x1f4_deck_xshell(xshell);
	    if (status) {
	    } else {
		x1f4_hook_xshell(xshell, wait);

		x1f4_pipe_xshell(xshell, fd_0, fd_1);
	    }
	}

	excess = x1f4_fini_xshell(&xshell);
	if (excess) {
	    if (status) {
	    } else {
		status = excess;
	    }
	}
    }

    return status;
}


static int
link_frame(const char *self, void **miss)
{
    int status;
    void *frame;

    frame = malloc(PROGRESS + 1);
    if (frame) {
	status = 0;
	*miss = frame;
    } else {
	status = 1;
	perror(self);
    }

    return status;
}


static int
lock_shell(int fd)
{
    int flags, status;

    flags = fcntl(fd, F_GETFL);
    if (flags == -1) {
	status = 1;
    } else {
	status = fcntl(fd, F_SETFL, flags | O_NONBLOCK);
    }

    return status;
}


static int
lose_shell(const char *self, int fd)
{
    int status;

    status = close(fd);
    if (status) {
	fprintf(stderr, "%s: cannot close preprocessor pipe: %s\n",
		self, strerror(errno));
    }

    return status;
}


static int
miss_shell(const char *self, int fd_0, int fd_1)
{
    int status = 0;

    if (fd_0 ^ -1) {
	if (lose_shell(self, fd_0)) {
	    status = 1;
	}
    }

    if (fd_1 ^ -1) {
	if (lose_shell(self, fd_1)) {
	    status = 1;
	}
    }

    return status;
}


static int
tail_shell(void *text)
{
    fprintf(stderr, "\n");

    return 0;
}


static int
text_shell(const char *self, int fd_0, int *fd_j, const char *text,
	   void **miss, pid_t *wait, int lock)
{
    char *frame;
    fd_set fd_set_0, fd_set_1, *set_1;
    int fd_1, fd_i, status;
    unsigned call = PROGRESS, off = PROGRESS, size;

    size = lock ? 1 : strlen(text);

    set_1 = &fd_set_1;

    frame = *miss;

    fd_1 = *fd_j;

    fd_i = fd_0;
    if (fd_i < fd_1) {
	fd_i = fd_1;
    }

    fd_i++;

    while (1) {
	FD_ZERO(&fd_set_0);
	FD_SET(fd_0, &fd_set_0);

	if (*text) {
	    FD_ZERO(&fd_set_1);
	    FD_SET(fd_1, &fd_set_1);
	} else {
	    if (set_1) {
		set_1 = NULL;
		fd_i = fd_0 + 1;
		status = lose_shell(self, fd_1);
		*fd_j = -1;
		if (status) {
		    break;
		}
	    }
	}

	if (SIGCHLD_received) {
	    SIGCHLD_received = 0;
	    status = wait_shell(wait);
	    if (status) {
		break;
	    }
	}

	status = select(fd_i, &fd_set_0, set_1, NULL, NULL);
	if (status == -1) {
	    if (errno != EINTR) {
		fprintf(stderr, "%s: cannot select preprocessor: %s\n", self,
			strerror(errno));
		break;
	    } else {
	    }
	} else {
	    if (FD_ISSET(fd_0, &fd_set_0)) {
		if (off) {
		} else {
		    off = call;
		    call <<= 1;
		    frame = realloc(*miss, call + 1);
		    if (frame) {
			*miss = frame;
			frame += off;
		    } else {
			perror(self);

			status = -1;

			break;
		    }
		}

		status = read(fd_0, frame, lock ? 1 : off);
		if (status == -1) {
		    if (errno != EINTR) {
			fprintf(stderr, "%s: cannot read preprocessor output"
				": %s\n", self, strerror(errno));
			break;
		    } else {
		    }
		} else {
		    if (status) {
			frame += status;
			off -= status;
		    } else {
			*frame = 0;

			break;
		    }
		}
	    }

	    if (set_1) {
		if (FD_ISSET(fd_1, &fd_set_1)) {
		    status = write(fd_1, text, size);
		    if (status == -1) {
			if (errno != EINTR) {
			    fprintf(stderr, "%s: cannot write preprocessor: "
				    "%s\n", self, strerror(errno));
			    break;
			} else {
			}
		    } else {
			if (status) {
			    if (lock) {
				text++;
			    } else {
				size -= status;
				text += status;
			    }
			}
		    }
		}
	    }
	}
    }

    return status;
}


static int
type_shell(void *text, const char *hell, unsigned size)
{
    fwrite(hell, 1, size, stderr);

    return 0;
}


static int
wait_shell(pid_t *wait)
{
    int status;
    pid_t pick, slip;

    slip = *wait;

    pick = waitpid(slip, &status, WNOHANG);
    if (pick ^ slip) {
	status = 0;
    } else {
	*wait = -1;
	if (WIFEXITED(status)) {
	    if (WEXITSTATUS(status)) {
		status = -1;
	    } else {
		status = 0;
	    }
	} else {
	    status = -1;
	}
    }

    return status;
}


static void
SIGCHLD_handler(int signal)
{
    SIGCHLD_received = 1;
}


int
libx1f4i0_cell_text(const char *self, const char *preprocessor,
		    const void *text, void **pick)
{
    int fd_0 = -1, fd_1 = -1, status;
    pid_t wait = -1;
    void (*andler) (int), *miss = NULL;

    SIGCHLD_received = 0;

    andler = signal(SIGCHLD, SIGCHLD_handler);

    if (preprocessor) {
	status = land_shell(self, preprocessor, &fd_0, &fd_1, &wait);
    } else {
	status = lead_shell(self, NULL, &fd_0, &fd_1, &wait);
    }
    if (status) {
    } else {
	status = link_frame(self, &miss);
	if (status) {
	} else {
	    status = text_shell
		(self, fd_0, &fd_1, text, &miss, &wait,
		 lock_shell(fd_0) | lock_shell(fd_1));
	}
    }

    if (miss_shell(self, fd_0, fd_1)) {
	status = 1;
    }

    if (wait ^ -1) {
	kill(wait, SIGKILL);
    }

    signal(SIGCHLD, andler);

    if (wait ^ -1) {
	if (wait_shell(&wait)) {
	    status = 1;
	}
    }

    if (status) {
	if (miss) {
	    free(miss);
	}
    } else {
	*pick = miss;
    }

    return status;
}
