/*
 * express33.c
 * Copyright (C) 2006-2013, Ciprian Niculescu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <config.h>

#if defined HAVE_FEATURES_H
# include <features.h>
#endif				/* HAVE_FEATURES_H */

#include <getopt.h>
#if defined HAVE_LIBx1f4i0
# include <libx1f4i0.h>
#endif				/* HAVE_LIBx1f4i0 */
#if defined HAVE_LIBx1f4l0
# include <libx1f4l0.h>
#endif				/* HAVE_LIBx1f4l0 */
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <aime.h>
#if !defined HAVE_LIBx1f4i0
# include <cardinal-wx.h>
# include <float1.h>
#endif				/* !HAVE_LIBx1f4i0 */
#include <inter.h>
#include <types.h>
#if !defined HAVE_LIBx1f4i0
# include <vkeylook-l.h>
# include <vtkey.h>
#endif				/* !HAVE_LIBx1f4i0 */

#define __EXTERN_STACKS__		0

#define __HAIRY_INTERFACE__		0
#define __STRAIGHT_INTERFACE__		1

#define __CAST_DUMP__			0
#define __FILE_DUMP__			1

#define __COPY_DATA__			0

#define __DUMP_TEST__			__CAST_DUMP__

#define __INTERFACE_TEST__		__HAIRY_INTERFACE__

#define BLOCK_SIZE			4096

typedef struct context_type {
    struct x1f4_variable_type *variable_data;
#if __INTERFACE_TEST__ == __HAIRY_INTERFACE__
    struct x1f4_vkeytree_type *variable_tree;
#endif				/* __INTERFACE_TEST__ == __HAIRY_INTERFACE__ */
    void **state;
} context_type;

typedef struct express_type {
    char delimiter;
    int do_memory, do_optimize, do_print, do_storage;
    struct x1f4_variable_type *variable_data;
    unsigned precision, print_flags;
    void **state;
} express_type;

#if __INTERFACE_TEST__ == __HAIRY_INTERFACE__
static int init_attributes(struct x1f4_attributes_type *);
static int init_functions(struct x1f4_attributes_type *);
static int init_variables(struct x1f4_attributes_type *);
#endif				/* __INTERFACE_TEST__ == __HAIRY_INTERFACE__ */
static int miss_fine(const char *, const char **, struct express_type *);
static int miss_line(const char *, const char *, struct express_type *);
static int miss_text(const char *, int, struct express_type *);
static int select_function(const char *, unsigned, const void *,
			   const struct x1f4_function_type **);
static int select_variable(const char *, unsigned, const void *,
			   const struct x1f4_variable_type **, void **);
#if __DUMP_TEST__ == __CAST_DUMP__
static int test_cast(void *, const char *, unsigned);
#endif				/* __DUMP_TEST__ == __CAST_DUMP__ */

#if __INTERFACE_TEST__ == __HAIRY_INTERFACE__
static void fini_attributes(struct x1f4_attributes_type *);
static void fini_functions(struct x1f4_attributes_type *);
static void fini_variables(struct x1f4_attributes_type *);
#endif				/* __INTERFACE_TEST__ == __HAIRY_INTERFACE__ */
static void usage(void);

#if __INTERFACE_TEST__ == __HAIRY_INTERFACE__
static int
init_attributes(struct x1f4_attributes_type *attributes_data)
{
    int status = 1;

    do {
	if (!init_functions(attributes_data)) {
	    if (!init_variables(attributes_data)) {
		status = 0;
		break;
	    }

	    fini_functions(attributes_data);
	}
    } while (0);

    return status;
}
#endif				/* __INTERFACE_TEST__ == __HAIRY_INTERFACE__ */


#if __INTERFACE_TEST__ == __HAIRY_INTERFACE__
static int
init_functions(struct x1f4_attributes_type *attributes_data)
{
    int status;
    const struct x1f4_function_type *function_data;
    struct x1f4_vkeytree_type *root;

    function_data = x1f4_e4_defaults;
    status = x1f4_make_vtkeyroot(&root, function_data->name, function_data);
    if (status) {
	status = X1f4_E4_ALLOC_ERROR;
    } else {
	function_data++;
	while (function_data->name) {
	    status = x1f4_make_vtkeynode
		(&root, function_data->name, function_data);
	    if (status) {
		status = X1f4_E4_ALLOC_ERROR;
		break;
	    } else {
		function_data++;
	    }
	}
    }

    if (status) {
	x1f4_free_vtkeydata(root);
    } else {
	attributes_data->function_set.context = root;
    }

    return status;
}
#endif				/* __INTERFACE_TEST__ == __HAIRY_INTERFACE__ */


#if __INTERFACE_TEST__ == __HAIRY_INTERFACE__
static int
init_variables(struct x1f4_attributes_type *attributes_data)
{
    int status;
    struct context_type *context_data;
    const struct x1f4_variable_type *variable_data;

    context_data = (void *) attributes_data->variable_set.context;
    variable_data = context_data->variable_data;
    if (!context_data->variable_data) {
	context_data->variable_tree = NULL;
	status = 0;
    } else {
	struct x1f4_vkeytree_type *root;

	status = x1f4_make_vtkeyroot
	    (&root, variable_data->name, variable_data);
	if (status) {
	    status = X1f4_E4_ALLOC_ERROR;
	} else {
	    variable_data++;
	    while (variable_data->name) {
		status = x1f4_make_vtkeynode
		    (&root, variable_data->name, variable_data);
		if (status) {
		    status = X1f4_E4_ALLOC_ERROR;
		    break;
		} else {
		    variable_data++;
		}
	    }
	}

	if (status) {
	    x1f4_free_vtkeydata(root);
	} else {
	    ((struct context_type *)
	     attributes_data->variable_set.context)->variable_tree = root;
	}
    }

    return status;
}
#endif				/* __INTERFACE_TEST__ == __HAIRY_INTERFACE__ */


static int
miss_fine(const char *self, const char **line,
	  struct express_type *express_data)
{
    const char *expression;
    int status;
#if defined HAVE_LIBx1f4l0
    struct list_type dlist;
#endif				/* HAVE_LIBx1f4l0 */
    unsigned flags = X1f4_E4_DDACCESS;
    void *x1f4_expression;
#if defined HAVE_LIBx1f4l0
    void *valist = NULL;
#endif				/* HAVE_LIBx1f4l0 */

    expression = *line;

#if defined HAVE_LIBx1f4l0
    libx1f4i0_init_valist
	(&valist, express_data->do_memory, express_data->do_storage);
#endif				/* HAVE_LIBx1f4l0 */

    {
	struct context_type context;
	struct x1f4_attributes_type attributes;
#if __EXTERN_STACKS__
	unsigned s1st = 0, s2nd = 0;
#endif				/* __EXTERN_STACKS__ */

	context.state = express_data->state;
	context.variable_data = express_data->variable_data;

	attributes.function_set.get = select_function;
#if __INTERFACE_TEST__ == __STRAIGHT_INTERFACE__
	attributes.function_set.context = x1f4_e4_defaults;
#endif				/* __INTERFACE_TEST__ == __STRAIGHT_INTER... */
	x1f4_llink_operator1s(&attributes.operator1s);
	x1f4_llink_operator2s(&attributes.operator2s);
	attributes.terminator = express_data->delimiter;
	attributes.variable_set.get = select_variable;
	attributes.variable_set.context = &context;

	flags |= X1f4_E4_ESTORAGE;

#if defined HAVE_LIBx1f4l0
	if (valist) {
	    flags |= X1f4_E4_RESOURCE;
	    libx1f4i0_line_valist
		(valist, express_data->do_memory, express_data->do_storage,
		 &attributes.resource_set.free,
		 &attributes.resource_set.link,
		 &attributes.resource_set.mode,
		 &attributes.resource_set.context, &dlist);
	}
#endif				/* HAVE_LIBx1f4l0 */

#if __INTERFACE_TEST__ == __HAIRY_INTERFACE__
	do {
	    status = init_attributes(&attributes);
	    if (status) {
		break;
	    }
#endif				/* __INTERFACE_TEST__ == __HAIRY_INTERFACE__ */

#if __EXTERN_STACKS__
	    flags |= X1f4_E4_MAXSTACK;
#endif				/* __EXTERN_STACKS__ */

#if __EXTERN_STACKS__
	    attributes.internal_set.s1st = &s1st;
	    attributes.internal_set.s2nd = &s2nd;
#endif				/* __EXTERN_STACKS__ */

	    status = x1f4_inIt_expression
		(&x1f4_expression, line, flags, &attributes);

#if __COPY_DATA__
	    if (!status) {
		void *expression;

		status = x1f4_copy_expression(&expression, x1f4_expression);
		if (!status) {
		    x1f4_fini_expression(&x1f4_expression);
		    x1f4_expression = expression;
		}
	    }
#endif				/* __COPY_DATA__ */

#if __EXTERN_STACKS__
	    if (!status) {
		stack_free = attributes.resource_set.free;
		stack_text = attributes.resource_set.context;

		status = attributes.resource_set.link(stack_text, &t1st, s1st);
		if (status) {
		} else {
		    status = attributes.resource_set.link
			(stack_text, &t2nd, s2nd);
		    if (status) {
			stack_free(stack_text, t1st);
		    }
		}

		if (status) {
		    x1f4_fini_expression(&x1f4_expression);
		} else {
		    x1f4_miss_expression(x1f4_expression, t1st, t2nd);
		}
	    }
#endif				/* __EXTERN_STACKS__ */

#if __INTERFACE_TEST__ == __HAIRY_INTERFACE__
	    fini_attributes(&attributes);

	} while (0);
#endif				/* __INTERFACE_TEST__ == __HAIRY_INTERFACE__ */
    }
    if (status) {
	if (status == X1f4_E4_ALLOC_ERROR) {
	    perror(self);
	} else {
	    unsigned length;

	    length = strlen(expression);
	    if (express_data->delimiter) {
		length--;
	    }
	    fprintf(stderr, "%s: cannot parse `%.*s'\n", self, (int) length,
		    expression);
	}
    } else {
	if (express_data->do_optimize) {
	    x1f4_line_expression(x1f4_expression);
	}

#if defined HAVE_LIBx1f4l0
	if (valist) {
	    if (express_data->do_storage) {
		unsigned size;

		x1f4_size_xalist(valist, &size);
		fprintf(stderr, " %7u\n", size);
	    }
	}
#endif				/* HAVE_LIBx1f4l0 */

	if (express_data->do_print) {
#if __DUMP_TEST__ == __CAST_DUMP__
	    x1f4_vprint_expression
		(stdout, test_cast, x1f4_expression,
		 express_data->print_flags);
#else
	    x1f4_print_expression
		(stdout, x1f4_expression, express_data->print_flags);
#endif				/* __DUMP_TEST__ == __CAST_DUMP__ */
	} else {
	    struct lead_type output;

	    status = x1f4_link_expression(x1f4_expression, &output);
	    if (status) {
		fprintf(stderr, "%s: cannot evaluate `%s'\n", self,
			expression);
	    } else {
		libx1f4i0_line_data
		    (x1f4_type_expression(x1f4_expression),
		     express_data->precision, &output);
	    }
	}

#if __EXTERN_STACKS__
	stack_free(stack_text, t2nd);
	stack_free(stack_text, t1st);
#endif				/* __EXTERN_STACKS__ */

	x1f4_fini_expression(&x1f4_expression);
    }

#if defined HAVE_LIBx1f4l0
    if (valist) {
	libx1f4i0_fini_valist
	    (self, valist, express_data->do_memory, express_data->do_storage,
	     &status);
    }
#endif				/* HAVE_LIBx1f4l0 */

    return status;
}


static int
miss_line(const char *self, const char *expression,
	  struct express_type *express_data)
{
    char delimiter;
    int status;

    delimiter = express_data->delimiter;
    if (!delimiter) {
	status = miss_fine(self, &expression, express_data);
	if (!status) {
	    putchar('\n');
	}
    } else {
	int once = 1;

	do {
	    status = miss_fine(self, &expression, express_data);
	    if (status) {
		if (!once) {
		    putchar('\n');
		}

		break;
	    } else {
		char c;

		c = *expression;
		while (c == delimiter) {
		    expression++;
		    c = *expression;
		}
		if (c) {
		    once = 0;
		    putchar(delimiter);
		    if (0) {
			putchar(' ');
		    }
		} else {
		    putchar('\n');
		    break;
		}
	    }
	} while (1);
    }

    return status;
}


static int
miss_text(const char *self, int fd, struct express_type *express_data)
{
    char buffer[BLOCK_SIZE], *fifth = NULL, *minor = NULL, *slide = NULL,
	*trans = NULL;
    int shift = 1, status = 0;
    unsigned count = 0;

    while (1) {
	char c;

	if (shift) {
	    shift = 0;
	    slide = minor;
	}

	if (slide == fifth) {
	    char *third;
	    unsigned scale;

	    scale = fifth - minor;
	    if (!(third = (char *) realloc(minor, scale + 02001))) {
		status = -1;
		break;
	    }
	    minor = third;
	    slide = minor + scale;
	    fifth = slide + 02000;
	}

	if (!count) {
	    count = read(fd, buffer, BLOCK_SIZE);
	    if ((int) count == -1) {
		status = -2;
		break;
	    }
	    if (!count) {
		if (slide == minor) {
		} else {
		    *slide = express_data->delimiter;
		    slide[1] = 0;
		    status = miss_line(self, minor, express_data);
		    if (status) {
			break;
		    }
		}
		break;
	    }
	    trans = buffer;
	}

	c = *trans++;
	count--;
	if (c == '\n') {
	    if (1) {
		if (2) {
		    *slide = express_data->delimiter;
		    slide[1] = 0;
		    status = miss_line(self, minor, express_data);
		    if (status) {
			break;
		    }
		}
	    }
	    shift = 1;
	} else {
	    *slide++ = c;
	}
    }

    if (minor) {
	free(minor);
    }

    return status;
}


static int
select_function(const char *f, unsigned length, const void *context,
		const struct x1f4_function_type **function)
{
#if __INTERFACE_TEST__ == __HAIRY_INTERFACE__
    int status = X1f4_E4_PARSE_ERROR;
    const void *key;

    if (!x1f4_look_vtkey_l(f, length, context, &key)) {
	status = 0;
	*function = key;
    }

    return status;
#else
    int status = X1f4_E4_PARSE_ERROR;
    const struct x1f4_function_type *function_data;

    function_data = context;
    if (function_data) {
	while (function_data->name) {
	    if (length == function_data->length
		&& !memcmp((void *) f, function_data->name, length)) {
		break;
	    }
	    function_data++;
	}
	if (function_data->name) {
	    status = 0;
	    *function = function_data;
	}
    }

    return status;
#endif				/* __INTERFACE_TEST__ == __HAIRY_INTERFACE__ */
}


static int
select_variable(const char *f, unsigned length, const void *context,
		const struct x1f4_variable_type **variable, void **state)
{
#if __INTERFACE_TEST__ == __HAIRY_INTERFACE__
    int status = X1f4_E4_PARSE_ERROR;
    const struct context_type *context_data;

    context_data = context;
    if (context_data->variable_tree) {
	const void *key;

	if (!x1f4_look_vtkey_l(f, length, context_data->variable_tree, &key)) {
	    status = 0;
	    *state = context_data->state
		[(struct x1f4_variable_type *) key
		 - context_data->variable_data];
	    *variable = key;
	}
    }

    return status;
#else
    int status = X1f4_E4_PARSE_ERROR;
    const struct context_type *context_data;
    const struct x1f4_variable_type *variable_data;

    context_data = context;
    variable_data = context_data->variable_data;
    if (variable_data) {
	while (variable_data->name) {
	    if (length == variable_data->length
		&& !memcmp((void *) f, variable_data->name, length)) {
		break;
	    }
	    variable_data++;
	}
	if (variable_data->name) {
	    status = 0;
	    *state = context_data->state
		[variable_data - context_data->variable_data];
	    *variable = variable_data;
	}
    }

    return status;
#endif				/* __INTERFACE_TEST__ == __HAIRY_INTERFACE__ */
}


#if __DUMP_TEST__ == __CAST_DUMP__
static int
test_cast(void *cast, const char *name, unsigned size)
{
    return 1 ^ fwrite(name, size, 1, cast);
}
#endif				/* __DUMP_TEST__ == __CAST_DUMP__ */


#if __INTERFACE_TEST__ == __HAIRY_INTERFACE__
static void
fini_attributes(struct x1f4_attributes_type *attributes_data)
{
    fini_variables(attributes_data);
    fini_functions(attributes_data);
}
#endif				/* __INTERFACE_TEST__ == __HAIRY_INTERFACE__ */


#if __INTERFACE_TEST__ == __HAIRY_INTERFACE__
static void
fini_functions(struct x1f4_attributes_type *attributes_data)
{
    x1f4_free_vtkeydata((void *) attributes_data->function_set.context);
}
#endif				/* __INTERFACE_TEST__ == __HAIRY_INTERFACE__ */


#if __INTERFACE_TEST__ == __HAIRY_INTERFACE__
static void
fini_variables(struct x1f4_attributes_type *attributes_data)
{
    struct x1f4_vkeytree_type *variable_tree;

    variable_tree = ((struct context_type *)
	attributes_data->variable_set.context)->variable_tree;
    if (variable_tree) {
	x1f4_free_vtkeydata(variable_tree);
    }
}
#endif				/* __INTERFACE_TEST__ == __HAIRY_INTERFACE__ */


static void
usage(void)
{
    puts("Usage: express33 [OPTIONS] [TYPE NAME VALUE]\n\
Evaluate expressions read on standard input.\n\
\n\
Options:\n\
  -M, --stat-storage		stat expression memory storage requirements\n\
  -P, --print			print expression\n\
  -d, --delimiter DELIMITER	set line expression delimiter to DELIMITER\n\
  -m, --stat-memory		stat memory operations\n\
  -o, --optimize		enable optimizations\n\
  -p, --precision DIGITS	set precision for printing reals to DIGITS\n\
      --detail-constants	detail constants when printing expression\n\
      --detail-operators	detail operators when printing expression\n\
      --list			list available functions and exit\n\
      --list-1			list available unary operators and exit\n\
      --list-2			list available binary operators and exit\n\
      --help			display this help and exit\n\
      --version			output version information and exit");
}


int
main(int argc, char **argv)
{
    int list_functions = 0, status = 0;
    struct express_type express;

    express.delimiter = 0;
    express.do_memory = 0;
    express.do_optimize = 0;
    express.do_print = 0;
    express.do_storage = 0;
    express.precision = 3;
    express.print_flags = 0;

    unsetenv("POSIXLY_CORRECT");

    {
	char *delimiter_specs = NULL, *precision_specs = NULL;

	while (1) {
	    char c;
	    static struct option long_options[] = {
/* *INDENT-OFF* */
		{   "delimiter",   0x01,   NULL,   0x64    },
		{   "detail-constants",
				    0x00,   NULL,   0x12    },
		{   "detail-operators",
				    0x00,   NULL,   0x11    },
		{   "help",	    0x00,   NULL,   0x68    },
		{   "list",	    0x00,   NULL,   0x01    },
		{   "list-1",	    0x00,   NULL,   0x02    },
		{   "list-2",	    0x00,   NULL,   0x03    },
		{   "optimize",	    0x00,   NULL,   0x6f    },
		{   "stat-memory",  0x00,   NULL,   0x6d    },
		{   "stat-storage", 0x00,   NULL,   0x4d    },
		{   "version",	    0x00,   NULL,   0x76    },
		{   "precision",    0x01,   NULL,   0x70    },
		{   "print",	    0x00,   NULL,   0x50    },
		{   NULL,	    0x00,   NULL,   0x00    }
/* *INDENT-ON* */
	    };

	    c = getopt_long(argc, argv, "MPd:mop:", long_options, NULL);

	    if (!~c) {
		break;
	    }

	    switch (c) {
	    case 001:
		list_functions = 1;
		break;
	    case 002:
		list_functions = 2;
		break;
	    case 003:
		list_functions = 3;
		break;
	    case 021:
		express.print_flags |= X1f4_E4_DETAIL_OPERATORS;
		break;
	    case 022:
		express.print_flags |= X1f4_E4_DETAIL_CONSTANTS;
		break;
	    case 'M':
		express.do_storage = 1;
		break;
	    case 'P':
		express.do_print = 1;
		break;
	    case 'd':
		delimiter_specs = optarg;
		break;
	    case 'm':
		express.do_memory = 1;
		break;
	    case 'o':
		express.do_optimize = 1;
		break;
	    case 'p':
		precision_specs = optarg;
		break;
	    case 'h':
		usage();

		return 0;
	    case 'v':
		printf("%s (%s) %s\n", argv[0], PACKAGE, VERSION);

		return 0;
	    case '?':
		return 1;
	    }
	}

	if (list_functions) {
	    precision_specs = NULL;
	}

	if (express.do_print) {
	    precision_specs = NULL;
	}

	if (delimiter_specs) {
	    char delimiter;

	    delimiter = *delimiter_specs;
	    if (!delimiter) {
		status = 1;
	    } else if (delimiter_specs[1]) {
		status = 1;
	    } else {
	    }
	    if (status) {
		fprintf(stderr, "%s: delimiter should be an one character st"
			"ring: `%s' is not\n", argv[0], delimiter_specs);

		return 1;
	    } else {
		express.delimiter = delimiter;
	    }
	}

	if (precision_specs) {
	    if (x1f4_parse_wxcardinal
		(&express.precision, precision_specs, NULL, 0)) {
		fprintf(stderr, "%s: cannot parse precision specification: `"
			"%s'\n", argv[0], precision_specs);

		return 1;
	    }
	}
    }

    if (list_functions == 1) {
	x1f4_print_functions(stdout, x1f4_e4_defaults);
    } else if (list_functions == 2) {
	const struct x1f4_operator_type *const *operator1s;

	x1f4_llink_operator1s(&operator1s);
	x1f4_print_operators(stdout, operator1s, 1);
    } else if (list_functions == 3) {
	const struct x1f4_operator_type *const *operator2s;

	x1f4_llink_operator2s(&operator2s);
	x1f4_print_operators(stdout, operator2s, 2);
    } else if ((argc - optind) % 3) {
	status = 1;
	fprintf(stderr, "%s: wrong number of arguments\nType `%s --help' for"
		" more information.\n", argv[0], argv[0]);
    } else {
	do {
#if __EXTERN_STACKS__
	    int (*stack_free) (void *, void *) = NULL;
#endif				/* __EXTERN_STACKS__ */
	    struct x1f4_variable_type *variable_data;
	    void **state = NULL;
#if __EXTERN_STACKS__
	    void *stack_text = NULL, *t1st = NULL, *t2nd = NULL;
#endif				/* __EXTERN_STACKS__ */

	    status = 1;

	    status = libx1f4i0_init_list
		(argv[0], argc, argv, optind, (void *) &variable_data, &state);
	    if (status) {
		break;
	    }

	    express.variable_data = variable_data;
	    express.state = state;

	    miss_text(argv[0], STDIN_FILENO, &express);

	    libx1f4i0_fini_list(argv[0], (void *) &variable_data);
	} while (0);
    }

    return status;
}
