/*
 * express17.c
 * Copyright (C) 2006-2013, Ciprian Niculescu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <config.h>

#if defined HAVE_FEATURES_H
# include <features.h>
#endif				/* HAVE_FEATURES_H */

#include <getopt.h>
#if defined HAVE_LIBx1f4i0
# include <libx1f4i0.h>
#endif				/* HAVE_LIBx1f4i0 */
#if defined HAVE_LIBx1f4l0
# include <libx1f4l0.h>
#endif				/* HAVE_LIBx1f4l0 */
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <aime.h>
#if !defined HAVE_LIBx1f4i0
# include <cardinal-wx.h>
# include <float1.h>
#endif				/* !HAVE_LIBx1f4i0 */
#include <inter.h>
#if !defined HAVE_LIBx1f4i0
# include <lcardinal.f-fb.h>
# include <lintegral.f-fb.h>
#endif				/* !HAVE_LIBx1f4i0 */
#include <types.h>
#if !defined HAVE_LIBx1f4i0
# include <vkeylook-l.h>
# include <vtkey.h>
#endif				/* !HAVE_LIBx1f4i0 */

#define __EXTERN_STACKS__		0

#define __HAIRY_INTERFACE__		0
#define __STRAIGHT_INTERFACE__		1

#define __CAST_DUMP__			0
#define __FILE_DUMP__			1

#define __COPY_DATA__			0

#define __DUMP_TEST__			__CAST_DUMP__

#define __INTERFACE_TEST__		__HAIRY_INTERFACE__

#define I_BILL(i)			(*((X1f4_E4_C_BILL *) (i)))
#define I_MODE(i)			(*((X1f4_E4_C_MODE *) (i)))
#define I_REAL(r)			(*((X1f4_E4_C_REAL *) (r)))
#define I_TEXT(t)			(*((X1f4_E4_C_TEXT *) (t)))

typedef struct context_type {
    struct x1f4_variable_type *variable_data;
#if __INTERFACE_TEST__ == __HAIRY_INTERFACE__
    struct x1f4_vkeytree_type *variable_tree;
#endif				/* __INTERFACE_TEST__ == __HAIRY_INTERFACE__ */
    void **state;
} context_type;

#if __INTERFACE_TEST__ == __HAIRY_INTERFACE__
static int init_attributes(struct x1f4_attributes_type *);
static int init_functions(struct x1f4_attributes_type *);
static int init_variables(struct x1f4_attributes_type *);
#endif				/* __INTERFACE_TEST__ == __HAIRY_INTERFACE__ */
static int select_function(const char *, unsigned, const void *,
			   const struct x1f4_function_type **);
static int select_variable(const char *, unsigned, const void *,
			   const struct x1f4_variable_type **, void **);
static int test_cast(void *, const char *, unsigned);

#if __INTERFACE_TEST__ == __HAIRY_INTERFACE__
static void fini_attributes(struct x1f4_attributes_type *);
static void fini_functions(struct x1f4_attributes_type *);
static void fini_variables(struct x1f4_attributes_type *);
#endif				/* __INTERFACE_TEST__ == __HAIRY_INTERFACE__ */
static void usage(void);

#if __INTERFACE_TEST__ == __HAIRY_INTERFACE__
static int
init_attributes(struct x1f4_attributes_type *attributes_data)
{
    int status = 1;

    do {
	if (!init_functions(attributes_data)) {
	    if (!init_variables(attributes_data)) {
		status = 0;
		break;
	    }

	    fini_functions(attributes_data);
	}
    } while (0);

    return status;
}
#endif				/* __INTERFACE_TEST__ == __HAIRY_INTERFACE__ */


#if __INTERFACE_TEST__ == __HAIRY_INTERFACE__
static int
init_functions(struct x1f4_attributes_type *attributes_data)
{
    int status;
    const struct x1f4_function_type *function_data;
    struct x1f4_vkeytree_type *root;

    function_data = x1f4_e4_defaults;
    status = x1f4_make_vtkeyroot(&root, function_data->name, function_data);
    if (status) {
	status = X1f4_E4_ALLOC_ERROR;
    } else {
	function_data++;
	while (function_data->name) {
	    status = x1f4_make_vtkeynode
		(&root, function_data->name, function_data);
	    if (status) {
		status = X1f4_E4_ALLOC_ERROR;
		break;
	    } else {
		function_data++;
	    }
	}

	if (status) {
	    x1f4_free_vtkeydata(root);
	}
    }

    if (!status) {
	attributes_data->function_set.context = root;
    }

    return status;
}
#endif				/* __INTERFACE_TEST__ == __HAIRY_INTERFACE__ */


#if __INTERFACE_TEST__ == __HAIRY_INTERFACE__
static int
init_variables(struct x1f4_attributes_type *attributes_data)
{
    int status;
    struct context_type *context_data;
    const struct x1f4_variable_type *variable_data;

    context_data = (void *) attributes_data->variable_set.context;
    variable_data = context_data->variable_data;
    if (!context_data->variable_data) {
	context_data->variable_tree = NULL;
	status = 0;
    } else {
	struct x1f4_vkeytree_type *root;

	status = x1f4_make_vtkeyroot
	    (&root, variable_data->name, variable_data);
	if (status) {
	    status = X1f4_E4_ALLOC_ERROR;
	} else {
	    variable_data++;
	    while (variable_data->name) {
		status = x1f4_make_vtkeynode
		    (&root, variable_data->name, variable_data);
		if (status) {
		    status = X1f4_E4_ALLOC_ERROR;
		    break;
		} else {
		    variable_data++;
		}
	    }

	    if (status) {
		x1f4_free_vtkeydata(root);
	    }
	}

	if (!status) {
	    ((struct context_type *)
	     attributes_data->variable_set.context)->variable_tree = root;
	}
    }

    return status;
}
#endif				/* __INTERFACE_TEST__ == __HAIRY_INTERFACE__ */


static int
select_function(const char *f, unsigned length, const void *context,
		const struct x1f4_function_type **function)
{
#if __INTERFACE_TEST__ == __HAIRY_INTERFACE__
    int status = X1f4_E4_PARSE_ERROR;
    const void *key;

    if (!x1f4_look_vtkey_l(f, length, context, &key)) {
	status = 0;
	*function = key;
    }

    return status;
#else
    int status = X1f4_E4_PARSE_ERROR;
    const struct x1f4_function_type *function_data;

    function_data = context;
    if (function_data) {
	while (function_data->name) {
	    if (length == function_data->length
		&& !memcmp((void *) f, function_data->name, length)) {
		break;
	    }
	    function_data++;
	}
	if (function_data->name) {
	    status = 0;
	    *function = function_data;
	}
    }

    return status;
#endif				/* __INTERFACE_TEST__ == __HAIRY_INTERFACE__ */
}


static int
select_variable(const char *f, unsigned length, const void *context,
		const struct x1f4_variable_type **variable, void **state)
{
#if __INTERFACE_TEST__ == __HAIRY_INTERFACE__
    int status = X1f4_E4_PARSE_ERROR;
    const struct context_type *context_data;

    context_data = context;
    if (context_data->variable_tree) {
	const void *key;

	if (!x1f4_look_vtkey_l(f, length, context_data->variable_tree, &key)) {
	    status = 0;
	    *state = context_data->state
		[(struct x1f4_variable_type *) key
		 - context_data->variable_data];
	    *variable = key;
	}
    }

    return status;
#else
    int status = X1f4_E4_PARSE_ERROR;
    const struct context_type *context_data;
    const struct x1f4_variable_type *variable_data;

    context_data = context;
    variable_data = context_data->variable_data;
    if (variable_data) {
	while (variable_data->name) {
	    if (length == variable_data->length
		&& !memcmp((void *) f, variable_data->name, length)) {
		break;
	    }
	    variable_data++;
	}
	if (variable_data->name) {
	    status = 0;
	    *state = context_data->state
		[variable_data - context_data->variable_data];
	    *variable = variable_data;
	}
    }

    return status;
#endif				/* __INTERFACE_TEST__ == __HAIRY_INTERFACE__ */
}


static int
test_cast(void *cast, const char *name, unsigned size)
{
    return 1 ^ fwrite(name, size, 1, cast);
}


#if __INTERFACE_TEST__ == __HAIRY_INTERFACE__
static void
fini_attributes(struct x1f4_attributes_type *attributes_data)
{
    fini_variables(attributes_data);
    fini_functions(attributes_data);
}
#endif				/* __INTERFACE_TEST__ == __HAIRY_INTERFACE__ */


#if __INTERFACE_TEST__ == __HAIRY_INTERFACE__
static void
fini_functions(struct x1f4_attributes_type *attributes_data)
{
    x1f4_free_vtkeydata((void *) attributes_data->function_set.context);
}
#endif				/* __INTERFACE_TEST__ == __HAIRY_INTERFACE__ */


#if __INTERFACE_TEST__ == __HAIRY_INTERFACE__
static void
fini_variables(struct x1f4_attributes_type *attributes_data)
{
    struct x1f4_vkeytree_type *variable_tree;

    variable_tree = ((struct context_type *)
	attributes_data->variable_set.context)->variable_tree;
    if (variable_tree) {
	x1f4_free_vtkeydata(variable_tree);
    }
}
#endif				/* __INTERFACE_TEST__ == __HAIRY_INTERFACE__ */


static void
usage(void)
{
    puts("Usage: express17 [OPTIONS] EXPRESSION [TYPE NAME VALUE]\n\
Evaluate EXPRESSION.\n\
\n\
Options:\n\
  -B, --block-size SIZE		set integral results comma separated blocks\n\
				size to SIZE\n\
  -M, --stat-storage		stat expression memory storage requirements\n\
  -P, --print			print expression\n\
  -a, --output-base BASE	set output base for integral results to BASE\n\
  -d, --digits COUNT		set integral results minimum number of digits\
\n\
				to COUNT\n\
  -m, --stat-memory		stat memory operations\n\
  -o, --optimize		enable optimizations\n\
  -p, --precision DIGITS	set precision for printing reals to DIGITS\n\
  -t, --type			print expression type\n\
      --detail-constants	detail constants when printing expression\n\
      --detail-operators	detail operators when printing expression\n\
      --list			list available functions and exit\n\
      --list-1			list available unary operators and exit\n\
      --list-2, --list-2h	list available binary operators and exit\n\
      --help			display this help and exit\n\
      --version			output version information and exit");
}


int
main(int argc, char **argv)
{
    int do_memory = 0, do_optimize = 0, do_print = 0, do_storage = 0,
	list_functions = 0, status = 0;
    unsigned block_size = 0, digits = 0, output_base = 10, precision = 3,
	print_flags = 0;

    unsetenv("POSIXLY_CORRECT");

    {
	char *block_size_specs = NULL, *digits_specs = NULL,
	    *output_base_specs = NULL, *precision_specs = NULL;

	while (1) {
	    char c;
	    static struct option long_options[] = {
/* *INDENT-OFF* */
		{   "block-size",   0x01,   NULL,   0x42    },
		{   "detail-constants",
				    0x00,   NULL,   0x12    },
		{   "detail-operators",
				    0x00,   NULL,   0x11    },
		{   "digits",	    0x01,   NULL,   0x64    },
		{   "help",	    0x00,   NULL,   0x68    },
		{   "list",	    0x00,   NULL,   0x01    },
		{   "list-1",	    0x00,   NULL,   0x02    },
		{   "list-2",	    0x00,   NULL,   0x03    },
		{   "list-2h",	    0x00,   NULL,   0x04    },
		{   "optimize",	    0x00,   NULL,   0x6f    },
		{   "output-base",  0x01,   NULL,   0x61    },
		{   "precision",    0x01,   NULL,   0x70    },
		{   "print",	    0x00,   NULL,   0x50    },
		{   "stat-memory",  0x00,   NULL,   0x6d    },
		{   "stat-storage", 0x00,   NULL,   0x4d    },
		{   "version",	    0x00,   NULL,   0x76    },
		{   NULL,	    0x00,   NULL,   0x00    }
/* *INDENT-ON* */
	    };

	    c = getopt_long(argc, argv, "B:MPa:d:mop:t", long_options, NULL);

	    if (!~c) {
		break;
	    }

	    switch (c) {
	    case 001:
		list_functions = 1;
		break;
	    case 002:
		list_functions = 2;
		break;
	    case 003:
		list_functions = 3;
		break;
	    case 004:
		list_functions = 4;
		break;
	    case 021:
		print_flags |= X1f4_E4_DETAIL_OPERATORS;
		break;
	    case 022:
		print_flags |= X1f4_E4_DETAIL_CONSTANTS;
		break;
	    case 'B':
		block_size_specs = optarg;
		break;
	    case 'M':
		do_storage = 1;
		break;
	    case 'P':
		do_print = 1;
		break;
	    case 'a':
		output_base_specs = optarg;
		break;
	    case 'd':
		digits_specs = optarg;
		break;
	    case 'm':
		do_memory = 1;
		break;
	    case 'o':
		do_optimize = 1;
		break;
	    case 'p':
		precision_specs = optarg;
		break;
	    case 't':
		do_print = 2;
		break;
	    case 'h':
		usage();

		return 0;
	    case 'v':
		printf("%s (%s) %s\n", argv[0], PACKAGE, VERSION);

		return 0;
	    case '?':
		return 1;
	    }
	}

	if (list_functions | do_print) {
	    block_size_specs = NULL;
	    digits_specs = NULL;
	    output_base_specs = NULL;
	    precision_specs = NULL;
	}

	if (do_print) {
	    precision_specs = NULL;
	}

	if (output_base_specs) {
	    if (x1f4_parse_wxcardinal
		(&output_base, output_base_specs, NULL, 0)) {
		fprintf(stderr, "%s: cannot parse output base specification:"
			" `%s'\n", argv[0], output_base_specs);

		return 1;
	    } else {
		if (output_base < 2 || 36 < output_base) {
		    fprintf(stderr, "%s: output base (%u) is out of range (0"
			    " .. 36)\n", argv[0], output_base);

		    return 1;
		}
	    }
	}

	if (block_size_specs) {
	    if (x1f4_parse_wxcardinal
		(&block_size, block_size_specs, NULL, 0)) {
		fprintf(stderr, "%s: cannot parse block size specification: "
			"`%s'\n", argv[0], block_size_specs);

		return 1;
	    }
	}

	if (digits_specs) {
	    if (x1f4_parse_wxcardinal(&digits, digits_specs, NULL, 0)) {
		fprintf(stderr, "%s: cannot parse minimum digits count speci"
			"fication: `%s'\n", argv[0], digits_specs);

		return 1;
	    }
	}

	if (precision_specs) {
	    if (x1f4_parse_wxcardinal(&precision, precision_specs, NULL, 0)) {
		fprintf(stderr, "%s: cannot parse precision specification: `"
			"%s'\n", argv[0], precision_specs);

		return 1;
	    }
	}
    }

    if (list_functions == 1) {
	x1f4_print_functions(stdout, x1f4_e4_defaults);
    } else if (list_functions == 2) {
	const struct x1f4_operator_type *const *operator1s;

	x1f4_llink_operator1s(&operator1s);
	x1f4_print_operators(stdout, operator1s, 1);
    } else if (list_functions == 3 || list_functions == 4) {
	const struct x1f4_operator_type *const *operator2s;

	x1f4_llink_operator2s(&operator2s);
	x1f4_print_boperators(stdout, operator2s, list_functions - 3);
    } else if ((argc - optind) % 3 != 1) {
	status = 1;
	fprintf(stderr, "%s: wrong number of arguments\nType `%s --help' for"
		" more information.\n", argv[0], argv[0]);
    } else {
	do {
	    const char *expression;
#if __EXTERN_STACKS__
	    int (*stack_free) (void *, void *) = NULL;
#endif				/* __EXTERN_STACKS__ */
#if defined HAVE_LIBx1f4l0
	    struct list_type dlist;
#endif				/* HAVE_LIBx1f4l0 */
	    struct x1f4_eerecord_type eerecord;
	    struct x1f4_variable_type *variable_data;
	    unsigned flags = X1f4_E4_DDACCESS;
	    void **state = NULL, *x1f4_expression;
#if defined HAVE_LIBx1f4l0
	    void *valist = NULL;
#endif				/* HAVE_LIBx1f4l0 */
#if __EXTERN_STACKS__
	    void *stack_text = NULL, *t1st = NULL, *t2nd = NULL;
#endif				/* __EXTERN_STACKS__ */

	    status = 1;

	    expression = argv[optind];
	    optind++;

	    status = libx1f4i0_init_list
		(argv[0], argc, argv, optind, (void *) &variable_data, &state);
	    if (status) {
		break;
	    }

#if defined HAVE_LIBx1f4l0
	    libx1f4i0_init_valist(&valist, do_memory, do_storage);
#endif				/* HAVE_LIBx1f4l0 */

	    {
		struct context_type context;
		struct x1f4_attributes_type attributes;
#if __EXTERN_STACKS__
		unsigned s1st = 0, s2nd = 0;
#endif				/* __EXTERN_STACKS__ */

		context.state = state;
		context.variable_data = variable_data;

		attributes.bcollect_set.eerecord_data = &eerecord;
		attributes.function_set.get = select_function;
#if __INTERFACE_TEST__ == __STRAIGHT_INTERFACE__
		attributes.function_set.context = x1f4_e4_defaults;
#endif				/* __INTERFACE_TEST__ == __STRAIGHT_INTER... */
		x1f4_llink_operator1s(&attributes.operator1s);
		x1f4_llink_operator2s(&attributes.operator2s);
		attributes.terminator = 0;
		attributes.variable_set.get = select_variable;
		attributes.variable_set.context = &context;

		flags |= X1f4_E4_BCOLLECT | X1f4_E4_ESTORAGE;

#if defined HAVE_LIBx1f4l0
		if (valist) {
		    flags |= X1f4_E4_RESOURCE;
		    libx1f4i0_line_valist
			(valist, do_memory, do_storage,
			 &attributes.resource_set.free,
			 &attributes.resource_set.link,
			 &attributes.resource_set.mode,
			 &attributes.resource_set.context, &dlist);
		}
#endif				/* HAVE_LIBx1f4l0 */

#if __INTERFACE_TEST__ == __HAIRY_INTERFACE__
		do {
		    status = init_attributes(&attributes);
		    if (status) {
			break;
		    }
#endif				/* __INTERFACE_TEST__ == __HAIRY_INTERFACE__ */

#if __EXTERN_STACKS__
		    flags |= X1f4_E4_MAXSTACK;
#endif				/* __EXTERN_STACKS__ */

#if __EXTERN_STACKS__
		    attributes.internal_set.s1st = &s1st;
		    attributes.internal_set.s2nd = &s2nd;
#endif				/* __EXTERN_STACKS__ */

		    status = x1f4_init_expression
			(&x1f4_expression, expression, flags, &attributes);

#if __COPY_DATA__
		    if (status) {
		    } else {
			void *expression;

			status = x1f4_copy_expression
			    (&expression, x1f4_expression);
			if (status) {
			} else {
			    x1f4_fini_expression(&x1f4_expression);
			    x1f4_expression = expression;
			}
		    }
#endif				/* __COPY_DATA__ */

#if __EXTERN_STACKS__
		    if (!status) {
			stack_free = attributes.resource_set.free;
			stack_text = attributes.resource_set.context;

			status = attributes.resource_set.link
			    (stack_text, &t1st, s1st);
			if (status) {
			} else {
			    status = attributes.resource_set.link
				(stack_text, &t2nd, s2nd);
			    if (status) {
				stack_free(stack_text, t1st);
			    }
			}

			if (status) {
			    x1f4_fini_expression(&x1f4_expression);
			} else {
			    x1f4_miss_expression(x1f4_expression, t1st, t2nd);
			}
		    }
#endif				/* __EXTERN_STACKS__ */

#if __INTERFACE_TEST__ == __HAIRY_INTERFACE__
		    fini_attributes(&attributes);

		} while (0);
#endif				/* __INTERFACE_TEST__ == __HAIRY_INTERFACE__ */
	    }
	    if (status) {
		if (status != X1f4_E4_PARSE_ERROR) {
		    perror(argv[0]);
		} else {
		    fprintf(stderr, "%s: cannot parse `%s'\n", argv[0],
			    expression);
		    fprintf(stderr, "%s: ", argv[0]);
		    x1f4_stat_expression(stderr, test_cast, &eerecord, NULL);
		    fprintf(stderr, "\n");
		}
	    } else {
		if (do_optimize) {
		    x1f4_line_expression(x1f4_expression);
		}

#if defined HAVE_LIBx1f4l0
		if (valist) {
		    if (do_storage) {
			unsigned size;

			x1f4_size_xalist(valist, &size);
			fprintf(stderr, " %7u\n", size);
		    }
		}
#endif				/* HAVE_LIBx1f4l0 */

		if (do_print) {
		    if (do_print == 2) {
			unsigned type;

			type = x1f4_type_expression(x1f4_expression);
			if (type == X1f4_E4_MODE) {
			    printf("mode\n");
			} else if (type == X1f4_E4_REAL) {
			    printf("real\n");
			} else if (type == X1f4_E4_TEXT) {
			    printf("text\n");
			} else {
			    printf("void\n");
			}
		    } else {
#if __DUMP_TEST__ == __CAST_DUMP__
			x1f4_vprint_expression
			    (stdout, test_cast, x1f4_expression, print_flags);
#else
			x1f4_print_expression
			    (stdout, x1f4_expression, print_flags);
#endif				/* __DUMP_TEST__ == __CAST_DUMP__ */
			putchar('\n');
		    }
		} else {
		    struct lead_type output;

		    status = x1f4_link_expression(x1f4_expression, &output);
		    if (status) {
			fprintf(stderr, "%s: cannot evaluate `%s'\n", argv[0],
				expression);
		    } else {
			unsigned type;

			type = x1f4_type_expression(x1f4_expression);
			if (type == X1f4_E4_BILL) {
			    x1f4_fprint_fblcardinal
				(stdout, 0, digits, block_size, output_base,
				 I_BILL(&output));
			    putchar('\n');
			} else if (type == X1f4_E4_MODE) {
			    x1f4_fprint_fblintegral
				(stdout, 0, digits, block_size, output_base,
				 I_MODE(&output));
			    putchar('\n');
			} else if (type == X1f4_E4_TEXT) {
			    printf("%s\n", I_TEXT(&output));
			} else if (type == X1f4_E4_REAL) {
			    x1f4_fprint_wfloat
				(stdout, 0, 0, precision, I_REAL(&output));
			    putchar('\n');
			}
		    }
		}

#if __EXTERN_STACKS__
		stack_free(stack_text, t2nd);
		stack_free(stack_text, t1st);
#endif				/* __EXTERN_STACKS__ */

		x1f4_fini_expression(&x1f4_expression);
	    }

#if defined HAVE_LIBx1f4l0
	    if (valist) {
		libx1f4i0_fini_valist
		    (argv[0], valist, do_memory, do_storage, &status);
	    }
#endif				/* HAVE_LIBx1f4l0 */

	    libx1f4i0_fini_list(argv[0], (void *) &variable_data);
	} while (0);
    }

    return status;
}
